/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.Query;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.ArrayList;
import java.util.Enumeration;

/**
 * 検索結果の DBTableModel を使用せず、直接エントリーデータを登録するタグです。
 *
 * このオブジェクトに、 dbid を与えることにより、dbid に対応した
 * Queryオブジェクト(のサブクラスのオブジェクト)が作成されます。
 *
 * ※ このタグは、Transaction タグの対象です。
 *
 * @og.formSample
 * ●形式：
 *    &lt;og:entryQuery
 *        queryType  = &quot;JDBCKeyEntry&quot;
 *        command    = &quot;{&#064;command}&quot; &gt;
 *            { call DYNAMIC_PKG.DYNAMIC_TEST( ?,?,?,? ) }
 *    &lt;/og:query&gt;
 *
 *        queryType    : Queryオブジェクトを作成する時のキー(オプション)
 *                       QueryFactory参照
 *        command      : 処理コマンド( NEWのみ )(オプション)
 *        skipRowCount : データの読み始めの初期値を指定(オプション)
 *        maxRowCount  : データの最大読み込み件数を指定(オプション)
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:entryQuery
 *       queryType        ○【TAG】Query を発行する為のクラスIDを指定します({@og.doc03Link queryType 初期値:JDBCKeyEntry})(必須)。
 *       command            【TAG】コマンド(NEW,RENEW)をセットします(PlsqlUpdateTag,UpdateTag の場合は、ENTRY)
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       maxRowCount        【TAG】(通常は使いません)データの最大読み込み件数を指定します (初期値:DB_MAX_ROW_COUNT[=1000])
 *       skipRowCount       【TAG】(通常は使いません)データの読み始めの初期値を指定します
 *       displayMsg         【TAG】検索結果を画面上に表示するメッセージリソースIDを指定します (初期値:VIEW_DISPLAY_MSG[=])
 *       notfoundMsg        【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])
 *       names              【TAG】PL/SQLを利用する場合の引数にセットすべき データの名称をCSV形式で複数指定します
 *       tableId            【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
 *       dbid               【TAG】(通常は使いません)Queryオブジェクトを作成する時のDB接続IDを指定します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:entryQuery&gt;
 *
 * ●使用例
 *    例１)
 *        &lt;table&gt;&lt;tr&gt;&lt;og:input name=&quot;PN&quot; value=&quot;{&#064;PN}&quot; /&gt;&lt;/tr&gt;
 *            &lt;tr&gt;&lt;og:input name=&quot;CD&quot; value=&quot;{&#064;CD}&quot; /&gt;&lt;/tr&gt;
 *        &lt;/table&gt;
 *
 *    例２)
 *        &lt;table&gt;&lt;tr&gt;&lt;og:input name=&quot;PN&quot; value=&quot;{&#064;PN}&quot; &gt;部品入力フィールド&lt;/og:input&gt;&lt;/tr&gt;
 *            &lt;tr&gt;&lt;og:input name=&quot;CD&quot; value=&quot;{&#064;CD}&quot; &gt;コードフィールド&lt;/og:input&gt;&lt;/tr&gt;
 *        &lt;/table&gt;
 *
 *        BODY 部分は、無視されます。コメント等に使用できます。
 *        HTMLファイルには、コメント部分は出力されません。
 *
 * @og.group ＤＢ登録
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class EntryQueryTag extends QueryTag {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0.0 (2005/01/31)" ;

	private static final long serialVersionUID = 400020050131L ;

	/**
	 * Query を実行します。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) JDBCKeyEntryQuery クラス名変更(⇒ Query_JDBCKeyEntry)
	 * @og.rev 4.0.0.0 (2005/01/31) execute( String[] ,String[] )を標準メソッドに格上げします。
	 *
	 * @param   query オブジェクト
	 */
	@Override
	protected void execute( final Query query ) {
		final String[] nameArray = getRequestKeys();
		final String[] values = getRequest( nameArray );
		try {
			query.execute( nameArray,values );		// 4.0.0 (2005/01/31)
			errCode = query.getErrorCode();
			errMessage = query.getErrorMessage();
			query.commit();
		}
		catch( HybsSystemException ex ) {
			query.rollback();
			query.realClose();
			throw ex;
		}
		finally {
			if( query != null ) { query.close(); }
		}
	}

	/**
	 * リクエスト情報のキー配列データを取得します。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) Vector を使用している箇所で、非同期でも構わない箇所を、ArrayList に置換え。
	 *
	 * @return	そのリクエストのキー配列情報
	 */
	protected String[] getRequestKeys() {
		final ArrayList<String> list = new ArrayList<String>();

		final Enumeration<?> enume = getParameterNames();		// 4.3.3.6 (2008/11/15) Generics警告対応

		while( enume.hasMoreElements() ) {
			final String key  = (String)(enume.nextElement());
			list.add( key );
		}

		final String[] rtn = list.toArray( new String[list.size()] );

		return rtn;
	}

	/**
	 * 【TAG】Query を発行する為のクラスIDを指定します({@og.doc03Link queryType 初期値:JDBCKeyEntry})。
	 *
	 * @og.tag
	 * 引数指定のINSERT/UPDATE文を実行する場合の、queryType 属性を使用します。
	 * このタグでは、execute( String[] , String[] )を実行します。
	 * 代表的なクラスとして、"JDBCKeyEntry" が標準で用意されています。
	 *
	 * タグにより使用できる／出来ないがありますが、これは、org.opengion.hayabusa.db
	 * 以下の Query_**** クラスの **** を与えます。
	 * これらは、Query インターフェースを継承したサブクラスです。
	 * {@og.doc03Link queryType Query_**** クラス}
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) JavaDocコメント用にメソッド追加。
	 *
	 * @param	id Query を発行する為の実クラス ID
	 * @see		org.opengion.hayabusa.db.Query  Queryのサブクラス
	 * @see		org.opengion.hayabusa.db.Query#execute( String[] , String[] )
	 */
	@Override
	public void setQueryType( final String id ) {
		super.setQueryType( nval( id,"JDBCKeyEntry" ) );
	}
}
