/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.sql.ResultSet;
// import java.sql.ResultSetMetaData;			// 6.0.4.0 (2014/11/28)
import java.sql.SQLException;
// import java.sql.Types;						// 6.0.4.0 (2014/11/28)
// import java.util.Locale;

// import org.opengion.fukurou.db.DBUtil;
import static org.opengion.fukurou.util.HybsConst.CR ;		// 6.1.0.0 (2014/12/26)
import org.opengion.fukurou.db.Transaction;
import org.opengion.fukurou.db.TransactionReal;
import org.opengion.fukurou.db.ResultSetValue;				// 6.0.4.0 (2014/11/28)
import org.opengion.fukurou.util.ApplicationInfo;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
// import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.LabelData;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * データベース関連の便利なメソッドを集めた簡易ユーティリティークラスです。
 * 全てのメソッドは、static メソッドになっています。
 *
 * @og.rev 2.1.1.1 (2002/11/15) Serializable インターフェースを削除する。
 * @og.rev 4.0.0.0 (2007/10/16) 名称変更(DBUtil ⇒ DBTableModelUtil) DBアクセス関係のメソッドはfukurou/db/DBUtilに移動
 * @og.group ＤＢ/Shell制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class DBTableModelUtil {
	/**
	 * インスタンスを作らないので、コンストラクタは、private に設定します。
	 */
	private DBTableModelUtil() {}

	/**
	 * 初期データベースに接続して、Queryを実行します。
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,DBTableModel として返されます。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) 新規追加
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 * @og.rev 4.0.0.0 (2007/10/10) dbid の初期値を、"DEFAULT" から null に変更
	 *
	 * @param   stmt ステートメント文字列
	 * @param   args オブジェクトの引数配列
	 * @param   resource リソースマネージャー
	 * @param   appInfo アプリ情報オブジェクト
	 *
	 * @return  検索結果の配列
	 */
	public static DBTableModel makeDBTable( final String stmt ,final String[] args, final ResourceManager resource, final ApplicationInfo appInfo ) {
		return makeDBTable( stmt ,args,resource,appInfo,null );
	}

	/**
	 * 検索するデータベースを指定して、Queryを実行します。
	 * ステートメントと引数により、Prepared クエリーの検索のみ実行します。
	 * 結果は,DBTableModel として返されます。
	 * 検索以外のSQLも実行できます。結果は、null を返します。
	 *
	 * @og.rev 3.0.0.0 (2002/12/25) 新規追加
	 * @og.rev 3.0.0.1 (2003/02/14) ヘッダー、フッター情報が null のときの処理追加。
	 * @og.rev 3.5.6.0 (2004/06/18) nullに対する無駄な比較を削除します。
	 * @og.rev 3.8.7.0 (2006/12/15) アクセスログ取得の為,ApplicationInfoオブジェクトを設定
	 * @og.rev 4.0.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 * @og.rev 5.1.9.0 (2010/08/01) Transaction 対応
	 * @og.rev 5.3.7.0 (2011/07/01) TransactionReal の引数変更
	 * @og.rev 5.3.8.0 (2011/08/01) Transaction発生箇所でclose()
	 *
	 * @param   stmt	ステートメント文字列
	 * @param   args	オブジェクトの引数配列
	 * @param   resource リソースマネージャー
	 * @param   appInfo アプリ情報オブジェクト
	 * @param   dbid	接続先ID
	 *
	 * @return  検索結果の配列
	 */
	public static DBTableModel makeDBTable( final String stmt ,final String[] args ,
											final ResourceManager resource, final ApplicationInfo appInfo, final String dbid ) {
		if( stmt == null || stmt.isEmpty() ) { return null; }

		DBTableModel table = null;

		final Query query = QueryFactory.newInstance( "JDBCPrepared" );

		ErrorMessage errMessage = null;
		Transaction tran = null;
		try {
			tran = new TransactionReal( appInfo );						// 5.3.8.0 (2011/08/01) Transaction発生箇所でclose()
			query.setTransaction( dbid,tran );							// 5.1.9.0 (2010/08/01) Transaction 対応
			query.setResourceManager( resource );	// 4.0.0 (2005/01/31)
			query.setStatement( stmt );
			query.execute( args );
			final int errCode = query.getErrorCode();
			final int executeCount = query.getExecuteCount();
			if( errCode < ErrorMessage.NG && executeCount >= 0 ) {		// 異常以外の場合
				table = query.getDBTableModel();
				if( query.isUpdate() ) { query.commit(); }
			}
			else {
				errMessage = query.getErrorMessage();
			}
		}
		catch( HybsSystemException ex ) {
			if( query != null ) { query.rollback(); }
			throw ex;
		}
		finally {
			QueryFactory.close( query );
			if( tran != null ) { tran.close(); }		// 5.3.8.0 (2011/08/01) Transaction発生箇所でclose()
		}

		if( errMessage != null ) {
			throw new HybsSystemException( errMessage.toString() );
		}

		return table;
	}

	/**
	 * 空の DBTableModelオブジェクトを作成します。
	 * これは、本来、ファクトリクラスで作成すべきですが、簡易作成メソッドとして
	 * DBUtil の static メソッドとして実装します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @return  DBTableModelオブジェクト
	 */
	public static DBTableModel newDBTable() {
		return new DBTableModelImpl();
	}

	/**
	 * カラム名の配列及びデータの2次元配列からDBテーブルモデルを作成します。
	 * カラム名がセットされていない若しくはデータがセットされていない場合は、nullを返します。
	 *
	 * @og.rev 4.2.1.0 (2008/04/26) 新規追加
	 *
	 * @param   clms カラム名の配列
	 * @param   vals 値の配列
	 * @param   resource リソースマネージャー
	 *
	 * @return  DBテーブルモデル
	 */
	public static DBTableModel makeDBTable( final String[] clms, final String[][] vals, final ResourceManager resource ) {
		if( clms == null || clms.length == 0
				|| vals == null || vals.length == 0 || vals[0] == null || vals[0].length == 0 ) {
			return null;
		}

		if( clms.length != vals[0].length ) {
			final String errMsg = "キーのカラム数とデータのカラム数が一致していません。"
						+ CR
						+ " clms.length=[" + clms.length + "]  vals.length=[" + vals[0].length + "]"
						+ " clms=" + StringUtil.array2csv( clms ) + CR
						+ " vals=" + StringUtil.array2csv( vals[0] )  ;	// 5.1.8.0 (2010/07/01) errMsg 修正
			throw new HybsSystemException( errMsg );
		}

		final int numberOfColumns = clms.length;
		final DBTableModel table = newDBTable() ;
		table.init( numberOfColumns );

		DBColumn[] dbColumn = new DBColumn[numberOfColumns];
		for( int column=0; column<numberOfColumns; column++ ) {
			dbColumn[column] = resource.makeDBColumn( clms[column] );
			table.setDBColumn( column,dbColumn[column] );
		}

		final int numberOfRows = vals.length;
		for( int row=0; row<numberOfRows; row++ ) {
			table.addColumnValues( vals[row] );
		}

		return table;
	}

	/**
	 * 検索結果オブジェクトからDBテーブルモデルを作成します。
	 * 検索結果オブジェクトまたはリソースオブジェクトがセットされていない場合は、nullを返します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 * @og.rev 5.5.5.4 (2012/08/18) TIMESTAMP 型もCLOBと同様に処理を分ける。
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
	 *
	 * @param   result 検索結果オブジェクト
	 * @param   skipRowCount 読み飛ばし件数
	 * @param	maxRowCount 最大検索件数
	 * @param   resource リソースマネージャー
	 *
	 * @return  DBテーブルモデル
	 * @throws	SQLException データベースアクセスエラー
	 */
	public static DBTableModel makeDBTable( final ResultSet result, final int skipRowCount, final int maxRowCount, final ResourceManager resource ) throws SQLException {
		if( result == null || resource == null ) { return null; }

		final ResultSetValue rsv = new ResultSetValue( result );

		final int clmSize =  rsv.getColumnCount();

		final DBTableModel table = DBTableModelUtil.newDBTable() ;
		table.init( clmSize );

		for( int clmNo=0; clmNo<clmSize; clmNo++ ) {
			final String	name	= rsv.getColumnName(clmNo) ;
			DBColumn dbColumn = resource.getDBColumn( name );
			if( dbColumn == null ) {
				dbColumn = makeDBColumn( name,clmNo,rsv,resource );
			}
			table.setDBColumn( clmNo,dbColumn );
		}

		// データ部の設定
		int numberOfRows = 0;
		while( numberOfRows < skipRowCount && rsv.next() ) {
			// 注意 rsv.next() を先に判定すると必ず１件読み飛ばしてしまう。
			numberOfRows ++ ;
		}
		numberOfRows = 0;

		while( numberOfRows < maxRowCount && rsv.next() ) {
			numberOfRows ++ ;
			table.addColumnValues( rsv.getValues() );
		}

		// 最大件数が、超えた場合でかつ次のデータがある場合は、オーバーフロー
		if( numberOfRows >= maxRowCount && rsv.next() ) {
			table.setOverflow( true );
		}

		return table;
	}
//	public static DBTableModel makeDBTable( final ResultSet result, final int skipRowCount, final int maxRowCount, final ResourceManager resource ) throws SQLException {
//		if( result == null || resource == null ) { return null; }
//
//		ResultSetMetaData metaData	= result.getMetaData();
//
//		int numberOfColumns =  metaData.getColumnCount();
//
//		DBTableModel table = DBTableModelUtil.newDBTable() ;
//		table.init( numberOfColumns );
//
//		// 項目名，項目タイプ，項目サイズ，書込みフラグを設定する。
//		DBColumn[] dbColumn = new DBColumn[numberOfColumns];
//
//		// 3.8.5.0 (2006/03/02) CLOB/ROWID などのカラムかどうかを判定します。
//		boolean   isOther = false;
//		int[] types  = new int[numberOfColumns];
//
//		for( int column=0; column<numberOfColumns; column++ ) {
//			String	name	 = metaData.getColumnLabel(column+1).toUpperCase(Locale.JAPAN) ;
//			dbColumn[column] = resource.getDBColumn( name );
//			if( dbColumn[column] == null ) {
//				LabelData labelData  = resource.getLabelData( name );
//				dbColumn[column] = makeDBColumn( name,labelData,metaData,column,resource.getLang() );
//			}
//			table.setDBColumn( column,dbColumn[column] );
//
//			// 3.8.5.0 (2006/03/02) CLOB カラムかどうかを判定します。
//	 		// 5.5.5.4 (2012/08/18) TIMESTAMP 型もCLOBと同様に処理を分ける。
//			int clmType = metaData.getColumnType(column+1);
//			types[column] = clmType;
//			if( clmType == Types.CLOB || clmType == Types.ROWID || clmType == Types.TIMESTAMP ) {	// JDK 6.0以降 ROWID
//				isOther = true;
//			}
//		}
//
//		// データ部の設定
//		int numberOfRows = 0;
//		while( numberOfRows < skipRowCount && result.next() ) {
//			// 注意 resultSet.next() を先に判定すると必ず１件読み飛ばしてしまう。
//			numberOfRows ++ ;
//		}
//		numberOfRows = 0;
//
//		// 3.8.5.1 (2006/05/08) 行列のループなので、 CLOB 使用可否でループを分ける。
//		if( isOther ) {
//			while( numberOfRows < maxRowCount && result.next() ) {
//				numberOfRows ++ ;
//				String[] columnValues = new String[numberOfColumns];
//				for( int column=0; column<numberOfColumns; column++ ) {
//					// 5.3.6.0 (2011/06/01) メソッド化
//					columnValues[column] = DBUtil.getValue( result, column, types[column] );
//				}
//				table.addColumnValues( columnValues );
//			}
//		}
//		else {
//			while( numberOfRows < maxRowCount && result.next() ) {
//				numberOfRows ++ ;
//				String[] columnValues = new String[numberOfColumns];
//				for( int column=0; column<numberOfColumns; column++ ) {
//					Object obj = result.getObject(column+1);
//					columnValues[column] = ( obj == null ? "" : String.valueOf( obj ) );
//				}
//				table.addColumnValues( columnValues );
//			}
//		}
//
//		// 最大件数が、超えた場合でかつ次のデータがある場合は、オーバーフロー
//		if( numberOfRows >= maxRowCount && result.next() ) {
//			table.setOverflow( true );
//		}
//
//		return table;
//	}

	/**
	 * 検索結果オブジェクトからエディット設定に基づいて変換されたDBテーブルモデルを作成します。
	 * 検索結果オブジェクトまたはリソースオブジェクトまたはエディット設定オブジェクトがセットされていない場合は、nullを返します。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規追加
	 *
	 * @param   result 検索結果オブジェクト
	 * @param   skipRowCount 読み飛ばし件数
	 * @param	maxRowCount 最大検索件数
	 * @param   resource リソースマネージャー
	 * @param	config エディット設定オブジェクト
	 *
	 * @return  DBテーブルモデル
	 * @throws	SQLException データベースアクセスエラー
	 */
	public static DBTableModel makeEditDBTable( final ResultSet result, final int skipRowCount, final int maxRowCount, final ResourceManager resource, final DBEditConfig config ) throws SQLException {
		if( result == null || resource == null ) { return null; }
		final DBTableModel table = new DBTableModelEditor();
		((DBTableModelEditor)table).create( result, skipRowCount, maxRowCount, resource, config );
		return table;
	}

	/**
	 * ResultSetValue から、DBColumn オブジェクトを作成します。
	 *
	 * DBColumn オブジェクト がリソースファイルに定義されていない場合に、
	 * データベースの検索結果のメタデータを利用して、DBColumn オブジェクトを
	 * 作成します。
	 *
	 * @og.rev 3.4.0.0 (2003/09/01) 表示パラメータ、編集パラメータ、文字パラメータの追加。
	 * @og.rev 3.4.0.2 (2003/09/05) DBType のデフォルト値を、'X' から 'XK' に変更します。
	 * @og.rev 3.6.0.7 (2004/11/06) DBColumn の official属性追加
	 * @og.rev 4.0.0.0 (2005/01/31) lang 変数を取得
	 * @og.rev 5.3.6.0 (2011/06/01) AbstractQueryから移動
	 * @og.rev 6.0.2.1 (2014/09/26) org.opengion.fukurou.db.DBUtil#type2ClassName(int) に移動
	 * @og.rev 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
	 *
	 * @param	name		カラム名
	 * @param	column		カラム番号
	 * @param   rsv			ResultSetValueオブジェクト
	 * @param   resource	リソースマネージャー
	 * @return	DBColumnオブジェクト
	 */
//	public static DBColumn makeDBColumn( final String name,final LabelData labelData,
//							final ResultSetMetaData metaData,final int column,final String lang ) {
	public static DBColumn makeDBColumn( final String name,final int column,
							final ResultSetValue rsv, final ResourceManager resource ) {
		final DBColumn dbColumn ;

		final LabelData labelData = resource.getLabelData( name );
		final String    lang      = resource.getLang();

		try {
//			String	clsName  = type2ClassName( metaData.getColumnType(column+1) );
//			String	clsName  = DBUtil.type2ClassName( metaData.getColumnType(column+1) );	// 6.0.2.1 (2014/09/26)
			final String	clsName  = rsv.getClassName(column);									// 6.0.4.0 (2014/11/28)
//			int 	size	 = metaData.getColumnDisplaySize(column+1);
			int 	size	 = rsv.getColumnDisplaySize(column);							// 6.0.4.0 (2014/11/28)
			if( size == 0 ) { size = 60; }
//			boolean writable = metaData.isWritable(column+1);
			final boolean writable = rsv.isWritable(column);										// 6.0.4.0 (2014/11/28)
			final String	dbType	 = "NUMBER".equals( clsName ) ? "S9" : "XK" ;
			final String	defValue = "NUMBER".equals( clsName ) ? "0"  : ""  ;
			final DBColumnConfig config = new DBColumnConfig(
				lang,							// 言語
				name,							// カラム名
				labelData,						// カラムのラベルデータオブジェクト
				clsName ,						// カラムのクラスを文字列にした名称
				String.valueOf( size ) ,		// カラムの文字桁数
				String.valueOf( writable ) ,	// カラムが書き込み可能かどうか
				null ,							// データの表示用レンデラー
				null ,							// データの編集用エディター
				null ,							// メニューの項目コードデータオブジェクト
				dbType ,						// データのタイプ
				defValue,						// データのデフォルト値
				null ,							// 表示用レンデラーのパラメータ
				null ,							// 編集用エディターのパラメータ
				null ,							// データのタイプのパラメータ
				null ,							// カラムロール
				false,							// 正式なカラムオブジェクトかどうか
				null							// データベース接続先ID
			);

			dbColumn = new DBColumn( config );		// 4.0.0 (2005/01/31)

		}
		// 6.0.4.0 (2014/11/28) ResultSetValue を使用するように変更。
//		catch( SQLException ex ) {
//		final String errMsg = "DBColumn を作成できませんでした。name=[" + name + " , label=[" + labelData + "]";
//			throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
//		}
		catch( RuntimeException ex2 ) {
			final String errMsg = "予期せぬエラーが発生しました。name=[" + name + " , label=[" + labelData + "]";
			throw new HybsSystemException( errMsg,ex2 );		// 3.6.0.0 (2004/09/17)
		}

		return dbColumn;
	}

//	/**
//	 * カラムのタイプを表現する文字列値を返します。
//	 *
//	 * この文字列を用いて、CCSファイルでタイプごとの表示方法を
//	 * 指定することができます。
//	 *
//	 * @og.rev 2.1.1.1 (2002/11/15) その他のケースを、VARCHAR2 を返すように修正。
//	 * @og.rev 4.0.0.0 (2006/01/31) CLOB を追加
//	 * @og.rev 5.3.6.0 (2011/06/01) AbstractQueryから移動
//	 * @og.rev 5.5.5.4 (2012/08/18) DECIMAL,TIMESTAMP を追加
//	 * @og.rev 5.5.5.4 (2012/08/18) DECIMAL,TIMESTAMP を追加
//	 * @og.rev 6.0.2.1 (2014/09/26) org.opengion.fukurou.db.DBUtil#type2ClassName(int) に移動
//	 *
//	 * @param	type タイプ番号
//	 *
//	 * @return	カラムのタイプを表現する文字列値
//	 * @see		java.sql.Types
//	 */
//	private static String type2ClassName( final int type ) {
//		final String rtn ;
//
//		switch( type ) {
//			case Types.CHAR:
//			case Types.VARCHAR:
//			case Types.BIT:
//				rtn = "VARCHAR2"; break;
//			case Types.LONGVARCHAR:			// 4.0.0 (2006/01/31)
//				rtn = "LONG"; break;
//			case Types.TINYINT:
//			case Types.SMALLINT:
//			case Types.INTEGER:
//			case Types.NUMERIC:
//			case Types.BIGINT:
//			case Types.FLOAT:
//			case Types.DOUBLE:
//			case Types.REAL:
//			case Types.DECIMAL:				// 5.5.5.4 (2012/08/18)
//				rtn = "NUMBER"; break;
//			case Types.DATE:
//			case Types.TIMESTAMP:			// 5.5.5.4 (2012/08/18)
//				rtn = "DATE"; break;
//			case Types.CLOB:				// 4.0.0 (2006/01/31)
//				rtn = "CLOB"; break;
//			default:
//				rtn = "NONE"; break;
//		}
//
//		return rtn;
//	}
}
