/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.HybsConst ;		// 6.1.0.0 (2014/12/26)
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;

/**
 * CellEditor の具象クラスで、カラムのデータを編集する場合に使用するクラスです。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public abstract class AbstractEditor implements CellEditor {
	private final int COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_COLUMNS_MAXSIZE" ) ;	// 表示フィールドの大きさ
	// 3.5.4.7 (2004/02/06) viewタグで表示する場合のカラムの大きさ
	private final int VIEW_COLUMNS_MAXSIZE = HybsSystem.sysInt( "HTML_VIEW_COLUMNS_MAXSIZE" ) ;

	/** システムの改行コードを設定します。*/
	protected static final String CR		 = HybsConst.CR;			// 6.1.0.0 (2014/12/26) refactoring
	/** StringBilderなどの初期値を設定します。	{@value} */
	protected static final int BUFFER_MIDDLE = HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

	// 3.3.1.1 (2003/07/03) name , attributes 属性を final にする。
	protected  Attributes	  attributes ;		// SubClass で誤って new することを防止
	protected  String		  name		 ;
	protected  String		  size1		 ;
	protected  String		  size2		 ;
//	protected  String		  optAttr	 ;		// 6.0.4.0 (2014/11/28) 
	protected  TagBuffer	  tagBuffer = new TagBuffer() ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 *
	 */
	public AbstractEditor() {
		// ここでは処理を行いません。
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.5.4.2 (2003/12/15) 漢字入力(IMEモード)をONにするのを、"K" のみとします。
	 * @og.rev 3.5.4.2 (2003/12/15) size が 1 の場合、CSSファイルでサイズ指定を行うクラスを出力します。
	 * @og.rev 3.5.4.6 (2004/01/30) 漢字入力(IMEモード)をONにするのを、"K" と、"KX" のみとします。
	 * @og.rev 3.5.5.5 (2004/04/23) maxlength の導入
	 * @og.rev 4.0.0.0 (2005/01/31) getFieldSize メソッドを XHTMLTag から DBColumn へ移動
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 * @og.rev 6.0.4.0 (2014/11/28) optionAttributes は、コンストラクタで設定します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	protected AbstractEditor( final DBColumn clm ) {
		name			 = clm.getName();
		final String	disabled = clm.isWritable() ? null : "disabled" ;

		final int maxlength = clm.getTotalSize();		// 4.0.0 (2005/01/31) メソッド名変更

		attributes = new Attributes();
		attributes.set( "type"		  ,"text" );
		attributes.set( "maxlength"   ,String.valueOf( maxlength ) );
		attributes.set( "disabled"	  ,disabled );

		attributes.addAttributes( clm.getEditorAttributes() );

		final String clazz = clm.getDbType();
		attributes.add( "class" ,clazz );
		if( maxlength <= 5 ) {
			attributes.add( "class" ,"S0" + maxlength );
		}

		final String optAttr = attributes.get( "optionAttributes" );
		tagBuffer.add( optAttr );								// 6.0.4.0 (2014/11/28)

		// 3.5.5.5 (2004/04/23) size の導入、初期値は、一覧表のサイズにします。
		String size = clm.getViewLength();		// 4.0.0 (2005/01/31)
		if( size != null ) {
			if( size.indexOf(',') >= 0 ) {
				size = size.substring( 0,size.indexOf(',') ) ;
			}
			size1 = size ;
			size2 = size ;
		}
		else {
			// 4.0.0 (2005/01/31) getFieldSize メソッドを XHTMLTag から DBColumn へ移動
			size1 = String.valueOf(clm.getFieldSize( maxlength,COLUMNS_MAXSIZE )) ;
			size2 = String.valueOf(clm.getFieldSize( maxlength,VIEW_COLUMNS_MAXSIZE )) ;
		}
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 3.5.5.5 (2004/04/23) viewSize の導入
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param	value 値
	 *
	 * @return	データの編集用文字列
	 */
	public String getValue( final String value ) {

		final TagBuffer tag = new TagBuffer( "input" );
		tag.add( "name"    , name );
		if( attributes.get( "id" ) == null || attributes.get( "id" ).isEmpty() ) { // 4.3.7.2 (2009/06/15)
			tag.add( "id"      , name ); // 4.3.6.0 (2009/04/01)
		}
		tag.add( "value"   , value );
		tag.add( "size"    , size1 );
		tag.add( tagBuffer.makeTag() );
//		tag.add( optAttr );		// 3.5.5.8 (2004/05/20)

		return tag.makeTag();
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 2.0.0.3 (2002/09/26) optionAttributes 属性に "$i" を使うとその行数に置き換る機能を追加。
	 * @og.rev 3.1.0.0 (2003/03/20) 名前と行番号の区切り記号を "^" から "__" に変更。
	 * @og.rev 3.5.4.2 (2003/12/15) getFieldSize を、XHTMLTag.getFieldSize に変更。
	 * @og.rev 3.5.4.7 (2004/02/06) viewタグで表示する場合のカラムの大きさ VIEW_COLUMNS_MAXSIZE 追加
	 * @og.rev 3.5.5.0 (2004/03/12) 名前と行番号の区切り記号("__")を、HybsSystem.JOINT_STRING  に変更。
	 * @og.rev 3.5.5.5 (2004/04/23) viewSize の導入、初期値は、一覧表にあわせました。
	 * @og.rev 3.5.5.5 (2004/04/23) 新規に Attributes オブジェクトを作成する方式を止めます。
	 * @og.rev 4.3.7.2 (2009/06/15) 属性でidが出力される場合は、idを出力しない
	 * @og.rev 5.1.7.0 (2010/06/01) 動的プルダウン実装見直し
	 *
	 * @param	row   行番号
	 * @param	value 値
	 *
	 * @return	データ表示/編集用の文字列
	 */
	public String getValue( final int row,final String value ) {
		final TagBuffer tag = new TagBuffer( "input" );
		final String newName = name + HybsSystem.JOINT_STRING + row;
		// tag.add( "name"    , name + HybsSystem.JOINT_STRING + row );
		tag.add( "name"  , newName ); // 4.3.6.0 (2009/04/01)
		if( attributes.get( "id" ) == null || attributes.get( "id" ).isEmpty() ) { // 4.3.7.2 (2009/06/15)
			tag.add( "id"    , newName ); // 4.3.6.0 (2009/04/01)
		}
		tag.add( "value"   , value );
		tag.add( "size"    , size2 );
		tag.add( tagBuffer.makeTag() );
//		tag.add( optAttr );		// 3.5.5.8 (2004/05/20)

		return tag.makeTag( row,value );
	}
}
