/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.mail;

import org.opengion.fukurou.util.StringUtil ;
import org.opengion.fukurou.util.HybsEntry ;
import org.opengion.fukurou.util.Argument ;
import org.opengion.fukurou.util.FileString ;

import java.util.Map;
import java.util.LinkedHashMap;

import java.util.logging.Logger;
import java.util.logging.Level;

/**
 * MailTransfer は、ＳＭＴＰプロトコルによるメール送信プログラムです。
 *
 * 引数には、host,from,to,cc,bcc,replyTo,subject,message,charset,file,debug,help が指定できます。
 * 引数のプロパテイのキー部は、大文字・小文字が厳格に適用されますので、正確に記述願います。
 *
 * Usage: java org.opengion.fukurou.fukurou.mail.MailTransfer
 *        -host=メールサーバー(必須)
 *        -from=送信元アドレス(必須)
 *        -to=送信先アドレスをカンマ区切りで並べる(必須)
 *        -port=送信サーバーのポートを指定(初期値:-1)
 *        -auth=認証方式(POP_BEFORE_SMTPのみ対応)
 *        -authPort=認証ポート(初期値:-1)
 *        -user=メールを取得するログインユーザー
 *        -passwd=メールを取得するログインパスワード
 *        -cc=関係者アドレスをカンマ区切りで並べる
 *        -bcc=隠し送信先アドレスをカンマ区切りで並べる
 *        -replyTo=返信先アドレスをカンマ区切りで並べる
 *        -subject=タイトル
 *        -message=本文を格納しているファイルのアドレス
 *        -charset=メール送信時のキャラクタセット [ISO-2022-JP / Windows-31J]
 *        -fileXX=添付ファイルのファイルパス( XX は複数登録時の添え字 )
 *        -debug=メイラーのデバッグモードを指定します。[false / true]
 *        -help=使用方法を出力して、終了します。
 *        -{&#064;XXXX}=YYYY  メッセージ本文の {&#064;XXXX} 文字列を、YYYY 文字列に変換します。
 *            {&#064;ARG.XXX} 予約文字変換 上記引数を割り当てます。
 *            {&#064;DATE.XXX} 予約文字変換 SimpleDateFormat 形式の文字を変換します。(日付、時刻等)
 *            {&#064;ENV.XXX}  予約文字変換 システムプロパティーの文字を変換します。(java -Dkey=value オプション)
 *
 * E-Mail で日本語を送信する場合、ISO-2022-JP(JISコード)化して、7bit で
 * エンコードして送信する必要がありますが、Windows系の特殊文字や、unicodeと
 * 文字のマッピングが異なる文字などが、文字化けします。
 * 対応方法としては、
 * １．Windows-31J + 8bit 送信
 * ２．ISO-2022-JP に独自変換 + 7bit 送信
 * の方法があります。
 * 今回、この２つの方法について、charset で指定できます。
 * 初期値は、ISO-2022-JP です。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class MailTransfer {
	private static Logger logger = Logger.getLogger( "org.opengion.fukurou.fukurou.mail.MailTransfer" );		// 4.3.3.5 (2008/11/08)

	/** メール送信時のデフォルトキャラクタセット {@value}  */
	public static final String CHARSET = "ISO-2022-JP" ;

	/** メイラーの初期デバッグモード {@value}  */
	public static final boolean DEBUG_MODE = false ;

	private static final Map<String,String> mustProparty   ;	// 必須パラメータ
	private static final Map<String,String> usableProparty ;	// 整合性チェックパラメータ

	static {
		mustProparty = new LinkedHashMap<String,String>();
		mustProparty.put( "host",	"メールサーバー(必須)" );
		mustProparty.put( "from",	"送信元アドレス(必須)" );
		mustProparty.put( "to",		"送信先アドレスをカンマ区切りで並べる(必須)" );

		usableProparty = new LinkedHashMap<String,String>();
		usableProparty.put( "port",		"送信サーバーのポート" );					// 6.0.3.0 (2014/11/13) SMTP認証
		usableProparty.put( "auth",		"認証方式(POP_BEFORE_SMTPのみ対応)" );		// 6.0.3.0 (2014/11/13) SMTP認証
		usableProparty.put( "authPort",	"認証ポート(" );							// 6.0.3.0 (2014/11/13) SMTP認証
		usableProparty.put( "user",		"認証ユーザー" );							// 6.0.3.0 (2014/11/13) SMTP認証
		usableProparty.put( "passwd",	"認証パスワード" );							// 6.0.3.0 (2014/11/13) SMTP認証
		usableProparty.put( "cc",		"関係者アドレスをカンマ区切りで並べる" );
		usableProparty.put( "bcc",		"隠し送信先アドレスをカンマ区切りで並べる" );
		usableProparty.put( "replyTo",	"返信先アドレスをカンマ区切りで並べる" );
		usableProparty.put( "subject",	"タイトル" );
		usableProparty.put( "message",	"本文を格納しているファイルのアドレス" );
		usableProparty.put( "charset",	"メール送信時のキャラクタセット [ISO-2022-JP / Windows-31J]" );
		usableProparty.put( "file",		"添付ファイルのファイルパス( XX は複数登録時の添え字 )" );
		usableProparty.put( "{@",		"{@XXXX}=YYYY  メッセージ本文の {@XXXX} 文字列を、YYYY 文字列に変換します。" );
		usableProparty.put( "{@ARG.",	"{@ARG.XXX} 予約文字変換 上記引数を割り当てます。" );
		usableProparty.put( "{@DATE.",	"{@DATE.XXX} 予約文字変換 SimpleDateFormat 形式の文字を変換します。(日付、時刻等)" );
		usableProparty.put( "{@ENV.",	"{@ENV.XXX} 予約文字変換 システムプロパティーの文字を変換します。(java -Dkey=value オプション)" );
		usableProparty.put( "debug",	"メイラーのデバッグモードを指定します。[false / true]" );
		usableProparty.put( "help",		"使用方法を出力して、終了します。" );
	}

	/**
	 * トランスファーを開始します。
	 *
	 * @og.rev 4.3.3.5 (2008/11/08) Argument オブジェクトへの引数を util → mail に訂正します。
	 * @og.rev 6.0.3.0 (2014/11/13) SMTP認証
	 *
	 * @param	args	引数配列(可変長引数)
	 */
//	public void start( final String[] args ) {
	public void start( final String... args ) {

		// パラメータの解析、取得
		logger.fine( "パラメータの解析、取得" );
		final Argument argment = new Argument( "org.opengion.fukurou.mail.MailTransfer" );		// 4.3.3.5 (2008/11/08)
		argment.setMustProparty( mustProparty );
		argment.setUsableProparty( usableProparty );

		argment.setArgument( args );

		// help パラメータが true に指定された場合の処理。
		if( argment.getProparty( "help",false ) ) {
			System.out.println( argment.toString() );
			return;
		}

		final String host		= argment.getProparty( "host" );
		final String chset	= argment.getProparty( "charset", CHARSET );

		final String port		= argment.getProparty( "port" );		// 6.0.3.0 (2014/11/13) SMTP認証
		final String auth		= argment.getProparty( "auth" );		// 6.0.3.0 (2014/11/13) SMTP認証
		final String authPort	= argment.getProparty( "authPort" );	// 6.0.3.0 (2014/11/13) SMTP認証
		final String user		= argment.getProparty( "user" );		// 6.0.3.0 (2014/11/13) SMTP認証
		final String passwd	= argment.getProparty( "passwd" );		// 6.0.3.0 (2014/11/13) SMTP認証

//		MailTX mail = new MailTX( host,chset );
		final MailTX mail = new MailTX( host,chset,port,auth,authPort,user,passwd );		// 6.0.3.0 (2014/11/13) SMTP認証
		logger.fine( "org.opengion.fukurou.mail.MailTX の設定" );

		mail.setFrom( argment.getProparty( "from" ) );
		mail.setTo( StringUtil.csv2Array( argment.getProparty( "to" ) ) );
		mail.setCc( StringUtil.csv2Array( argment.getProparty( "cc" ) ) );
		mail.setBcc( StringUtil.csv2Array( argment.getProparty( "bcc" ) ) );
		mail.setReplyTo( StringUtil.csv2Array( argment.getProparty( "replyTo" ) ) );
		mail.setSubject( argment.getProparty( "subject" ) );
		mail.setDebug( argment.getProparty( "debug", DEBUG_MODE ) );

		// message は、本文を格納しているファイル名が指定されています。
		final String msgFile = argment.getProparty( "message" ) ;
		logger.fine( "本文を格納しているファイルの取得。message=" + msgFile );
		if( msgFile != null ) {
			final FileString fs = new FileString();
			fs.setFilename( msgFile );
			String message = fs.getValue();

			// {@XXX} 変換は、Argument クラスの機能を使う。
			message = argment.changeParam( message );
			mail.setMessage( message );
		}

		// 添付ファイルは、fileXX 形式で、複数指定されています。
		logger.fine( "添付ファイルを取り込みます。" );
		final HybsEntry[] entry = argment.getEntrys( "file" );
		String[] files = new String[entry.length];
		for( int i=0; i<files.length; i++ ) {
			files[i] = entry[i].getValue() ;
		}
		mail.setFilename( files );

		// 送信
		logger.fine( "送信処理を行います。" );
		mail.sendmail();
	}

	/**
	 * main メソッドです。
	 *
	 * @param	args	引数配列
	 */
	public static void main ( final String[] args ) {
		final MailTransfer transfer = new MailTransfer();
		try {
			logger.info( "メール送信処理を開始します  ---------------------------------------------" );
			transfer.start( args );
			logger.info( "正常に終了しました。" );
		}
		catch( Throwable th ) {
			final String errMsg = "メール送信中に例外が発生しました。 " + th.getMessage() ;
			logger.log( Level.SEVERE,errMsg, th );
			throw new RuntimeException( errMsg,th );	// 4.0.0 (2005/01/31)
		}
	}
}
