/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.Cleanable;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
// import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem ;
import org.opengion.hayabusa.common.SystemManager ;
import org.opengion.hayabusa.common.HybsSystemException ;
import org.opengion.hayabusa.common.BuildNumber ;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager ;
import org.opengion.hayabusa.resource.UserInfoFactory ;
import org.opengion.hayabusa.resource.UserInfo ;
import org.opengion.hayabusa.resource.GUIInfo ;

import static org.opengion.fukurou.util.StringUtil.nval ;

import javax.servlet.http.HttpSession ;
import javax.servlet.http.HttpServletRequest ;
import javax.servlet.http.HttpServletResponse ;
import javax.servlet.ServletException;
import javax.servlet.jsp.tagext.Tag ;

import java.io.UnsupportedEncodingException;
import java.io.IOException;
import java.io.File;
import java.util.Map;
import java.util.HashMap;

/**
 * HTMLファイルの head タグに相当する機能を付加します。
 * レイアウトを構築する場合に、HTMLファイルの構造に準拠した形式で記述できるように
 * なります。JavaScript や body タグを 各JSPファイル内で独自に設定できるように
 * なる為、カスタマイズが容易になります。
 *
 * 機能的には、旧バージョンの
 *    common/session_init.jsp
 *    common/HTML.jsp
 *    &lt;og:responseHeader cacheControl = "max-age=0" /&gt;
 * を組み込んだ、common/jsp_init.jsp の代わりに使用できるタグになります。
 *
 * 属性に、下記の値を設定できます。それ以外は、BODY部に直接記述することで追加できます。
 * BODY部、および、属性の引数には、{&#064;XXXX}が使用できます。
 * 例えば、{&#064;SYS.CONTEXT_NAME}を使用すれば、相対アドレスでの指定ではなく、
 * 絶対アドレスで指定可能になるため、ブラウザのキャッシュが利用できます。
 *
 * システム変数による、初期値制御されているのは、下記のフラグです。
 * これらは、headerタグで個別に指定する事はできません。
 *
 *     MAX_INACTIVE_INTERVAL = "1800"
 *     USE_HTML5_HEADER = "(Ver5=false Ver6=true)"
 *     SCROLL_BAR_TYPE  = "common/scrollBar2"
 *     GUI_TOP_TITLE    = "Web アプリケーションシステム" ;
 *     VIEW_USE_BLOCKUI = "true"
 *
 * @og.formSample
 * ●形式：&lt;og:head /&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:head
 *       title              【TAG】HTML の タイトルを設定します(初期値:session に"GUI_TITLE" キーで登録されている値)
 *       headType           【TAG】ヘッダータイプ[xml|init|query|result|menu|simple|handy|frame|popup]を設定します(初期値:null)
 *       useScrollBar       【TAG】ScrollBar.js JavaScript を使用するかどうか[true/false]を指定します(初期値:true)
 *       useGantt           【TAG】option/gantt.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)
 *       useTreeBOM         【TAG】option/treeBOM.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)
 *       useAjaxTreeTable   【TAG】Ajaxを利用したツリー形式のテーブル表示を使用するかを指定します
 *       noTransition       【TAG】画面遷移を行わない形式の登録方法を使用するかを指定します(NO_TRANSITION_MODE=false)
 *       useTabLink         【TAG】og:tabLinkタグで使用される common/tagLink.js をインクルードするかどうか[true/false]を指定します(初期値:false)
 *       useTabstrip        【TAG】tabstrip.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)
 *       useRealTimeCheck   【TAG】カラムチェックをリアルタイムで行うかどうか[true/false]を指定します(初期値:true)
 *       useIE7Header       【TAG】IE8でIE7エミュレートモードで動作させるタグを出力するかどうか[true/false]を指定します(初期値:USE_IE7_HEADER[=true])
 *       useViewPort        【TAG】タブレット等の画面サイズ調整METAタグ(viewport)出力するかどうか[true/false]を指定します(初期値:USE_VIEWPORT_HEADER[=false])
 *       useAjaxSubmit      【TAG】フォームのSubmit処理をJavaScriptを利用して行うか(ポップアップエラー表示)(初期値:USE_AJAX_SUBMIT=false)
 *       useFocusHilight    【TAG】フォーカスが当っている入力ボックスの色を変更するかどうか[true/false]を指定します(初期値:true)
 *       useHrefIcon        【TAG】href(リンク)の拡張子に応じたアイコンを表示するかどうか[true/false]を指定します(初期値:false)
 *       useBlockUI         【TAG】検索中表示を行うかどうかを[true/false]を指定します(初期値:VIEW_USE_BLOCKUI[=true])
 *       addEvent           【TAG】javaScript を初期ロードし、起動するメソッドを指定します(初期値:null)
 *       adjustEvent        【TAG】特定の javaScript を初期ロードし、起動するメソッドを指定します(初期値:null)
 *       useTbodyMovable    【TAG】option/ogTbodyMovable.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:head&gt;
 * ●使用例
 *       &lt;og:head
 *          headType        = "[xml|init|query|result|menu|simple|handy|frame|popup]"
 *          title           = "HEAD タグの TITLE属性"
 *          useScrollBar    = "[true/false]"    scrollBar2.js
 *          useGantt        = "[false/true]"    option/gantt.js
 *          useTreeBOM      = "[false/true]"    option/treeBOM.js
 *          useAjaxTreeTable= "[false/true]"    option/ajaxTreeTable.js
 *          noTransition    = "[false/true]"    option/noTransition.js
 *          useTabLink      = "[false/true]"    option/tabLink.js
 *          useTabstrip     = "[false/true]"    option/tabstrip.js
 *          useRealTimeCheck= "[true/false]"    realTimeCheck.js	5.4.5.0 (2012/03/01) 追加(復活)
 *          useFocusHilight = "[true/false]"    focusHilight.js		5.4.5.0 (2012/03/01) 追加(復活)
 *          useHrefIcon     = "[false/true]"    hrefIcon.css		5.6.5.2 (2013/06/21) 追加
 *          useBlockUI      = "[true/false]"    jquery/blockUI.js	5.7.9.0 (2014/08/08) 追加
 *          adjustEvent     = "[Bar|Matrix|Table|Gantt|Task]"       5.6.3.2 (2013/04/12) 追加
 *          useTbodyMovable = "[false/true]"    option/ogTbodyMovable.js   5.6.3.3 (2013/04/19) 追加
 *       &gt; ... Body ...
 *       &lt;/og:head&gt;
 *
 * ●廃止属性(5.2.1.0 (2010/10/01))
 *      //  useTimeSet      = "[true/false]"    timeSet.js , timeCheck.js
 *      //  useToggleFrame  = "[true/false]"    toggleFrame.js      5.2.1.0 (2010/10/01) 廃止
 *      //  useInputFocus   = "[true/false]"    inputFocus.js       4.3.5.0 (2009/02/01) 廃止
 *      //  useHilightRow   = "[true/false]"    hilightRow.js       5.2.1.0 (2010/10/01) 廃止
 *      //  usePrintLine    = "[true/false]"    printLine.js        5.2.1.0 (2010/10/01) 廃止
 *      //  useAutoConvert  = "[true/false]"    autoConvert.js      5.2.1.0 (2010/10/01) 廃止
 *      //  useFocusFirstElm= "[true/false]"    focusFirstElm.js    5.2.1.0 (2010/10/01) 廃止
 *      //  useAdjustFrameHeight = "[true/false]" adjustFrameHeight 5.2.1.0 (2010/10/01) 廃止
 *
 * @og.rev 4.0.0.0 (2007/04/16) 新規作成
 * @og.rev 5.7.4.3 (2014/03/28) ソースコード大幅見直し
 * @og.group その他
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class HeadTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.6.0 (2015/06/19)" ;

	private static final long serialVersionUID = 626020150619L ;

	private static final String JSP_ADRS = "/" + HybsSystem.getContextName() + "/jsp/" ;
	private static final String JSP_DIR  = HybsSystem.sys( "REAL_PATH" ) + "jsp/" ;			// 5.2.1.0 (2010/10/01)

	/** adjustEvent 引数に渡す事の出来る 文字列リスト  */
	private static final String ADJUST_EVENT_LIST = "|Bar|Matrix|Table|Gantt|Task|" ;		// 5.6.3.2 (2013/04/12)

	// headType が null の場合、使用されるJavaScript,stylesheet (XMLタイプ以外すべて)
	private static final String DEFAULT_CSS		= "common/default.css";
	private static final String CALENDAR_CSS	= "common/calendar.css";
	private static final String CUSTOM_CSS		= "custom/custom.css";

	// init,query,result,menu 両タイプで使用されるJavaScript,stylesheet (null でなく、XML でないすべて)
	private static final String JQUERY_JS		= "common/jquery/jquery.js";
	private static final String JQUERY_OG_JS	= "common/jqog.js";
	private static final String DEFAULT_JS		= "common/default.js";					// 5.2.1.0 (2010/10/01)
	private static final String TABLINK_JS		= "common/option/tabLink.js";			// useTabLink

	// query,result 両タイプで使用されるJavaScript
	private static final String EVENT_SCRIPT_JS		= "common/eventScript.js";			// 5.2.1.0 (2010/10/01)
	private static final String REAL_TIME_CHECK_FLG	= "USE_REAL_TIME_CHECK";			// 5.4.5.0 (2012/03/01)
	private static final String FOCUS_HILIGHT_FLG	= "USE_FOCUS_HILIGHT";				// 5.4.5.0 (2012/03/01)
	private static final String AUTO_MANUAL_JS		= "common/autoManual.js";			// 6.2.5.0 (2015/06/05)
	private static final String JQUERY_UI_JS		= "common/jquery/jquery-ui.js";		// 6.2.6.0 (2015/06/19) helpの移動

	private static final String JQUERY_FORM_JS		= "common/jquery/jquery.form.js";	// noTransition || useAjaxSubmit
	private static final String JQUERY_DRAG_JS		= "common/jquery/jqDnR.js";			// noTransition || useAjaxSubmit
	private static final String JQUERY_MODAL_JS		= "common/option/jqModal.js";		// noTransition || useAjaxSubmit
	private static final String JQUERY_MODAL_CSS	= "common/option/jqModal.css";		// noTransition || useAjaxSubmit

	// query タイプで使用されるJavaScript
	private static final String QUERY_SCRIPT_JS		= "common/queryScript.js";		// 5.2.1.0 (2010/10/01)
	private static final String QUERYCONV_SCRIPT_JS	= "common/queryConvert.js";		// 5.6.2.3 (2013/03/22) QUERYPRE_SCRIPT_JS   → QUERYCONV_SCRIPT_JS に変更
	// 5.7.7.1 (2014/06/13) jquery.blockUI.js による検索中表示
	private static final String JQUERY_BLOCKUI_JS	= "common/jquery/jquery.blockUI.js";	// useBlockUI
	private static final String CUSTOM_BLOCKUI_JS	= "common/jquery/blockUI.js";			// useBlockUI

	// result タイプで使用されるJavaScript,stylesheet
	private static final String SCROLL_BAR_JS		= HybsSystem.sys( "SCROLL_BAR_TYPE" ) + ".js";	// 4.3.5.0 (2009/02/01)
	private static final String SCROLL_BAR_CSS		= HybsSystem.sys( "SCROLL_BAR_TYPE" ) + ".css";	// 4.3.5.0 (2009/02/01)

	private static final String RESULT_SCRIPT_JS	= "common/resultScript.js";			// 5.2.1.0 (2010/10/01)

	private static final String GANTT_JS			= "common/option/gantt.js";				// useGantt
	private static final String TREEBOM_JS			= "common/option/treeBOM.js";			// useTreeBOM
	private static final String AJAX_TREETABLE_JS	= "common/option/ajaxTreeTable.js";		// useAjaxTreeTable
	private static final String AJAX_HYBS_FORM		= "common/option/ajaxHybsForm.js";		// noTransition || useAjaxSubmit
	private static final String NO_TRANSITION_JS	= "common/option/noTransition.js";		// noTransition
	private static final String AJAX_SUBMIT_JS		= "common/option/ajaxSubmit.js";		// useAjaxSubmit

	private static final String TBODY_MOVABLE_JS	= "common/option/ogTbodyMovable.js";	// 5.6.3.3 (2013/04/19) useTbodyMovable 

	// 5.6.3.2 (2013/04/12) adjustEvent の引数に応じて、初期に組み込む script,css 群を定義します。
	private static final String AJST_GANTT_JS		= "common/option/adjustGanttTable.js";			// 5.6.3.2 (2013/04/12) adjustTable,adjustGant,adjustTask
	private static final String AJST_GANTT_CSS		= "common/option/adjustGanttTable.css";			// 5.6.3.2 (2013/04/12) adjustTable,adjustGant,adjustTask
	private static final String AJST_BAR_JS			= "common/option/adjustBar.js";					// 5.6.3.2 (2013/04/12) adjustBar
	private static final String AJST_MATRIX_JS		= "common/option/adjustMatrix.js";				// 5.6.3.2 (2013/04/12) adjustMatrix

	private static final String HREF_ICON_CSS		= "common/hrefIcon.css";						// 5.6.5.2 (2013/06/21) useHrefIcon

	// 6.0.2.5 (2014/10/31) ブラウザ判定して、htmlタグに、class属性を追加します。
	// 現段階では、AJST_GANTT_JS で呼ばれ、AJST_GANTT_CSS で、判定結果を使っているだけです。
	private static final String ADD_BRWS_CLS_JS		= "common/option/addBrowserClass.js";			// 6.0.2.5 (2014/10/31) ブラウザ判定

	// 4.2.1.0 (2008/04/16) htmlタグにクラスを付けるため、HEAD1と2に分離
	// class名は"headtype"+headtype (IDの先頭は数字では駄目なのでheadtypeを付ける)
	private static final String HEAD1 =
		"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Frameset//EN\""	+ CR
		+ "  \"http://www.w3.org/TR/html4/frameset.dtd\">"				+ CR
		+ "<html class=\"headtype-";

	private static final String HEAD2 =
		"\">"				+ CR
		+ "<head>"			+ CR
		+ "  <meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />" + CR
		+ "  <meta http-equiv=\"Content-Style-Type\" content=\"text/css\" />"			+ CR
		+ "  <meta http-equiv=\"Content-Script-Type\" content=\"text/javascript\" />"	+ CR;

	// 5.7.1.0 (2013/12/06) HTML5 用のヘッダー1
	// HTML5では、DOCTYPE宣言がなくなったのですが、無いとブラウザが、互換モードになる為入れておきます。
	private static final String HTML5_HEAD1 =
		"<!DOCTYPE html>"	+ CR
		+ "<html class=\"headtype-";

	// 5.7.1.0 (2013/12/06) HTML5 用のヘッダー2
	// HTML5では、content-style-type と Content-Script-Type も設定不要です。
	private static final String HTML5_HEAD2 =
		"\">"				+ CR
		+ "<head>"			+ CR
		+ "  <meta charset=\"UTF-8\" />" + CR ;

	private static final String XML_HEAD =
		"<?xml version=\"1.0\" encoding=\"UTF-8\" ?>"	+ CR;

	private static final String TAB_HEAD =
		"<?xml version=\"1.0\" encoding=\"UTF-8\" ?>"	+ CR
		+ "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\""	+ CR
		+ "    \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">"	+ CR
		+ "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:mp xmlns:ts >"	+ CR
		+ "<head>"																+ CR
		+ "  <meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />" + CR
		+ "  <meta http-equiv=\"Content-Style-Type\" content=\"text/css\" />"			+ CR
		+ "  <meta http-equiv=\"Content-Script-Type\" content=\"text/javascript\" />"	+ CR
		+ "  <?IMPORT NAMESPACE=\"mp\" IMPLEMENTATION=\""
		+ JSP_ADRS + "common/option/multipage.htc\" />"				+ CR
		+ "  <?IMPORT NAMESPACE=\"ts\" IMPLEMENTATION=\""
		+ JSP_ADRS + "common/option/tabstrip.htc\" />"				+ CR;

	// 4.3.7.0 (2009/06/01) IE8対応。IE7用Metaタグ追加
	private static final String IE7_HEAD =
		"  <meta http-equiv=\"X-UA-Compatible\" content=\"IE=EmulateIE7\" />" + CR;

	// 5.5.8.0 (2012/11/01) タブレット対応
	private static final String MOBILE_HEAD =
		"  <meta content=\"width=device-width, initial-scale=1\" name=\"viewport\"/>" + CR;

	// 5.2.1.0 (2010/10/01) キャッシュは、各ファイルのタイムスタンプベースで行う。
	private static final Map<String,String> srcAdrs = new HashMap<String,String>();

	// 5.7.4.3 (2014/03/28) headType="popup" 時に、BODY にINSERTする文字列。
	// POPUPでは、自分自身にsubmiするため、base タグで target の記述が必要です。
	private static final String POPUP_BODY = "<base target=\"_self\" />" ;

	// 5.7.7.1 (2014/06/13) headType="result" 時に、BODY にINSERTする文字列。
	// 5.7.8.1 (2014/07/18) ３ペイン等で、frame が多段階になっている場合の対応
	// 6.2.4.0 (2015/05/15) fnUnblockUI 未定義エラーに対応
	private static final String UNBLOCKUI_READY
//					=	"	$(function(){" + CR
//					+	"		var qfrm=window.parent.frames['QUERY'];" + CR
//					+	"		if(qfrm){ $(document).ready(function(){ qfrm.$.fnUnblockUI(); }); }" + CR
//					+	"		else {" + CR
//					+	"			qfrm=window.parent.parent.frames['QUERY'];" + CR
//					+	"			if(qfrm){ $(document).ready(function(){ qfrm.$.fnUnblockUI(); }); }" + CR
//					+	"		}" + CR
//					+	"	});" ;
					=	"	$(function(){" + CR
					+	"		var qfrm=window.parent.frames['QUERY'];" + CR
					+	"		if(qfrm == null) {" + CR
					+	"			qfrm=window.parent.parent.frames['QUERY'];" + CR
					+	"		}" + CR
					+	"		if( qfrm && qfrm.$ && qfrm.$.fnUnblockUI ) {" + CR
					+	"			$(document).ready(function(){ qfrm.$.fnUnblockUI(); });" + CR
					+	"		}" + CR
					+	"	});" ;

	//enum化
	// 5.7.4.3 (2014/03/28) frame,popup追加
//	private enum HEAD_TYPE_ENUM { xml , init , query , result , menu , simple , handy };
//	private enum HEAD_TYPE_ENUM { xml , init , query , result , menu , simple , handy , frame , popup };
	// 5.7.8.1 (2014/07/18) 後処理を簡素化する為に、enum関数を作成
	/**
	 * ヘッドタイプの enum 化します。
	 *
	 * @og.rev 5.7.8.1 (2014/07/18) 後処理を簡素化する為に、enum関数を作成
	 */
	private enum HEAD_TYPE_ENUM {
		xml , init , query , result , menu , simple , handy , frame , popup ;

		/**
		 * enum に含まれるかどうか(true:含まれる/false:含まれない)を返します。
		 *
		 * @param	判定する文字列
		 *@return	enum に含まれるかどうか(true:含まれる/false:含まれない)
		 */
		public boolean into( final String str ) {
			return str != null && str.contains( this.name() );
		}
	};

	private String	body		;
	private String	title		;
	private HEAD_TYPE_ENUM headType;

	private boolean useScrollBar	= true;
	private boolean useTabstrip		;
	private boolean useGantt		;
	private boolean useTreeBOM		;
//	private boolean useGuest		= false;										// 6.0.2.5 (2014/10/31) 廃止
	private boolean useAjaxTreeTable;												// 4.3.1.0 (2008/09/02)
	private boolean noTransition	= HybsSystem.sysBool( "NO_TRANSITION_MODE" );	// 4.3.1.0 (2008/09/02)
	private boolean useAjaxSubmit	= HybsSystem.sysBool( "USE_AJAX_SUBMIT" );
	private boolean useRealTimeCheck= true;											// 5.4.5.0 (2012/03/01)
	private boolean useTabLink		;												// 4.3.3.0 (2008/10/01)
	private boolean useFocusHilight	= true;											// 5.4.5.0 (2012/03/01)
	private boolean useHrefIcon		;												// 5.6.5.2 (2013/06/21) 追加
	private boolean useIE7Header	= HybsSystem.sysBool( "USE_IE7_HEADER" );		// 4.3.7.0 (2009/06/01) IE8対応
	private boolean useViewPort		= HybsSystem.sysBool( "USE_VIEWPORT_HEADER" );	// 5.5.8.0 (2012/11/01)
	private boolean useBlockUI		= HybsSystem.sysBool( "VIEW_USE_BLOCKUI" );		// 5.7.9.0 (2014/08/08)
	private String  addEvent		;												// 5.6.3.0 (2013/04/01) addEvent 追加
	private String  adjustEvent		;												// 5.6.3.2 (2013/04/12) adjustEvent 追加
	private boolean useTbodyMovable	;												// 5.6.3.3 (2013/04/19) 新規追加

	private boolean useHTML5		= HybsSystem.sysBool( "USE_HTML5_HEADER" );		// 5.7.1.0 (2013/12/06) HTML5対応

	// 5.2.1.0 (2010/10/01) キャッシュのクリアは、adminメニューの[ﾌﾟｰﾙ削除]で可能
	static {
		final Cleanable clr = new Cleanable() {
			/**
			 * 初期化(クリア)します。
			 * 主に、キャッシュクリアで利用します。
			 */
			public void clear() {
				srcAdrs.clear();
			}
		};
		SystemManager.addCleanable( clr );
	}

	/**
	 * このタグハンドラの親(最も近くを囲んでいるタグハンドラ)を設定する
	 * setParent( Tag ) を オーバーライドします。
	 * この、オーバーライドは、カスタムタグのライフサイクルに基づき、
	 * 各種セッターメソッドの前に呼ばれる メソッドとして、利用しています。
	 *
	 * 目的は、sessionInit() メソッドの実行により、{&#064;SYS.XXXX} 変数を利用可能にすることです。
	 *
	 * @og.rev 4.3.4.1 (2008/12/08) 新規作成
	 *
	 * @param  tag タグハンドラの親
	 */
	@Override
	public void setParent( final Tag tag ) {
		super.setParent( tag );
		sessionInit() ;
	}

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 4.2.1.0 (2008/04/16) headTypeのenum化
	 * @og.rev 4.3.4.1 (2008/12/08) sessionInit() を、setParent( Tag ) メソッドへ移動、title 属性の設定
	 * @og.rev 5.2.3.0 (2010/12/01) 画面アクセス履歴管理対応
	 * @og.rev 6.2.6.0 (2015/06/19) titleが未設定の時は、画面名称(画面ID)とする。
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 4.3.5.7 (2009/03/22) アクセスカウントの不具合対応。sessionInitから移動
		if( headType == HEAD_TYPE_ENUM.result ) {
			final GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
			guiInfo.addAccessCount();
			// 5.2.3.0 (2010/12/01) 画面アクセス履歴管理対応
			final UserInfo userInfo = (UserInfo)getSessionAttribute( HybsSystem.USERINFO_KEY );
			userInfo.setAccessGui( guiInfo );
		}

		final HttpServletResponse response = (HttpServletResponse)pageContext.getResponse();
		if( headType == HEAD_TYPE_ENUM.xml ) {
			response.setHeader( "contentType","text/xml; charset=UTF-8" );
		}
		response.setHeader( "Cache-Control","max-age=0" );

		if( headType == HEAD_TYPE_ENUM.query ){
			removeSessionAttribute( HybsSystem.TBL_MDL_CONKEY );
		}

		// 4.3.4.1 (2008/12/08) title 属性を sessionInit() でなく、doStartTag() メソッドで処理
		// title 属性を最優先で設定する。
		if( title != null ) {
			setSessionAttribute( "GUI_TITLE", title );
		}
		else {
			// 6.2.6.0 (2015/06/19) titleが未設定の時は、画面ID:画面名称とする。
			// 画面の説明が入ると、大きなデータが出力されてしまうので。
//			title = String.valueOf( getSessionAttribute( "GUI_TITLE" ) );
			title = getGUIInfoAttri( "LABEL" ) + '(' + getGUIInfoAttri( "KEY" ) + ')' ;
		}

		return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 5.7.4.3 (2014/03/28) headType="popup" 時は、BODY に、POPUP_BODY を出力します。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		// 5.7.4.3 (2014/03/28) headType="popup" 時は、BODY に、POPUP_BODY を出力します。
		// そのため、すでに、body が書かれている場合の対応が必要です。
//		body = nval( getBodyString(),null );

		final String tmpBody = nval( getBodyString(),null );

		// tmpBody == null なら、body のままなので、処理無し。
		if( tmpBody != null ) {
			body = (body == null) ? tmpBody : body + CR + tmpBody ;
		}

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.2.1.0 (2008/04/16) headTypeのenum化、HTMLタグにclass属性を出力
	 * @og.rev 4.3.1.0 (2008/09/02) AJaxツリーテーブル、画面遷移なしモードの対応
	 * @og.rev 4.3.3.0 (2008/10/01) リアルタイムチェック時にjQueryをインクルード・tabLinkのJS追加
	 * @og.rev 4.3.5.0 (2009/02/01) JavaScript整理に伴う変更
	 * @og.rev 4.3.6.0 (2009/04/01) eventColumn追加
	 * @og.rev 4.3.8.0 (2009/08/01) useAjaxSubmit追加
	 * @og.rev 5.1.3.0 (2010/02/01) CSSは、headTypeがnullでもinclude(forward.jspでエラーが発生した場合への対応)
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition,ajaxSubmitのコントロールは、requestで行う。
	 * @og.rev 5.2.0.0 (2010/09/01) Javaスクリプトで登録したイベントをイベントハンドラにバインドする
	 * @og.rev 5.2.1.0 (2010/10/01) bulkSet.js追加 、JavaScript の統合、廃止
	 * @og.rev 5.3.6.0 (2011/06/01) QUERYのFocusの制御を追加
	 * @og.rev 5.4.4.2 (2012/02/03) type simple追加
	 * @og.rev 5.4.5.0 (2012/03/01) realTimeCheck,focusHilight制御追加
	 * @og.rev 5.5.8.0 (2012/11/01) viewport
	 * @og.rev 5.6.0.3 (2013/01/24) EventScriptの前に呼ぶqueryScriptPre追加
	 * @og.rev 5.6.2.1 (2013/03/08) useGantt2="true" の条件追加
	 * @og.rev 5.6.2.3 (2013/03/22) QUERYPRE_SCRIPT_JS   → QUERYCONV_SCRIPT_JS に変更
	 * @og.rev 5.6.3.0 (2013/04/01) addEvent 追加。useGantt2="true" 時の処理内容と分離
	 * @og.rev 5.6.3.2 (2013/04/12) useGantt2 廃止。adjustEvent の引数で使い分ける。
	 * @og.rev 5.6.3.3 (2013/04/19) useTbodyMovable 追加
	 * @og.rev 5.6.5.2 (2013/06/21) useHrefIcon 追加
	 * @og.rev 5.7.1.0 (2013/12/06) HTML5対応 , handyタイプ追加 , 処理方法を変更。
	 * @og.rev 5.7.4.3 (2014/03/28) ソースコードのメンテナンス
	 * @og.rev 5.7.6.2 (2014/05/16) IEのHTML5機能が有効か無効かの判定値をsessionに設定します。
	 * @og.rev 5.7.7.1 (2014/06/13) blockUI 対応
	 * @og.rev 5.7.8.1 (2014/07/18) headType が、query でも、登録系もあるため、query.jsp のみ ON する。
	 * @og.rev 5.7.9.0 (2014/08/08) headType="query"の時は、RequestURI に含まれる場合のみ ON する。
	 * @og.rev 6.0.2.5 (2014/10/31) ブラウザ判定 ADD_BRWS_CLS_JS を追加。
	 * @og.rev 6.2.5.0 (2015/06/05) 自動ヘルプ(showViewManual)のquery.jsp 対応
	 * @og.rev 6.2.6.0 (2015/06/19) helpの移動をするには、このJavaScriptが必要。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 4.3.3.0 (2008/10/01) デフォルト値として、リクエストパラメーターを使用
		noTransition  = nval( getRequest().getParameter( HybsSystem.NO_TRANSITION_MODE_KEY ), noTransition ) ;
		useAjaxSubmit = nval( getRequest().getParameter( HybsSystem.USE_AJAX_SUBMIT_KEY ), useAjaxSubmit ) ;
		// 4.3.8.0 (2009/08/01) useAjaxSubmit追加 noTransitionがtrueの場合はajaxSubmitはfalse
		if( noTransition ) { useAjaxSubmit = false; }

		setNoTransitionRequest( noTransition );
		setAjaxSubmitRequest( useAjaxSubmit );

		// if( headType == XML_TYPE ) { // 4.2.1.0 (2008/04/16)
		if( headType == HEAD_TYPE_ENUM.xml ){
			jspPrint( XML_HEAD );
			return EVAL_PAGE ;		// ページの残りを評価する。
		}
		// 5.7.1.0 (2013/12/06) ======= ※ ここより下流には、HEAD_TYPE_ENUM.xml の headType は来ない。 =======

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		if( useTabstrip ) { buf.append( TAB_HEAD ); }	// tabstrip 機能を使用する場合
		else {
			// 5.7.1.0 (2013/12/06) HTML5対応
			if( useHTML5 ) { buf.append( HTML5_HEAD1 ).append( headType ).append( HTML5_HEAD2 ); }
			else           { buf.append( HEAD1       ).append( headType ).append( HEAD2       ); }
		}

		// 4.3.7.0 (2009/06/01) IE8対応(IE7互換モード)
		if( useIE7Header ){ buf.append ( IE7_HEAD ); }

		// 5.5.8.0 (2012/11/01) タブレット等の画面サイズ調整METAタグ
		if( useViewPort ){ buf.append( MOBILE_HEAD ); }

		// titleの設定。もし、spanタグでマークアップされていれば、そのBODY部にtitleタグを追加します。
		if( title.startsWith( "<span" ) ) {
			final int adrs1 = title.indexOf( '>' );
			final int adrs2 = title.length() - 7;		// </span> で終了しているはず。
			buf.append( title.substring( 0,adrs1+1 ) ).append( "<title>" );
			buf.append( title.substring( adrs1+1,adrs2 ) ).append( "</title>" );
			buf.append( title.substring( adrs2 ) );
		}
		else {
			buf.append( "  <title>" ).append( title ).append( "</title>" ).append( CR );
		}

		// 5.7.1.0 (2013/12/06) HEAD_TYPE_ENUM.handy の設定を終わらせてしまう。
		if( headType == HEAD_TYPE_ENUM.handy ){
			// CALENDAR_CSS を組み込まない。
			addStyleTag(  DEFAULT_CSS	,buf );
			addStyleTag(  CUSTOM_CSS	,buf );
		}

		// 5.7.1.0 (2013/12/06) HEAD_TYPE_ENUM.simple と handy の設定を終わらせてしまう。
		if( headType == HEAD_TYPE_ENUM.simple || headType == HEAD_TYPE_ENUM.handy ){
			// 以前は、useHrefIcon == true で、HREF_ICON_CSS を組み込んでいたが、組み込まない。

			// 注意：body の直前には、改行は入れないほうが良い。(Tomcat4.1.30 では)
			if( body != null ) { buf.append( body ); }
			buf.append( "</head>" );

			jspPrint( buf.toString() );
			return EVAL_PAGE ;		// ページの残りを評価する。
		}
		// 5.7.1.0 (2013/12/06) ======= ※ ここより下流には、HEAD_TYPE_ENUM.simple と handy headType は来ない。 =======

		// 5.7.4.3 (2014/03/28) 処理の順番の関係で、イベント制御用JSフラグ出力 と入れ替えます。
		addStyleTag( DEFAULT_CSS	,buf );
		addStyleTag( CALENDAR_CSS	,buf );
		addStyleTag( CUSTOM_CSS		,buf );

		// 5.4.5.0 (2012/03/01) イベント制御用JSフラグ出力
		if( headType == HEAD_TYPE_ENUM.query || headType == HEAD_TYPE_ENUM.result ){
			buf.append( "  <script type=\"text/javascript\">" ).append( CR );
			addScriptFlg( REAL_TIME_CHECK_FLG ,useRealTimeCheck ,buf );
			addScriptFlg( FOCUS_HILIGHT_FLG   ,useFocusHilight  ,buf );
			buf.append( "  </script>" ).append( CR );
		}

		// init,query,result,menu タイプで使用されるJavaScript
		if ( headType != null ){
			addScriptTag( JQUERY_JS		,buf ); // 4.3.5.0 (2009/02/01)
			addScriptTag( JQUERY_OG_JS	,buf ); // 4.3.5.0 (2009/02/01)
			addScriptTag( DEFAULT_JS	,buf );
			if( useTabLink ) { addScriptTag( TABLINK_JS	,buf );	} // 4.3.3.0 (2008/10/01)
		}

		// 5.6.0.3 (2013/01/24) EventScrpt前に呼び出すquery専用スクリプト
		if( headType == HEAD_TYPE_ENUM.query ){
			addScriptTag( QUERYCONV_SCRIPT_JS ,buf );
		}

		// query,result タイプで使用されるJavaScript
		if( headType == HEAD_TYPE_ENUM.query || headType == HEAD_TYPE_ENUM.result ){
			addScriptTag( EVENT_SCRIPT_JS, buf );	// 5.2.1.0 (2010/10/01) JavaScript新規統合
			addScriptTag( AUTO_MANUAL_JS , buf );	// 6.2.5.0 (2015/06/05) 自動ヘルプ(showViewManual)のquery.jsp 対応
			addScriptTag( JQUERY_UI_JS   , buf );	// 6.2.6.0 (2015/06/19) helpの移動
			// 4.3.3.0 (2008/10/01)
			if( noTransition || useAjaxSubmit ){ // 4.3.8.0 (2009/08/01)
				addScriptTag( JQUERY_FORM_JS, buf );
				addScriptTag( JQUERY_DRAG_JS, buf );
				addScriptTag( JQUERY_MODAL_JS, buf );
				addStyleTag( JQUERY_MODAL_CSS, buf );
			}

			// 5.6.4.3 (2013/05/24) useTbodyMovable 付け忘れ
			if( useTbodyMovable ) {
				addScriptTag( TBODY_MOVABLE_JS, buf );		// 5.6.3.3 (2013/04/19) useTbodyMovable 追加
			}
		}

		// 5.7.7.1 (2014/06/13) blockUI は、HybsSystem.VIEW_USE_BLOCKUI が、true の場合のみ有効とする。
//		boolean isBlockUi = HybsSystem.sysBool( "VIEW_USE_BLOCKUI" );
//		boolean isBlockUi = HybsSystem.sysBool( "VIEW_USE_BLOCKUI" )
//								&& headType != null
//								&& headType.into( ((HttpServletRequest)getRequest()).getRequestURI() );

		// 4.3.5.0 (2008/02/01) 位置を両タイプで利用される前のインクルードから後に変更
		// query タイプで使用されるJavaScript
		if( headType == HEAD_TYPE_ENUM.query ){
			addScriptTag( QUERY_SCRIPT_JS ,buf );	// 5.2.1.0 (2010/10/01) JavaScript新規統合

			// 5.3.6.0 (2011/06/01) QUERYのFocusの制御を追加
			final boolean isFocus = nval( getRequestValue( HybsSystem.QUERY_FOCUS_KEY ), true );
			// 機能を利用しない場合のみ変数を出力
			if( !isFocus ) { addScriptValue( HybsSystem.QUERY_FOCUS_KEY, "false", buf ); }

			// 5.7.7.1 (2014/06/13) blockUI 対応
			// 5.7.9.0 (2014/08/08) headType="query"の時は、RequestURI に含まれる場合のみ ON する。
//			if( isBlockUi ) {
			if( useBlockUI && headType.into( ((HttpServletRequest)getRequest()).getRequestURI() ) ) {
				addScriptTag( JQUERY_BLOCKUI_JS ,buf );
				addScriptTag( CUSTOM_BLOCKUI_JS ,buf );
			}
		}

		// result タイプで使用されるJavaScript
		if( headType == HEAD_TYPE_ENUM.result ) {
			addScriptTag( RESULT_SCRIPT_JS ,buf );	// 5.2.1.0 (2010/10/01) JavaScript新規統合

			if( useScrollBar	) {
									//  4.3.5.0 (2009/02/01) CSSも読み込む
									addScriptTag( SCROLL_BAR_JS, buf );
									addStyleTag ( SCROLL_BAR_CSS, buf );
			}
			if( useGantt		) { addScriptTag( GANTT_JS			,buf ); }
			if( useTreeBOM		) { addScriptTag( TREEBOM_JS		,buf ); }
			if( useAjaxTreeTable) { addScriptTag( AJAX_TREETABLE_JS ,buf ); }		// 4.3.1.0 (2008/09/02)
			if( noTransition	) {
									addScriptTag( AJAX_HYBS_FORM	,buf );			// 4.3.8.0 (2009/08/01)
									addScriptTag( NO_TRANSITION_JS	,buf );
								  } // 4.3.1.0 (2008/09/02)
			if( useAjaxSubmit	) {
									addScriptTag( AJAX_HYBS_FORM	,buf );
									addScriptTag( AJAX_SUBMIT_JS	,buf );
								  } // 4.3.8.0 (2009/08/01)

			// 5.6.3.2 (2013/04/12) useGantt2 廃止。adjustEvent の引数で使い分ける。
			if( adjustEvent != null ) {
				if( "Bar".equals( adjustEvent ) ) {
					addScriptTag(   AJST_BAR_JS		,buf );
				}
				else if( "Matrix".equals( adjustEvent ) ) {
					addScriptTag(   AJST_MATRIX_JS	,buf );
				}
				else if( "Table".equals( adjustEvent ) || "Gantt".equals( adjustEvent ) || "Task".equals( adjustEvent ) ) {
					addScriptTag(   ADD_BRWS_CLS_JS	,buf );			// 6.0.2.5 (2014/10/31) ブラウザ判定
					addStyleTag (   AJST_GANTT_CSS	,buf );
					addScriptTag(   AJST_GANTT_JS	,buf );
				}
				addEventScript( "adjust" + adjustEvent	,buf );
			}

			// 5.6.3.2 (2013/04/12) addEvent 追加 を、メソッド化
			addEventScript( addEvent,buf );

			// 5.7.6.2 (2014/05/16) IEのHTML5機能が有効か無効かの判定値をsessionに設定します。
			// result タイプ のみで有効にしておきます。
			final String h_HTML5 = getRequest().getParameter( HybsSystem.IE_HTML5_KEY );
			if( h_HTML5 != null ) {
				setSessionAttribute( HybsSystem.IE_HTML5_KEY,h_HTML5 );
				// buf.append( "<div>IE_HTML5=" ).append( h_HTML5 ).append( "</div>" ).append( CR );
			}

			// 5.7.7.1 (2014/06/13) blockUI 対応
			if( useBlockUI ) {
				buf.append( "  <script type=\"text/javascript\">" ).append( CR );
				buf.append( UNBLOCKUI_READY ).append( CR );
				buf.append( "  </script>" ).append( CR );
			}
		}

		// 5.6.5.2 (2013/06/21) useHrefIcon
		if( useHrefIcon ) {
			addStyleTag(  HREF_ICON_CSS	,buf );
		}

		// 注意：body の直前には、改行は入れないほうが良い。(Tomcat4.1.30 では)
		if( body != null ) {	buf.append( body ); }
		buf.append( "</head>" );

		jspPrint( buf.toString() );
		return EVAL_PAGE ;		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 * @og.rev 5.4.5.0 (2012/03/01) realTimeCheck,focusHilight
	 * @og.rev 5.5.8.0 (2012/11/01) useViewPort  追加
	 * @og.rev 5.6.2.1 (2013/03/08) useGantt2  追加
	 * @og.rev 5.6.3.0 (2013/04/01) addEvent 追加
	 * @og.rev 5.6.3.2 (2013/04/12) useGantt2 廃止。adjustEvent の引数で使い分ける。
	 * @og.rev 5.6.3.3 (2013/04/19) useTbodyMovable 追加
	 * @og.rev 5.6.5.2 (2013/06/21) useHrefIcon 追加
	 * @og.rev 5.7.1.0 (2013/12/06) HTML5対応
	 * @og.rev 5.7.9.0 (2014/08/08) useBlockUI 追加
	 * @og.rev 6.0.2.5 (2014/10/31) useGuest 廃止
	 */
	@Override
	protected void release2() {
		super.release2();
		body			= null;
		title			= null ;
		headType		= null;
		useScrollBar	= true;
		useTabstrip		= false;
		useGantt		= false;
		useTreeBOM		= false;
//		useGuest		= false;										// 6.0.2.5 (2014/10/31) 廃止
		useAjaxTreeTable= false;										// 4.3.1.0 (2008/09/02)
		noTransition	= HybsSystem.sysBool( "NO_TRANSITION_MODE" );	// 4.3.1.0 (2008/09/02)
		useAjaxSubmit	= HybsSystem.sysBool( "USE_AJAX_SUBMIT" );
		useRealTimeCheck= true;											// 5.4.5.0 (2012/03/01) 復活
		useTabLink		= false;										// 4.3.3.0 (2008/10/01)
		useFocusHilight	= true;											// 5.4.5.0 (2012/03/01) 復活
		useHrefIcon		= false;										// 5.6.5.2 (2013/06/21) 追加
		useIE7Header    = HybsSystem.sysBool( "USE_IE7_HEADER" );		// 4.3.7.0 (2009/06/01) IE8対応
		useViewPort		= HybsSystem.sysBool( "USE_VIEWPORT_HEADER" );	// 5.5.8.0 (2012/11/01)
		useBlockUI		= HybsSystem.sysBool( "VIEW_USE_BLOCKUI" );		// 5.7.9.0 (2014/08/08)
		addEvent		= null;											// 5.6.3.0 (2013/04/01) addEvent 追加
		adjustEvent		= null;											// 5.6.3.2 (2013/04/12) adjustEvent 追加
		useTbodyMovable	= false;										// 5.6.3.3 (2013/04/19) 新規追加
		useHTML5		= HybsSystem.sysBool( "USE_HTML5_HEADER" );		// 5.7.1.0 (2013/12/06) HTML5対応
	}

	/**
	 * スクリプトを制御するための変数文字列を作成します。
	 * JavaScriptのファイル統合によりファイルを行うか否かでは、機能のON/OFFを
	 * 行うことができません。
	 * このため、機能のON/OFFを制御するための変数を書き出して、JavaScript内で
	 * これをコントロールします。
	 * &lt;script type="text/javascript" &lt;!-- var (key)="(value)"; --&gt;&lt;/script&gt;
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規作成
	 *
	 * @param	key		JavaScript変数名
	 * @param	value	JavaScriptに渡す値
	 * @param	buf		指定文字列を追加するStringBuilder(nullは指定できません)
	 *
	 * @return	追加されたStringBuilder
	 * @og.rtnNotNull
	 */
	private StringBuilder addScriptValue( final String key, final String value, final StringBuilder buf ) {
		if( value != null && value.length() > 0 ) {
			final String tag = "  <script type=\"text/javascript\">"
				+ " var " + key + "=\"" + value + "\"; </script>"
				+ CR ;
			buf.append( tag );
		}
		return buf ;
	}

	/**
	 * スクリプトを制御するためのファイルをインクルードする文字列を作成します。
	 * ファイル名を指定して、JavaScriptの src 属性に設定した 文字列を作成します。
	 * ここで、作成するファイル名には、v=識別番号 のパラメータをセットします。
	 * 識別番号は、ファイルの最終更新時刻をセットします。これが取得できない場合は、
	 * BuildNumber.BUILD_ID を使用します。
	 * このキーで、ファイルが更新されるか、エンジンのバージョンが更新されると
	 * ブラウザキャッシュが更新されることになります。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規作成
	 *
	 * @param	filename	JavaScriptファイル名
	 * @param	buf	指定文字列を追加するStringBuilder
	 *
	 * @return	追加されたStringBuilder
	 * @og.rtnNotNull
	 */
	private StringBuilder addScriptTag( final String filename, final StringBuilder buf ) {
		String tag = srcAdrs.get( filename );
		if( tag == null ) {
			final File file = new File( JSP_DIR + filename );
			// 5.6.2.1 (2013/03/08) ファイルの最終更新時刻が取得できない場合は、BuildNumber.BUILD_ID を使用
			final long ts = file.lastModified();
			// 6.1.0.0 (2014/12/26) refactoring : Avoid if (x != y) ..; else ..;
//			final String tsv = (ts != 0L) ? String.valueOf( ts ) : BuildNumber.BUILD_ID ;
			final String tsv = (ts == 0L) ? BuildNumber.BUILD_ID : String.valueOf( ts ) ;

			tag = "  <script type=\"text/javascript\" src=\""
					+ JSP_ADRS
					+ filename
					+ "?v="
					+ tsv										// 5.6.2.1 (2013/03/08) 
					+ "\" ><!-- --></script>"
					+ CR ;
			srcAdrs.put( filename,tag );
		}
		buf.append( tag );
		return buf ;
	}

	/**
	 * var xxx = true;のような文字列を作成します。
	 * xxxとtrue/falseを外部から与えます。
	 *
	 * @og.rev 5.4.5.0 (2012/03/01) 新規作成
	 *
	 * @param	var	変数名
	 * @param	flg	変数に設定する論理値 [true/false]
	 * @param	buf	指定文字列を追加するStringBuilder
	 *
	 * @return	追加されたStringBuilder
	 * @og.rtnNotNull
	 */
	private StringBuilder addScriptFlg( final String var, final boolean flg, final StringBuilder buf ) {
		buf.append( "    var ")
			.append( var )
			.append( " = ")
			.append( flg )
			.append( ';' )		// 6.0.2.5 (2014/10/31) char を append する。
			.append( CR );
		return buf ;
	}

	/**
	 * stylesheetを制御するためのファイルをインクルードする文字列を作成します。
	 * ファイル名を指定して、stylesheetの href 属性に設定した 文字列を作成します。
	 * ここで、作成するファイル名には、v=識別番号 のパラメータをセットします。
	 * 識別番号は、ファイルの最終更新時刻をセットします。これが取得できない場合は、
	 * BuildNumber.BUILD_ID を使用します。
	 * このキーで、ファイルが更新されるか、エンジンのバージョンが更新されると
	 * ブラウザキャッシュが更新されることになります。
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規作成
	 *
	 * @param	filename	stylesheetファイル名
	 * @param	buf	指定文字列を追加するStringBuilder
	 *
	 * @return	追加されたStringBuilder
	 * @og.rtnNotNull
	 */
	private StringBuilder addStyleTag( final String filename, final StringBuilder buf ) {
		String tag = srcAdrs.get( filename );
		if( tag == null ) {
			final File file = new File( JSP_DIR + filename );
			// 5.6.2.1 (2013/03/08) ファイルの最終更新日が取得できない場合は、BuildNumber.BUILD_ID を使用
			final long ts = file.lastModified();
			// 6.1.0.0 (2014/12/26) refactoring : Avoid if (x != y) ..; else ..;
//			final String tsv = (ts != 0L) ? String.valueOf( ts ) : BuildNumber.BUILD_ID ;
			final String tsv = (ts == 0L) ? BuildNumber.BUILD_ID : String.valueOf( ts ) ;

			tag = "  <link rel=\"stylesheet\" type=\"text/css\" href=\""
					+ JSP_ADRS
					+ filename
					+ "?v="
					+ tsv										// 5.6.2.1 (2013/03/08) 
					+ "\" />"
					+ CR ;
			srcAdrs.put( filename,tag );
		}
		buf.append( tag );
		return buf ;
	}

	/**
	 * addEvent でJavaScript をロードする処理を行う文字列を作成します。
	 * イベント名を外部から与えます。
	 *
	 * @og.rev 5.6.3.2 (2013/04/12) 新規作成
	 *
	 * @param	event	イベント名
	 * @param	buf	指定文字列を追加するStringBuilder
	 *
	 * @return	追加されたStringBuilder
	 * @og.rtnNotNull
	 */
	private StringBuilder addEventScript( final String event, final StringBuilder buf ) {
		if( event != null && event.length() > 0 ) {
			buf.append( "  <script type=\"text/javascript\"> addEvent( this, \"load\", " )
				.append( event )
				.append( " ); </script>" )
				.append( CR );
		}
		return buf ;
	}

	/**
	 * セッションの初期処理を行います。
	 * common/session_init.jsp をタグに組み込みました。
	 *
	 * @og.rev 3.8.0.0 (2005/06/07) 同一セッションでのマルチ起動対策を行います。
	 * @og.rev 3.8.0.4 (2005/08/08) custom を呼び出せるように追加
	 * @og.rev 3.8.1.2 (2005/12/19) 認証無しの場合のユーザー制御修正
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage ⇒ getResource().getLabel )
	 * @og.rev 4.0.0.0 (2007/10/31) ロール指定でのログイン機能追加
	 * @og.rev 4.2.1.0 (2008/04/16) headTypeのenum化
	 * @og.rev 4.2.2.0 (2008/05/28) GUESTユーザの復活
	 * @og.rev 4.2.4.1 (2008/07/08) pageContext.include時にflushしないように修正
	 * @og.rev 4.3.3.1 (2008/10/08) システムパラメータの初期値設定は userInfo 作成ブロックの外で行うように変更します。
	 * @og.rev 4.3.4.1 (2008/12/08) title 属性を使用しなくします。(doStartTag() メソッドで処理)
	 * @og.rev 5.1.4.0 (2010/03/01) lang を書き換え可能とする。
	 * @og.rev 5.1.8.0 (2010/07/01) useGuest属性が有効にならないバグを修正。
	 * @og.rev 5.1.8.0 (2010/07/01) ユーザーIDの大文字・小文字を区別する(大文字化を廃止)
	 * @og.rev 5.2.0.0 (2010/09/01) アクセス禁止アドレスによる不正アクセス防止機能追加
	 * @og.rev 6.0.2.5 (2014/10/31) custom/init.jsp は、なくてもエラーにはしない。
	 */
	private void sessionInit() {
		final HttpSession session = pageContext.getSession();
		final HttpServletRequest request = (HttpServletRequest)pageContext.getRequest();

		// 3.5.3.0 (2003/10/27) encode.jsp 廃止
		try {
			request.setCharacterEncoding( "UTF-8" );
		}
		catch (UnsupportedEncodingException ex) {
			throw new HybsSystemException( ex );
		}

		// 3.8.0.0 (2005/06/07) 同一セッションでのマルチ起動対策を行います。
		final String cntR = request.getParameter( HybsSystem.MULTI_SESSION_CHECK );
		if( cntR != null ) {
			final String cntS = (String)session.getAttribute( HybsSystem.MULTI_SESSION_CHECK );
			if( cntS != null && !cntR.equals( cntS ) ) {
				final ResourceManager resource = ResourceFactory.newInstance( request.getLocale().getLanguage() );

				final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
					.append( resource.getLabel( "ERR0035.T" ) ).append( CR )
					.append( resource.getLabel( "ERR0035.1" ) ).append( CR )
					.append( resource.getLabel( "ERR0035.2" ) ).append( CR )
					.append( resource.getLabel( "ERR0035.3" ) ).append( CR );

				throw new HybsSystemException( buf.toString() );
			}
		}

		// システムパラメータの初期値設定
		// 4.3.3.1 (2008/10/08) 戻るリンクの不具合対応で、後方の userInfo 作成ブロックの外に出してくる。
		if( ! HybsSystem.isParamSet() ) {
			final String	serverName	= request.getServerName();
			final String	scheme		= request.getScheme();
			final int		port		= request.getServerPort();
			final String	contextPath = request.getContextPath();

			HybsSystem.setRequestParam( serverName,scheme,port,contextPath );
		}

		// 3.3.0.0 (2003/06/23) 操作途中以外でのセッション切断に対応。
		// ログイン時の初期化を行います。
		UserInfo userInfo = (UserInfo)getSessionAttribute( HybsSystem.USERINFO_KEY );
		String userID = request.getRemoteUser() ;

		// 3.8.1.2 (2005/12/19)
		if( userInfo == null || ( userID != null && ! userID.equalsIgnoreCase( userInfo.getUserID() ))) {
		// セッションタイムアウトの確認 command が、null,"NEW","INIT" 以外は、エラー
			final String command = request.getParameter( "command" );
			if( command == null || "NEW".equals( command ) || "INIT".equals( command ) ) {
				if( userID == null ) {
					userID = request.getParameter( "USERID" );		// 6.0.2.5 (2014/10/31) 認証なし時のユーザー指定
					if( userID == null ) {
						userID = "GUEST" ;		// 認証されていない場合は、強制的に GUESTユーザー
					}
				}
				userInfo = UserInfoFactory.newInstance( userID,request.getRemoteAddr(),request.getParameter( "ROLES" ) );

				// 5.1.4.0 (2010/03/01) lang を書き換え可能とする。
				final String newLang = request.getParameter( "NEWLANG" ) ;
				if( newLang != null ) { userInfo.setLang( newLang ); }

				session.setAttribute( HybsSystem.USERINFO_KEY, userInfo );

				// セッションタイムアウトを変更
				session.setMaxInactiveInterval( HybsSystem.sysInt( "MAX_INACTIVE_INTERVAL" ) );
				SystemManager.addSession( session );

				// 6.0.2.5 (2014/10/31) custom/init.jsp は、なくてもエラーにはしない。
				try {
					// 4.2.4.1 (2008/07/08)
					pageContext.include( "/jsp/custom/init.jsp", false );
				} catch ( IOException ex ) {
					final String errMsg = "/jsp/custom/init.jsp の include に失敗しました。 " + CR
										+ ex.getMessage()  + CR + toString();
					System.err.println( errMsg );					// 6.0.2.5 (2014/10/31)
	//				throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
				} catch ( ServletException ex ) {
					final String errMsg = "/jsp/custom/init.jsp の include に失敗しました。 " + CR
										+ ex.getMessage()  + CR + toString();
					System.err.println( errMsg );					// 6.0.2.5 (2014/10/31)
	//				throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
				}
			}
			else {
				final ResourceManager resource = ResourceFactory.newInstance( request.getLocale().getLanguage() );

				// エラー表示ページを表示する。4.0.0 (2005/01/31)
				final String msg = String.valueOf( HybsSystem.sysInt( "MAX_INACTIVE_INTERVAL" ) ) ;

				// ERR0002.T=ユーザーログインエラー
				// ERR0002.1=セッションのタイムアウトか，またはユーザーログインエラーが発生しました。
				// ERR0002.2=現在のタイムアウトは {0} 秒にセットされています。
				// ERR0002.3=長時間使用しない場合は,画面を閉じる事により,サーバー側の負荷を軽減する事ができます。
				// ERR0002.4=<h2>もう一度，ログインし直して下さい。</h2>

				final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
					.append( resource.getLabel( "ERR0002.T" ) ).append( CR )
					.append( resource.getLabel( "ERR0002.1" ) ).append( CR )
					.append( resource.getLabel( "ERR0002.2",new String[] { msg } ) ).append( CR )
					.append( resource.getLabel( "ERR0002.3" ) ).append( CR )
					.append( resource.getLabel( "ERR0002.4" ) ).append( CR );

				throw new HybsSystemException( buf.toString() );		// 4.0.0 (2005/01/31)
			}
		}

		boolean useGamenId = false;
		String gamenId = request.getParameter( "GAMENID" );
		final String requestURI = request.getRequestURI();

		final int index2 = requestURI.lastIndexOf( '/' );
		final String jspID   = requestURI.substring( index2+1 );
		if( gamenId == null ) {
			final int index1 = requestURI.lastIndexOf( '/',index2-1 );
			gamenId = requestURI.substring( index1+1,index2 );
		}
		else {
			useGamenId = true;
		}

		final ResourceManager resource = ResourceFactory.newInstance( userInfo.getLang() );
		// 4.0.0 (2004/12/31)
		if( ! userInfo.isGUIInfoSet() ) {
			resource.makeGUIInfos( userInfo );
		}

		GUIInfo guiInfo = null;
		String gui_title ;
		// 3.8.0.4 (2005/08/08) custom を呼び出せるように追加
		if( "menu".equals( gamenId ) || "jsp".equals( gamenId ) || "custom".equals( gamenId ) || "common".equals( gamenId ) ) {
			// 3.5.4.5 (2004/01/23) トップメニューの設定
	 		// 4.3.4.1 (2008/12/08) title 属性を使用しなくします。(doStartTag() メソッドで処理)
			gui_title = HybsSystem.sys( "GUI_TOP_TITLE" );
			if( gui_title == null ) { gui_title = "Hayabusa " + gamenId ; }
		}
		else {
			guiInfo = userInfo.getGUIInfo( gamenId );		// 4.0.0 (2004/12/31)

			if( guiInfo != null ) {
				session.setAttribute( HybsSystem.GUIINFO_KEY, guiInfo );
				session.setAttribute( "JSPID", jspID );
			}
			else {
				// エラー表示ページを表示する。4.0.0 (2005/01/31)
				final String msg = userInfo.getJname();

				final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
					.append( resource.getLabel( "ERR0003.T" ) ).append( CR )
					.append( resource.getLabel( "ERR0002.1",new String[] { msg } ) ).append( CR )
					.append( resource.getLabel( "ERR0002.2" ) ).append( CR );

				throw new HybsSystemException( buf.toString() );		// 4.0.0 (2005/01/31)
			}

			// 画面のタイトルをリクエスト変数でも指定できるようにします。
			// 相当、やっつけ仕事が入っていますが、必ず見直します。
	 		// 4.3.4.1 (2008/12/08) title 属性を使用しなくします。(doStartTag() メソッドで処理)
			gui_title = request.getParameter( "GUI_TITLE" );
			if( gui_title == null || gui_title.isEmpty() ) {
				final String cmd = request.getParameter( "command" );
				if( cmd == null || cmd.equals( "NEW" ) ||
					 cmd.equals( "RENEW" ) || cmd.equals( "INIT" ) ||
					 cmd.equals( "RESET" ) || cmd.isEmpty()  ) {
					gui_title = guiInfo.getLongName();
				}
			}
		}

		// 5.2.0.0 (2010/09/01) アクセス禁止アドレスによる不正アクセス防止機能追加
		if( useGamenId ) {
			final int idx1 = requestURI.lastIndexOf( "/jsp/" ) + 5;
			final int idx2 = requestURI.lastIndexOf( '/' );
			if( idx1 >= 0 && idx2 >= 0 && idx1 < idx2 ) {
				final String reqAddr = requestURI.substring( idx1, idx2 );
				if( reqAddr.indexOf( '/' ) < 0 && !userInfo.isValidAddr( reqAddr ) ) {
					final String msg = userInfo.getJname();
					final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
						.append( resource.getLabel( "ERR0003.T" ) ).append( CR )
						.append( resource.getLabel( "ERR0002.1",new String[] { msg } ) ).append( CR );
					throw new HybsSystemException( buf.toString() );
				}
			}
		}

 		// 4.3.4.1 (2008/12/08) title 属性を使用しなくします。(doStartTag() メソッドで処理)
		if( gui_title != null ) {
			session.setAttribute( "GUI_TITLE", gui_title );
		}
	}

	/**
	 * 【TAG】HTML の タイトルを設定します(初期値:session に"GUI_TITLE" キーで登録されている値)。
	 *
	 * @og.tag
	 *  head タグのBody部に、title タグを設定します。
	 * &lt;title&gt;タイトル &lt;/title&gt; を追加します。
	 * 設定は、このタグでの指定が最優先されます。次に、リクエスト情報の
	 * GUI_TITLE があれば、使用されます。最後に、初期値である、session の
	 * "GUI_TITLE" キーで登録されている値を使用します。
	 * session に登録されている値とは、GUIリソースで定義された、名前(長)です。
	 *
	 * 初期値は、session に"GUI_TITLE" キーで登録されている値です。
	 * トップメニューのタイトルは、システムパラメータ の、GUI_TOP_TITLE キー
	 * で登録されている値です。
	 *
	 * @param	titl	タイトル
	 */
	public void setTitle( final String titl ) {
		title = nval( getRequestParameter( titl ),title );
	}

	/**
	 * 【TAG】ヘッダータイプ[null/xml/init/query/result/menu/simple/handy/frame/popup]を設定します(初期値:null)。
	 *
	 * @og.tag
	 * ヘッダータイプ とは、個別の設定をしなくても、基本的なバリエーションを
	 * 簡易的に設定する場合に使用するキーワードです。
	 * ここでは、null,xml,init,query,result,menu,simple,handy,frame,popup の指定ができます。
	 * query を指定すると、query.jsp で使用する基本的な設定を再現できます。
	 * timeSet.js , toggleFrame.js , inputFocus.js が標準で設定されます。
	 * result も同様で、result.jsp で使用する基本的な設定を再現できます。
	 * hilightRow.js , ScrollBar.js , printLine.js ,
	 * timeCheck.js , inputFocus.js が標準で設定されます。
	 * inputFocus.js に関しては、query,result 両タイプとも設定されます。
	 * init の場合は、通常はユーザーに見えない画面ですが、登録時エラーで
	 * 処理が停止する場合があり、CSSファイルのみ組み込む必要があります。
	 * menu はmultiMenu.jsp等のメニュー画面用の設定です。
	 * なにも指定しない場合(null)では、index.jsp , forward.jsp などの、
	 * 直接画面に表示されないタイプのヘッダーになります。
	 * simple は何もインクルードしないタイプです。携帯電話用画面等でdefault.css
	 * すらインクルードしたくないが、JSPの作りは同じにしておきたい場合に利用します。
	 * xml は、Webサービスや、Ajax などで使用する XML レスポンス用の
	 * 特別なヘッダーを作成します。
	 * handy は、ハンディーターミナル等の必要最小限のインクルードを行います。
	 * frame は、index.jsp を想定した、HTML5適用時に、frameset を使う箇所のみHTML4 のタグを出力します。
	 * popup は、popup.jsp を想定した、キーワードですが、現時点では、result ＋ baseタグ出力です。
	 * 初期値は、null です。
	 *
	 * @og.rev 4.2.1.0 (2008/04/16) headTypeのenum化に伴う変更
	 * @og.rev 5.4.4.2 (2012/02/03) simpleタイプ追加
	 * @og.rev 5.7.1.0 (2013/12/06) handyタイプ追加(JavaDoc修正)
	 * @og.rev 5.7.4.3 (2014/03/28) frame,popup追加
	 *
	 * @param	type	ヘッダータイプ [null/xml/init/query/result/menu/simple/handy/frame/popup]
	 */
	public void setHeadType( final String type ) {
		String hType = nval( getRequestParameter( type ),null );
		if( "null".equals( hType )){ // nullを入れた時
			hType = null;
		}
		if( hType != null ) {
			try {
				headType = HEAD_TYPE_ENUM.valueOf( hType );
			}
			catch (IllegalArgumentException  iaex) {
				final StringBuilder errBuf = new StringBuilder( BUFFER_MIDDLE );
				errBuf.append( "headType は" );
				for( final HEAD_TYPE_ENUM obj : HEAD_TYPE_ENUM.values() ) {
					errBuf.append( ',' );
					errBuf.append( obj.name() );
				}
				errBuf.append("から選んでください。");
				throw new HybsSystemException( errBuf.toString(),iaex );	// 4.3.1.1 (2008/08/23)
			}
		}

		// 5.7.4.3 (2014/03/28) headType="frame" 時は、従来のHTML4 を出力します。
		if( headType == HEAD_TYPE_ENUM.frame ) {
			useHTML5 = false;
			// index.jsp を想定しているため、従来の headType=null と同じ動きをさせます。
			headType = null;
		}

		// 5.7.4.3 (2014/03/28) headType="popup" 時は、BODY に、POPUP_BODY を出力します。
		if( headType == HEAD_TYPE_ENUM.popup ) {
			body = POPUP_BODY ;
			// popup は、query.jsp と result.jsp を合成するが、result と同じでよい。
			headType = HEAD_TYPE_ENUM.result;
		}
	}

	/**
	 * 【TAG】ScrollBar.js JavaScript を使用するかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、ScrollBar.js を使用するかどうかを指定します。
	 * (従来のTYPE2です。)
	 * result タイプで有効です。
	 * 初期値は、使用する(true)です。
	 *
	 * @param	flag	スクロールバー使用可否 [true:使用する/false:使用しない]
	 */
	public void setUseScrollBar( final String flag ) {
		useScrollBar = nval( getRequestParameter( flag ),useScrollBar );
	}

	/**
	 * 【TAG】tabstrip.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、tabstrip.js を使用するかどうかを指定します。
	 * init,query,result タイプで有効です。
	 * この機能は特別な為、初期値は、使用しない(false)です。
	 *
	 * @param	flag	タブストリップ使用可否 [true:使用する/false:使用しない]
	 */
	public void setUseTabstrip( final String flag ) {
		useTabstrip = nval( getRequestParameter( flag ),useTabstrip );
	}

	/**
	 * 【TAG】option/gantt.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、option/gantt.js を使用するかどうかを指定します。
	 * result タイプで有効です。
	 * この機能は特別な為、初期値は、使用しない(false)です。
	 *
	 * @param	flag	ガント使用可否 [true:使用する/false:使用しない]
	 */
	public void setUseGantt( final String flag ) {
		useGantt = nval( getRequestParameter( flag ),useGantt );
	}

	/**
	 * 【TAG】option/treeBOM.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、option/treeBOM.js を使用するかどうかを指定します。
	 * result タイプで有効です。
	 * この機能は特別な為、初期値は、使用しない(false)です。
	 *
	 * @param	flag	ツリーBOM使用可否 [true:使用する/false:使用しない]
	 */
	public void setUseTreeBOM( final String flag ) {
		useTreeBOM = nval( getRequestParameter( flag ),useTreeBOM );
	}

//	/**
//	 * 【TAG】ゲストユーザを利用するかどうか[true/false]を指定します(初期値false)。
//	 *
//	 * @og.tag
//	 *
//	 * ユーザIDが指定されていなかった場合にユーザのIDをGUESTとしてログインさせるかを指定します。
//	 * TOMCATの認証機能を外した領域でユーザを限定せずにログインさせたい場合等に利用します。
//	 * なお、システム定数RESOURCE_USER_SQLで帰ってくる結果に「GUEST」IDのユーザが必要です。
//	 * 初期値はfalse(利用しない)となります。
//	 *
//	 * @param	flag	ゲストユーザを使用するかどうか [true:使用する/false:使用しない]
//	 *
//	 * @og.rev 4.2.2.0 (2008/05/28) 新規作成
//	 * @og.rev 6.0.2.5 (2014/10/31) 廃止
//	 */
//	public void setUseGuest( final String flag ) {
//		useGuest = nval( getRequestParameter( flag ),useGuest );
//	}

	/**
	 * 【TAG】Ajaxを利用したツリー形式のテーブル表示を使用するかを指定します。
	 *
	 * @og.tag
	 * Ajaxを利用したツリー形式のテーブル表示を使用するかを指定します。
	 * viewタグのviewFormType="HTMLAjaxTreeTable"を使用する場合は、必ずtrueを指定して下さい。
	 *
	 * また、ツリー形式のテーブルに対して登録を行う場合は、noTransition属性も
	 * 同時にtrueにしておく必要があります。
	 *
	 * 初期値は、false(使用しない)です。
	 *
	 * @og.rev 4.3.1.0 (2008/09/02) 新規追加
	 *
	 * @param   flag Ajaxツリーテーブル使用 [true:する/false:しない]
	 */
	public void setUseAjaxTreeTable( final String flag ) {
		useAjaxTreeTable = nval( getRequestParameter( flag ), useAjaxTreeTable );
	}

	/**
	 * 【TAG】画面遷移を行わない形式の登録方法を使用するかを指定します
	 *		(初期値:NO_TRANSITION_MODE[={@og.value org.opengion.hayabusa.common.SystemData#NO_TRANSITION_MODE}])。
	 *
	 * @og.tag
	 * 画面遷移なしの登録を行うかどうかを指定します。
	 * trueが指定された場合、画面遷移を行わず選択行のみを動的に切り替えることで、
	 * 編集、登録を行います。サーバーへのデータ送信には、jQueryのFormPluginを
	 * 利用しています。
	 * また、データを送信する対象のフォームには、id="noTransition"を指定して下さい。
	 * (該当のIDのフォームが存在しない場合は、JSPの1番目のフォームが送信されます。)
	 * (初期値:システム定数のNO_TRANSITION_MODE[={@og.value org.opengion.hayabusa.common.SystemData#NO_TRANSITION_MODE}])。
	 *
	 * システムパラメーターまたは、このタグの属性でtrueに指定された場合でも、submitされた
	 * commandがINSERT,COPY,MODIFY,DELETE,ENTRY,UNDOの場合のみ、バックグラウンドで
	 * 実行されます。
	 * それ以外のコマンドでは通常通り、画面遷移を伴うsubmitとなります。
	 *
	 * また、システムパラメーターまたは、このタグの属性でtrueに指定された場合、バックグラウンド
	 * で実行されるJSPには、その値が引き継がれるため、これらのJSPの各タグでtrueを指定する
	 * 必要はありません。
	 *
	 * @og.rev 4.3.1.0 (2008/09/02) 新規追加
	 * @og.rev 4.3.8.0 (2009/08/01) noTransitionにnullを許容する
	 * @og.rev 5.1.3.0 (2010/02/01) リクエストに戻す
	 *
	 * @param   noTrnstn 画面遷移を行わない方法を使用するか [true:画面遷移なし/false:通常方法]
	 */
	public void setNoTransition( final String noTrnstn ) {
		noTransition = nval( getRequestParameter( noTrnstn ), noTransition );
	}

	/**
	 * 【TAG】og:tabLinkタグで使用される common/tagLink.js をインクルードするかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * og:tabLinkタグで使用される common/tagLink.js をインクルードするかを指定します。
	 * trueが指定された場合、該当のJavaScriptをインクルードします。
	 * og:tabLinkを使用する場合は、必ずtrueを指定して下さい。
	 * 初期値は、false(インクルードしない)です。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規追加
	 *
	 * @param   flag タブリンク使用可否 [true:する/false:しない]
	 */
	public void setUseTabLink( final String flag ) {
		useTabLink = nval( getRequestParameter( flag ), useTabLink );
	}

	/**
	 * 【TAG】カラムチェックをリアルタイムで行うかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * &lt;del&gt;common/realTimeCheck.jsをインクルードするかの指定です。
	 * インクルードするとカラムチェックがリアルタイムで行われます。
	 * (フォーカスが離れた時に動作)
	 * &lt;del&gt;
	 * リアルタイムチェックイベントを無効にするかどうかを決めます。
	 * falseを指定すると、リアルタイムチェックを行いません。
	 * query,result 両タイプで有効です。
	 * このチェックはDBTypeを用いた簡易的なチェックです。
	 * チェックにかかった場合はバルーンで警告が表示されます。
	 * &lt;del&gt;初期値は、システムリソースのUSE_REAL_TIME_CHECKで設定することが可能です。&lt;del&gt;
	 * 初期値はtrue(利用する)です。
	 *
	 * @og.rev 4.3.5.0 (2009/02/01) 新規追加
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 * @og.rev 5.4.5.0 (2012/03/01) 復活 ローカルリソース時等で使いたくない場合がある
	 *
	 * @param   flag リアルタイムカラムチェック使用可否 [true:する/false:しない]
	 */
	public void setUseRealTimeCheck( final String flag ) {
		useRealTimeCheck = nval( getRequestParameter( flag ), useRealTimeCheck );
	}

	/**
	 * 【TAG】フォーカスが当っている入力ボックスの色を変更するかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * &lt;del&gt;標準組み込みJavaScriptである、common/focusHilight.jsをインクルードするかの指定です。
	 * フォーカスが当っている入力ボックスの背景色を変更します。
	 * インクルードすると自動的に有効になります。
	 * &lt;del&gt;
	 * 入力ボックス色変更JavaScriptのイベントを無効にするフラグの制御を行います。
	 * falseを指定するとイベントを無効にします。
	 * ポップアップ時にプルダウンが正常動作しない場合はこれをfalseとしてください。
	 * query,result 両タイプで有効です。
	 * 初期値は使用する(true)です。
	 *
	 * @og.rev 4.3.5.0 (2009/02/01) 新規追加
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 * @og.rev 5.4.5.0 (2012/03/01) モーダルダイアログのプルダウン挙動対応で復活
	 *
	 * @param   flag focusHilight.jsをインクルード [true:する/false:しない]
	 */
	public void setUseFocusHilight( final String flag ) {
		useFocusHilight = nval( getRequestParameter( flag ), useFocusHilight );
	}

	/**
	 * 【TAG】href(リンク)の拡張子に応じたアイコンを表示するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 標準組み込みCSSである、hrefIcon.cssをインクルードするかの指定です。
	 * これは、href(リンク)の拡張子に応じたアイコンを、jsp/image/ext フォルダ以下の画像ファイルと
	 * 拡張子と関連付けて、アイコン画像をリンクの頭に表示します。
	 * 初期値は使用しない(false)です。
	 *
	 * @og.rev 5.6.5.2 (2013/06/21) 新規追加
	 *
	 * @param   flag リンク拡張子アイコン表示 [true:する/false:しない]
	 */
	public void setUseHrefIcon( final String flag ) {
		useHrefIcon = nval( getRequestParameter( flag ), useHrefIcon );
	}

	/**
	 * 【TAG】検索中表示を行うかどうかを[true/false]を指定します
	 *		(初期値:VIEW_USE_BLOCKUI[={@og.value org.opengion.hayabusa.common.SystemData#VIEW_USE_BLOCKUI}])。
	 *
	 * @og.tag
	 * 検索中表示は、headType="query" で、jquery/blockUI.js を呼出し、queryButtonSubmit の click で、
	 * 起動されます。ただし、result風画面(例えば、エントリー系)では、想定通りの動きをしない場合が
	 * ありますので、特別に、headType="query" の場合は、getRequestURI() から、JSP画面が、query.jsp の
	 * 場合のみに限定しています。
	 * 解除は、headType="request" 時に、window.parent.frames['QUERY']; を見つけて、fnUnblockUI() を呼び出します。
	 * 一応、３ペインを想定して、上記オブジェクトが見つからない場合、window.parent.parent.frames['QUERY']; も
	 * 実行します。(親フレームのさらに親フレーム。ただし、名称は、QUERY でなければなりません。)
	 * この属性は、上記の条件をプラスしたうえで適用されます。つまり、true に設定しても強制的に
	 * 検索中表示を出すことはできません。逆に、false にすると、強制的に、検索中表示を出さなくします。
	 * (初期値:システム定数のVIEW_USE_BLOCKUI[={@og.value org.opengion.hayabusa.common.SystemData#VIEW_USE_BLOCKUI}])。
	 *
	 * @og.rev 5.7.9.0 (2014/08/08) 新規追加
	 *
	 * @param   flag 検索中表示使用可否 [true:する/false:しない]
	 */
	public void setUseBlockUI( final String flag ) {
		useBlockUI = nval( getRequestParameter( flag ), useBlockUI );
	}

	/**
	 * 【TAG】IE8でIE7エミュレートモードで動作させるタグを出力するかどうか[true/false]を指定します
	 *		(初期値:USE_IE7_HEADER[={@og.value org.opengion.hayabusa.common.SystemData#USE_IE7_HEADER}])。
	 *
	 * @og.tag
	 * IE8でIE7エミュレートモード動作をさせるためのMETAタグを出力するか指定します。
	 * (初期値:システム定数のUSE_IE7_HEADER[={@og.value org.opengion.hayabusa.common.SystemData#USE_IE7_HEADER}])。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01)
	 *
	 * @param   flag IE7エミュレートモード [true:使用する/false:使用しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_IE7_HEADER
	 */
	public void setUseIE7Header( final String flag ) {
		useIE7Header = nval( getRequestParameter( flag ), useIE7Header );
	}

	/**
	 * 【TAG】タブレット等の画面サイズ調整METAタグ(viewport）を出力するかどうか[true/false]を指定します
	 *		(初期値:USE_VIEWPORT_HEADER[={@og.value org.opengion.hayabusa.common.SystemData#USE_VIEWPORT_HEADER}])。
	 *
	 * @og.tag
	 * タブレット等の画面サイズ調整METAタグ(viewport）を出力するかどうか[true/false]を指定します。
	 * trueを指定すると、METAタグで
	 * content="width=device-width, initial-scale=1" name="viewport"
	 * の値が出力されます。
	 * (初期値:システム定数のUSE_VIEWPORT_HEADER[={@og.value org.opengion.hayabusa.common.SystemData#USE_VIEWPORT_HEADER}])。
	 *
	 * @og.rev 5.5.8.0 (2012/11/01) 新規作成
	 *
	 * @param   flag タブレット画面サイズ調整するか [true:する/false:しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_VIEWPORT_HEADER
	 */
	public void setUseViewPort( final String flag ) {
		useViewPort = nval( getRequestParameter( flag ), useViewPort );
	}

	/**
	 * 【TAG】フォームのSubmit処理をJavaScriptを利用して行うか(ポップアップエラー表示)
	 *		(初期値:USE_AJAX_SUBMIT[={@og.value org.opengion.hayabusa.common.SystemData#USE_AJAX_SUBMIT}])。
	 *
	 * @og.tag
	 * この属性をtrueにセットする事で、ajaxを利用したsubmitを行います。
	 * submit後のページでエラーが発生した場合は画面が切り替わらずに
	 * ポップアップ形式でエラー表示されます(command=ENTRYの場合のみ)。
	 * 動的カラム(eventColumn属性)をRESULT部分で利用する場合は、戻るボタンが
	 * 正常に動作しないため、この属性をtrueにする方が望ましいです。
	 *
	 * データを送信する対象のフォームには、id="ajaxSubmit"を指定して下さい。
	 * (該当のIDのフォームが存在しない場合は、JSPの1番目のフォームが送信されます。)
	 *
	 * システムパラメーターまたは、このタグの属性でtrueに指定された場合でも、submitされた
	 * commandがENTRYの場合のみ、バックグラウンドで実行されます。
	 * それ以外のコマンドでは通常通り、画面遷移を伴うsubmitとなります。
	 *
	 * また、システムパラメーターまたは、このタグの属性でtrueに指定された場合、バックグラウンド
	 * で実行されるJSPには、その値が引き継がれるため、これらのJSPの各タグでtrueを指定する
	 * 必要はありません。
	 *
	 * (初期値:システム定数のUSE_AJAX_SUBMIT[={@og.value org.opengion.hayabusa.common.SystemData#USE_AJAX_SUBMIT}])。
	 *
	 * @og.rev 4.3.8.0 (2009/08/01) 新規追加
	 * @og.rev 5.1.3.0 (2010/02/01) リクエストに戻す
	 *
	 * @param   flag ajaxSubmit使用 [true:する/false:しない]
	 */
	public void setUseAjaxSubmit( final String flag ) {
		useAjaxSubmit = nval( getRequestParameter( flag ), useAjaxSubmit );
	}

	/**
	 * 【TAG】javaScript を初期ロードし、起動するメソッド名を指定します(初期値:null)。
	 *
	 * @og.tag
	 * これは、addEvent( this, "load", method ); をJavaScript として組み込みます。
	 *
	 * @og.rev 5.6.3.0 (2013/04/01) 新規作成
	 *
	 * @param   method javaScriptを初期ロードし起動するメソッド名
	 */
	public void setAddEvent( final String method ) {
		addEvent = nval( getRequestParameter( method ), addEvent );
	}

	/**
	 * 【TAG】javaScript を初期ロードし、起動する定義名を指定します(初期値:null)。
	 *
	 * @og.tag
	 * これは、指定の定義に対応する、javaScript を組み込みます。
	 * Bar、 Matrix、 Table、 Gantt、 Task が指定できます。
	 * これらを組み込む際には、useGantt = "false" , useScrollBar = "false" も自動セットされます。
	 * 組み込まれるJavaScript,CSSファイルは、jsp/common/option/ 以下にあります。
	 * <table border="1" frame="box" rules="all" >
	 *   <caption>組み込みJavaScriptの説明</caption>
	 *   <tr><th>利用タグ    </th><th>タグ機能          </th><th>adjustEvent</th><th>組み込まれるJavaScript</th><th>組み込まれるCSS     </th><th>初期ロードされるメソッド名</th></tr>
	 *   <tr><th>og:iHead    </th><th>ヘッダー補足      </th><th>Bar        </th><th>adjustBar.js          </th><th>                    </th><th>adjustBar   </th></tr>
	 *   <tr><th>og:iBar     </th><th>縦棒（明細）      </th><th>Bar        </th><th>adjustBar.js          </th><th>                    </th><th>adjustBar   </th></tr>
	 *   <tr><th>og:iMatrix  </th><th>配置図／座席表    </th><th>Matrix     </th><th>adjustMatrix.js       </th><th>                    </th><th>adjustMatrix</th></tr>
	 *   <tr><th>og:iTable   </th><th>左右分割          </th><th>Table      </th><th>adjustGanttTable.js   </th><th>adjustGanttTable.css</th><th>adjustTable </th></tr>
	 *   <tr><th>og:iGantt   </th><th>ガント処理        </th><th>Gantt      </th><th>adjustGanttTable.js   </th><th>adjustGanttTable.css</th><th>adjustGantt </th></tr>
	 *   <tr><th>og:iGanttBar</th><th>ガント積上        </th><th>Gantt      </th><th>adjustGanttTable.js   </th><th>adjustGanttTable.css</th><th>adjustGantt </th></tr>
	 *   <tr><th>og:iTask    </th><th>ガント積上(内部)  </th><th>Task       </th><th>adjustGanttTable.js   </th><th>adjustGanttTable.css</th><th>adjustTask  </th></tr>
	 * </table>
	 *
	 * @og.rev 5.6.3.2 (2013/04/12) 新規作成
	 *
	 * @param   ajEvent javaScriptを初期ロードし起動する定義名
	 */
	public void setAdjustEvent( final String ajEvent ) {
		adjustEvent = nval( getRequestParameter( ajEvent ), ajEvent );

		if( adjustEvent != null ) {
			if( check( adjustEvent , ADJUST_EVENT_LIST ) ) {
				useGantt     = false;
				useScrollBar = false;
			}
			else {
				final String errMsg = "adjustEvent に指定する値は、[" + ADJUST_EVENT_LIST + "] から選んでください。"
									+ CR
									+ " adjustEvent = " + adjustEvent ;
				throw new HybsSystemException( errMsg );
			}
		}
	}

	/**
	 * 【TAG】option/ogTbodyMovable.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * option組み込みJavaScriptである、ogTbodyMovable.js を使用するかどうかを指定します。
	 * これは、id="viewTable" の tbody をマウス、またはカーソルキーで上下に動かす動作を行います。
	 * この動作と、TbodyMovableTag を組み合わせれば、左右分割等で同期をとりながら上下に動かしたり、
	 * 値を入れ替える（つまり、元の位置から動かない）や＋１、－１するなどの機能を使う事で、
	 * 視覚的に上下を入れ替えたのその値を登録することが可能になります。
	 * ガントテーブルでのマウスでの入れ替えなどに効果的です。
	 * 初期値は、使用しない(false)です。
	 *
	 * @og.rev 5.6.3.3 (2013/04/19) 新規作成
	 *
	 * @param   flag 行上下移動の使用可否 [true:する/false:しない]
	 * @see		org.opengion.hayabusa.taglib.TbodyMovableTag
	 */
	public void setUseTbodyMovable( final String flag ) {
		useTbodyMovable = nval( getRequestParameter( flag ),useTbodyMovable );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "body"		,body		)
				.println( "title"		,title		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
