/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.taglet;

import com.sun.tools.doclets.Taglet;
import com.sun.javadoc.Tag;
import java.util.Map;

// import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * ソースコメントから、タグリブマニュアルの属性クラス定義(DOC03)へのリンクを作成します。
 * これは、入力可能なキーがサブクラスで指定したクラス名（の一部）になるため、
 * 動的に変化します。タグリブマニュアル中に記述するのが煩雑になる為、タグ化します。
 * Taglet インターフェースの実装クラスを作成します。
 * og.doc3 タグ(属性クラスリンク)を処理します。
 *
 * このタグは、インラインタグ 定義されているため、@og.tag などに含まれると処理されません。
 * そこで、 DocletTagWriter#printTag( final Tag[] tag ) に処理を追加する必要があります。
 *
 * @og.rev 5.6.3.3 (2013/04/19) 新規作成
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TagletDoc03Link extends AbstractTaglet {

	private static final String NAME   = "og.doc03Link";

	/**
	 * このタグレットがインラインタグで
	 * 使用されている場合、true を返します。
	 * そうでない場合については false が設定されます。
	 *
	 * @return インラインタグの場合は true、そうでない場合は false を返します
	 */
	@Override
	public boolean isInlineTag() {
		return true;
	}

	/**
	 * 実行時にドックレットがタグレットを読み込んで使用するには、
	 * そのタグレットが、次のシグニチャでマップ を引数として受け取る、
	 * レジスタ と呼ばれる static メソッドをもっている必要があります。
	 * このメソッドは、タグレット名をキーとして、カスタムタグレットの
	 * インスタンスをマップに追加します。 タグレットをオーバーライドする場合、
	 * 名前の競合を避けるため、新しいタグレットのインスタンスをマップに
	 * 追加する前に、オーバーライドされる側のタグレットをマップから
	 * 削除する必要があります。
	 *
	 * @param tagletMap タグレットマップ
	 */
	public static void register( final Map<String,Taglet> tagletMap ) {
	   final TagletDoc03Link tagTag = new TagletDoc03Link();
	   final Taglet tag = tagletMap.get(NAME);
	   if(tag != null) {
		   tagletMap.remove(NAME);
	   }
	   tagletMap.put(NAME, tagTag);
	}

	/**
	 * このカスタムタグの名前を返します。
	 *
	 * @return カスタムタグの名前
	 */
	@Override
	public String getName() {
		return NAME;
	}

	/**
	 * このカスタムタグのタグ表現を受け取り、
	 * 文字列としての表現を返し、生成されたページに出力します。
	 *
	 * @param tagTag このカスタムタグのタグ表現
	 *
	 * @return このタグの文字列としての表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString( final Tag tagTag ) {
//		return DocletUtil.doc03LinkTag( tagTag );
		return DocletUtil.doc03LinkTag( tagTag.text() );
	}

	/**
	 * このカスタムタグのタグ表現の配列を受け取り、
	 * 文字列としての表現を返し、生成されたページに出力します。
	 * このタグレットがインラインタグを表す場合、
	 * このメソッドは null を返します。
	 *
	 * @param tagTags	このカスタムタグを表すタグの配列
	 *
	 * @return このタグの文字列としての表現
	 * @see		com.sun.tools.doclets.Taglet#toString( Tag[] )
	 */
	@Override
	public String toString( final Tag[] tagTags ) {
		if(tagTags.length == 0) {
			return null;
		}

		final StringBuilder result = new StringBuilder( BUFFER_MIDDLE );
		for(int i = 0; i < tagTags.length; i++) {
			result.append( DocletUtil.doc03LinkTag( tagTags[i] ) );
		}
		return result.toString();
	}
}
