/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.db;

import org.opengion.fukurou.util.ApplicationInfo;

/**
 * Transaction インターフェースを継承した、リアルタイムトランザクションクラスです。
 *
 * これは、通常のトランザクションクラスと異なり、トランザクション処理をしません。
 * つまり、トランザクション処理を行わないときに、メソッド等の共通的な使用ができるようにする、
 * Connection のラップクラスになります。
 * ただし、このクラスが生成されてから、クローズされるまでは、内部に保留した Connection は、
 * ずっと使い続けられますので、ConnectionFactory から取り出した Connection と同様に
 * トランザクション性は維持されます。
 * このクラスより作成された、Statement は、取り出し側で、きちんとクロース処理を
 *
 * このクラスでは、コミット、ロールバック、クローズ処理は、リアルタイムに行われます。
 *
 * @og.rev 5.1.9.0 (2010/08/01) 新規作成
 * @og.rev 5.3.8.0 (2011/08/01) クラスの内部構造変更
 *
 * @version  5.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class TransactionReal extends TransactionImpl {

	/**
	 * ApplicationInfo を指定して作成する、コンストラクター
	 *
	 * このクラスは、トランザクション処理をしない場合に、従来の Connection の
	 * 代わりに使用することを想定したクラスのオブジェクトを作成します。
	 *
	 * @og.rev 5.3.7.0 (2011/07/01) dbidを引数から削除
	 * @og.rev 5.3.8.0 (2011/08/01) 親クラスを呼ぶように変更
	 *
	 * @param	appInfo	内部統制用のアクセス情報
	 */
	public TransactionReal( final ApplicationInfo appInfo ) {
		super( appInfo );
	}

	/**
	 * コネクションの、終了時処理を行います。
	 *
	 * 引数は、正常かどうかを判定するフラグです。異常の場合は、true をセットします。
	 * これは、ConnectionFactory のプールに戻すかどうかを判断するのに使われます。
	 * 一度でも、エラーが発生したコネクションは、破棄します。
	 * それ以外は、プールに戻します。
	 *
	 * @og.rev 5.3.7.0 (2011/07/01) close時に、コネクションを null 化しておく。
	 * @og.rev 5.3.8.0 (2011/08/01) 終了処理を行い、親クラスのrealClose() を呼ぶ。
	 *
	 * @param	errFlag  [true:エラー状態/false:通常]
	 *
	 * @return 正常:true/異常:false
	 */
	@Override
	public boolean close( final boolean errFlag ) {
		// 5.3.8.0 (2011/08/01) 終了処理を行い、親クラスのrealClose() を呼ぶ。
		super.close( errFlag );
		finish();
		realClose();

		return true;
	}
}
