/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.transfer;

import org.opengion.fukurou.util.StringUtil;

/**
 * 伝送要求に対してのHTTP経由で旧伝送DB(CB01)の読取を行います。
 *
 * 読取対象は、(データコード) (送り先) (テキスト種別) ([プロトコル]://[ホスト名]:[ポート番号]/[コンテキスト名]/)
 * の形式で指定します。
 *
 * 例)読取対象 : 3 D9 B991 http://localhost:8824/gf/
 *
 * その他の処理内容については、{@link org.opengion.fukurou.transfer.TransferRead_HTTP}及び
 * {@link org.opengion.fukurou.transfer.TransferRead_CB01}のJavaDocを参照して下さい。
 *
 * @og.group 伝送システム
 *
 * @version  5.0
 * @author   Hiroki.Nakamura
 * @since    JDK1.6
 */
public class TransferRead_HTTP_CB01 extends TransferRead_HTTP {

	private String remoteHost	;		// リモート接続先URL
	private String remoteReadObj;	// リモート接続先の読取対象

	/**
	 * ローカルの読取対象を、リモート接続先の読取対象とリモート接続先URLに分解します。
	 *
	 * @param localReadObj ローカルの読取対象
	 */
	@Override
	protected void splitReadObj( final String localReadObj ) {
		final String[] obj = StringUtil.csv2Array( localReadObj, ' ' );
		if( obj.length < 4 ) {
			final String errMsg = "読取対象は、(データコード) (送り先) (テキスト種別) ([プロトコル]://[ホスト名]:[ポート番号]/[コンテキスト名]/) の形式で指定して下さい。[READOBJ=" + localReadObj + "]";
			throw new RuntimeException( errMsg );
		}
		final String hcdd = obj[0];
		final String hto  = obj[1];
		final String hsyu = obj[2];
		remoteHost = obj[3];
		if( hcdd == null || hcdd.isEmpty()
		 || hto  == null || hto.isEmpty()
		 || hsyu == null || hsyu.isEmpty()
		 || remoteHost == null || remoteHost.isEmpty() ) {
			final String errMsg = "読取対象は、(データコード) (送り先) (テキスト種別) ([プロトコル]://[ホスト名]:[ポート番号]/[コンテキスト名]/) は必須です。[READOBJ=" + localReadObj + "]";
			throw new RuntimeException( errMsg );
		}

		remoteReadObj = hcdd + " " + hto + " " + hsyu;
	}

	/**
	 * リモート接続先URLを返します。
	 * このメソッドは、{@link #splitReadObj(String)}の後に呼び出しする必要があります。
	 *
	 * @return リモート接続先URL
	 */
	@Override
	public String getRemoteHost() {
		if( remoteHost == null || remoteHost.isEmpty() ) {
			final String errMsg = "先に#splitReadObj(String)を実行して下さい";
			throw new RuntimeException( errMsg );
		}
		return remoteHost;
	}

	/**
	 * リモート接続先の読取対象を返します。
	 * このメソッドは、{@link #splitReadObj(String)}の後に呼び出しする必要があります。
	 *
	 * @return 接続URL
	 */
	@Override
	public String getRemoteReadObj() {
		if( remoteHost == null || remoteHost.isEmpty() ) {
			final String errMsg = "先に#splitReadObj(String)を実行して下さい";
			throw new RuntimeException( errMsg );
		}
		return remoteReadObj;
	}
}
