/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.fukurou.util.LogWriter;
import org.opengion.fukurou.util.FileString;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)
import org.opengion.fukurou.util.StringUtil ;					// 6.2.0.0 (2015/02/27)

import javax.servlet.ServletRequest ;
import javax.servlet.http.HttpServletRequest ;

/**
 * BODY部に記述されたエンジン固有の文字列({&#064;XXXX}など)を、
 *  ユーザー情報のエンコーディングに変換するタグです。
 *
 * XML形式で 日本語コードのパースが、JSPエンジン(Tomcat)でサポート
 * されるまでの、暫定的なタグです。
 * なお、このタグの内部に存在するカスタムタグは、先に実行されるため
 * 漢字コードは、変換されません。
 *
 * @og.formSample
 * ●形式：&lt;og:text &gt;･･･&lt;/og:text&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:text
 *       value              【TAG】value 値に直接書かれた漢字コードをShift_JIS に変換します
 *       include            【TAG】動的にファイルを include します
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:text&gt;
 *
 * ●使用例
 *      ・&lt;og:text &gt;
 *            &lt;p&gt;あいおえお：&lt;input name="PN" value="{&#064;PN}" /&gt; &lt;/p&gt;
 *        &lt;/og:text&gt;
 *      ・&lt;og:text value="あいうえお" /&gt;
 *
 *     動的にファイルを include することが出来ます。
 *      ・&lt;og:text include="{&#064;query}.txt" /&gt;
 *
 * @og.group 画面部品
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TextTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0.0 (2007/11/30)" ;

	private static final long serialVersionUID = 400020071130L ;

	private String  value		;
	private boolean useInclude	;

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		return EVAL_BODY_BUFFERED ;	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 3.1.1.0 (2003/03/28) ボディの内容を取得する処理を、CommonTagSupport で行う。
	 * @og.rev 4.0.0.0 (2007/10/12) 処理中にエラーを発生させないようにしする。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		if( !useInclude && value == null ) {
			// 4.0.0.0 (2007/10/12) 処理中にエラーを発生させない
			try {
				value = getBodyString();
			}
			catch( HybsSystemException ex ) {	// 主に、UserInfo が見つからない場合
				value = getBodyContent().getString() ;
			}

			if( value != null && value.indexOf( "<og:" ) >= 0 ) {
				final String errMsg = "このタグの BODY部に opengion タグが記述されています。"
							+ "BODY部ではopengion タグは使用できません。";
				throw new HybsSystemException( errMsg );
			}
		}

		return SKIP_BODY ;
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		jspPrint( value );

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 2.0.0.4 (2002/09/27) カスタムタグの release() メソッドを、追加
	 * @og.rev 3.1.1.2 (2003/04/04) Tomcat4.1 対応。release2() を doEndTag()で呼ぶ。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		value		= null;
		useInclude	= false;
	}

	/**
	 * 【TAG】value 値に設定します。
	 *
	 * @og.tag
	 * ここで、value に設定した場合は、BODY 部は無視されます。
	 * なお、このタグでは、エラー発生時でも継続して処理を続けられるようにします。
	 * error.jsp などのエラー処理画面で、このタグを使用するケースがある為です。
	 *
	 *  &lt;og:text value="あいうえお" /&gt;
	 *
	 * @og.rev 2.2.0.0 (2002/12/17) 中国語(国際化)対応 エンコードの取得方法変更
	 * @og.rev 3.0.0.0 (2002/12/25) StringUtil#changeString 廃止
	 * @og.rev 4.0.0.0 (2005/12/31) エラー発生時でも異常終了させずに処理を続けます。
	 *
	 * @param   val 設定値
	 */
	public void setValue( final String val ) {
		if( !useInclude ) {
			try {
				value = getRequestParameter( val );
			}
			catch( HybsSystemException ex ) {
				value = val ;
				LogWriter.log( "val=" + val + " [" + ex.getMessage() + "]" );
			}
		}
	}

	/**
	 * 【TAG】動的にファイルを include します。
	 *
	 * @og.tag
	 * 指定のファイル名は、自身のディレクトリからの相対パスで表されます。
	 *
	 * @og.rev 4.0.0.0 (2007/05/25) 新規追加
	 *
	 * @param   file ファイル名
	 */
	public void setInclude( final String file ) {
		useInclude = true;

		final String relativePath = getRequestParameter( file );
		final String resourcePath = getContextRelativePath(getRequest(), relativePath);
		final String realPath = HybsSystem.url2dir( resourcePath.substring(1) );

		final FileString fs = new FileString();
		fs.setFilename( realPath );
		value = fs.getValue();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "value"		,value		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}

	/**
	 * 動的にファイルを include する為の、コンテキストパスを求めます。
	 *
	 * 指定のファイル名は、自身のディレクトリからの相対パスで表されます。
	 *
	 * @og.rev 4.0.0.0 (2007/05/25) 新規追加
	 * @og.rev 4.0.0.0 (2007/11/30) if の評価方法を変更します。
	 *
	 * @param	request			ServletRequestオブジェクト
	 * @param	relativePath	ファイル名
	 *
	 * @return	コンテキストパス
	 */
	private String getContextRelativePath( final ServletRequest request,final String relativePath ) {
		// 6.1.0.0 (2014/12/26) refactoring
//		if( relativePath != null && relativePath.length() > 0 && relativePath.charAt(0) == '/'
		if( StringUtil.startsChar( relativePath , '/' )				// 6.2.0.0 (2015/02/27) １文字 String.startsWith
			|| !(request instanceof HttpServletRequest) ) {
				return relativePath ;
		}

//		if(relativePath.startsWith("/")) {
//			return relativePath ;
//		}
//		if(!(request instanceof HttpServletRequest)) {
//			return relativePath ;
//		}
		final HttpServletRequest hrequest = (HttpServletRequest) request;
		String uri = (String)request.getAttribute("javax.servlet.include.servlet_path");
		if( uri != null && uri.lastIndexOf('/') >= 0 ) {
			final String pathInfo = (String)request.getAttribute("javax.servlet.include.path_info");
			if(pathInfo == null) {
				uri = uri.substring(0, uri.lastIndexOf('/'));
			}
		}
		else {
			uri = hrequest.getServletPath();
			if(uri.lastIndexOf('/') >= 0) {
				uri = uri.substring(0, uri.lastIndexOf('/'));
			}
		}
		return uri + '/' + relativePath;
	}
}
