/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.SelectionCellEditor;					// 6.2.2.0 (2015/03/27)
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;						// 5.7.3.0 (2014/02/07)
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.XHTMLTag;

/**
 * BITBOX エディターは、カラムのデータをコードリソースに対応した
 * チェックボックスで編集する場合に使用するクラスです。
 * このチェックボックスは、複数選択されたコードリソースの値を、
 * BIT演算で、論理和した結果のデータで管理します。
 *
 * 一覧登録する場合は、チェックボックスのみ表示されます。
 * ラベル表示が必要な場合は、編集パラメータに、"useLabel"と
 * 記述しておくことで、ラベルを出力することが可能です。
 *
 * 例)
 *   コードリソースには、1,2,4,8,… とBitを表す数を指定します。
 *   これを、チェックボックスで複数選択し、値を論理和で加算します。
 *
 * このエディタはeventColumnに対応していません。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 6.2.2.4 (2015/04/24) 新規作成
 * @og.group データ編集
 *
 * @version  6.2
 * @author	 Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public class Editor_BITBOX extends AbstractEditor implements SelectionCellEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.2.4 (2015/04/24)" ;

	private final Selection	selection ;
	private final boolean writable	;
	private final boolean useLabel	;		// 4.3.3.0 (2008/10/01)

	private String errMsg	;				// 6.0.4.0 (2014/11/28)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 */
	public Editor_BITBOX() {
		selection = null;
		writable  = false;
		useLabel  = false;		// 4.3.3.0 (2008/10/01)
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_BITBOX( final DBColumn clm ) {
	//	super( clm );
		name  = clm.getName();

		final String addKeyLabel = clm.getAddKeyLabel();		// 6.2.0.0 (2015/02/27) キー:ラベル形式

		// 5.7.3.0 (2014/02/07) SelectionFactory 対応
		selection = SelectionFactory.newSelection( "BITBOX" , clm.getCodeData(), addKeyLabel );

		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			errMsg = "codeData が未設定です。"
								+ " name="  + name
								+ " label=" + clm.getLabel()
								+ " editType="  + clm.getEditor() ;
			System.out.println( errMsg );
		}

		writable = clm.isWritable();

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		attributes = new Attributes()
					.set( clm.getEditorAttributes() )				// #addAttributes( Attributes ) の代替え
					.add( "class","BITBOX" );

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) )
				 .add( attributes.get( "optionAttributes" ) );		// 6.0.4.0 (2014/11/28)

		// 4.3.3.0 (2008/10/01)
		useLabel = "useLabel".equalsIgnoreCase( clm.getEditorParam() ) ? true : false;
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_BITBOX( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			return "<span class=\"error\">" + errMsg + " value=" + value + "</span>";
		}

		final String bitbox ;
		if( writable ) {
			bitbox = selection.getOption( name,value,true );
		}
		else {
			bitbox = selection.getValueLabel( value,true );
		}

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "pre" )
					.add( tagBuffer.makeTag() )
					.addBody( bitbox )
					.makeTag();
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 6.2.2.4 (2015/04/24) 新規追加
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// 6.0.4.0 (2014/11/28) selection が null の場合、警告表示します。
		if( selection == null ) {
			return "<span class=\"error\">" + errMsg + " value=" + value + " row=" + row + "</span>";
		}

		final String bitbox ;
		if( writable ) {
			// 4.3.3.0 (2008/10/01)
			bitbox = selection.getOption( name + HybsSystem.JOINT_STRING + row,value,useLabel );
		}
		else {
			bitbox = selection.getValueLabel( value,useLabel );
		}

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "pre" )
					.add( tagBuffer.makeTag() )
					.addBody( bitbox )
					.makeTag( row,value );
	}
}
