/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * NUMBER レンデラーは、カラムのデータを数字表示する場合に使用するクラスです。
 * x,yの形式で表示パラメータを指定可能です。
 * 負数の場合はspanタグclass="minus"を付けて出力します。
 *
 * フォーマットには、java.text.NumberFormat を使用せずに、独自クラスを使用しており
 * double 以上の精度をもつ値でも正確に変換できます。
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.group データ表示
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_NUMBER extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.0.4.0 (2014/11/28)" ;

	private final String	defValue ;
	private final int		minFraction;
	private final String	noDisplayVal ;		// 5.6.2.3 (2013/03/22)

	// 5.2.2.0 (2010/11/01) defval,size の初期値設定の変更
	private static final CellRenderer[] dbCell = {
							new Renderer_NUMBER(),				// 0
							new Renderer_NUMBER("",1),			// 1
							new Renderer_NUMBER("",2),			// 2
							new Renderer_NUMBER("0",0),			// 3
							new Renderer_NUMBER("0",1),			// 4
							new Renderer_NUMBER("0",2)			// 5
	};

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.3.0.0 (2003/06/23) 初期値設定追加。
	 * @og.rev 4.0.0.0 (2007/11/29) 初期値を""に変更。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数初期化
	 *
	 */
	public Renderer_NUMBER() {
//		defValue     = "";			// 4.0.0.0 (2007/11/29)
//		minFraction  = 0;
//		noDisplayVal = null;		// 5.5.1.0 (2012/04/03)
		this( "",0,null );			// 6.0.2.4 (2014/10/17)
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.3.0.0 (2003/06/23) 初期値設定追加。NumberFormatクラスは、廃止します。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
	 *
	 * @param	defval	初期値
	 * @param	size  	小数点
	 */
	private Renderer_NUMBER( final String defval,final int size ) {
//		defValue     = defval;
//		minFraction  = size;
//		noDisplayVal = null;		// 5.5.1.0 (2012/04/03)
		this( defval,size,null );	// 6.0.2.4 (2014/10/17)
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 6.0.2.4 (2014/10/17) noDisplayVal 対応漏れのため、追加
	 *
	 * @param	defval	初期値
	 * @param	size  	小数点
	 * @param	noDispVal  	非表示文字の設定
	 */
	private Renderer_NUMBER( final String defval,final int size , final String noDispVal ) {
		defValue     = defval;
		minFraction  = size;
		noDisplayVal = noDispVal;		// 5.5.1.0 (2012/04/03)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 * @og.rev 5.2.2.0 (2010/11/01) defval,size の初期値設定の変更
	 * @og.rev 5.3.5.0 (2011/05/01) ↑の判定ロジックのバグ修正
	 * @og.rev 6.0.4.0 (2014/11/28) 表示は、ViewLength属性を元に行う。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 * @og.rtnNotNull
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		final String defval    = clm.getDefault();
		int   size             = clm.getSizeY();
		final String noDispVal = clm.getNoDisplayVal();	// 6.0.2.4 (2014/10/17)

		// 6.0.4.0 (2014/11/28) 表示は、ViewLength属性があれば、それを使う。
		final String viewLength = clm.getViewLength();
		if( viewLength != null ) {
			final int ch = viewLength.indexOf( ',' ) ;		// DBColumn で、"." を "," に変換済み
			if( ch > 0 ) {
				size = Integer.parseInt( viewLength.substring( ch+1 ) );
			}
			else {
				size = 0;
			}
		}

		// 5.2.2.0 (2010/11/01) defval,size の初期値設定の変更
		// 5.3.5.0 (2011/05/01) ↑の判定ロジックのバグ修正
//		if( size < 3 ) {
		if( size < 3 && noDispVal == null ) {		// 6.0.2.4 (2014/10/17)
			if( defval == null || defval.isEmpty() ) {
				return dbCell[size];				// 0,1,2
			}
			else if( "0".equals( defval ) ) {		// 5.3.0.0 (2010/12/01) リテラルで比較する。
				return dbCell[size+3];				// 3,4,5
			}
		}

//		return new Renderer_NUMBER( defval,size );
		return new Renderer_NUMBER( defval,size,noDispVal );	// 6.0.2.4 (2014/10/17)
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) 内部に、DBColumn オブジェクトをキープしないように変更
	 * @og.rev 3.3.0.0 (2003/06/23) NumberFormatクラスは、廃止します。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		return getValue( value , true );

//		// 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
//		if( noDisplayVal != null && noDisplayVal.equalsIgnoreCase( value ) ) { return "" ; }
//
//		String rtn ;
//		if( value == null || (value.trim()).length() == 0 ) {
//			rtn = defValue;
//		}
//		else {
//			rtn = value;
//		}
//
//		rtn = StringUtil.numberFormat( rtn,minFraction );
//		if( rtn != null && (rtn.trim()).length() > 0 && rtn.charAt(0) == '-' ) {
//			rtn = "<span class=\"minus\">" + rtn + "</span>";
//		}
//		return rtn;
	}

	/**
	 * データ出力用の文字列を作成します。
	 * ファイル等に出力する形式を想定しますので、HTMLタグを含まない
	 * データを返します。
	 * 基本は、#getValue( String ) をそのまま返します。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) データ出力用のレンデラー
	 *
	 * @param   value 入力値
	 *
	 * @return  データ出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	@Override
	public String getWriteValue( final String value ) {
		return getValue( value , false );
	}

	/**
	 * データ表示用/出力用の文字列を作成します。
	 * 第二引数の isView == true で、データ表示用文字列を、false で
	 * データ出力用の文字列を作成します。
	 * 処理の共通化を行うためのメソッドです。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) ロジックの共通化
	 *
	 * @param   value  入力値
	 * @param   isView データ表示用かどうか(true:表示用/false:出力用)
	 *
	 * @return  データ表示用/出力用の文字列
	 * @og.rtnNotNull
	 * @see		#getValue( String )
	 */
	private String getValue( final String value , final boolean isView ) {
		// 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
		if( noDisplayVal != null && noDisplayVal.equalsIgnoreCase( value ) ) { return "" ; }

		String rtn ;
//		if( value == null || (value.trim()).length() == 0 ) {
		if( value == null || value.trim().isEmpty()  ) {			// 6.1.0.0 (2014/12/26) refactoring
			rtn = defValue;
		}
		else {
			rtn = value;
		}

		rtn = StringUtil.numberFormat( rtn,minFraction );
//		if( isView && rtn != null && (rtn.trim()).length() > 0 && rtn.charAt(0) == '-' ) {
//		if( isView && rtn != null && rtn.length() > 0 && rtn.charAt(0) == '-' ) {		// 6.2.0.0 (2015/02/27) trim()不要
		if( isView && StringUtil.startsChar( rtn , '_' ) ) {							// 6.2.0.0 (2015/02/27) １文字 String.startsWith
			rtn = "<span class=\"minus\">" + rtn + "</span>";
		}
		return rtn;
	}
}
