/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.TagBuffer;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.SelectionCellEditor;					// 6.2.2.0 (2015/03/27)
import org.opengion.hayabusa.db.Selection;
import org.opengion.hayabusa.db.SelectionFactory;
import org.opengion.fukurou.util.StringFormat;

/**
 * DBRADIO エディターは、カラムの編集パラメーターのSQL文の実行結果より、動的にラジオボタンを
 * 作成して編集する場合に使用するエディタークラスです。
 *
 * 編集パラメータには、ラジオボタンを作成するための、SQL文を記述します。
 * このSQL文は、select KEY,LABEL from xx ･･･ という構文で、KEY部分とLABEL部分が
 * 選択されます。各カラムの意味は次のようになります。
 *  第１カラム(必須) : ラジオボタンのキー(値)
 *  第２カラム       : ラベル(指定されない場合は、ラベルリソースの短縮ラベルを使用します)
 *  第３カラム       : クラス そのオプションに色づけなどを行う為の指定します。
 *                     NULL(または、ゼロ文字列)の場合は、適用されません。
 *  第４カラム       : この値は'false'又は'0'である場合にそのラジオボタンを選択不可にします。
 *                     NULL(または、ゼロ文字列)の場合は、選択可能になります。
 *
 * 各カラムの値(value値)に、AAA:BBB:CCC:DDD という値を設定できます。これは、
 * $1,$2,$3,$4 に割り当てなおして、QUERYを実行します。また、$1 は、本来の値として、
 * メニューの初期値設定等に使用します。上記の例では、AAA が値で、それ以降は、
 * 引数になります。
 * 又、$Cには自分自身のカラム名を割り当てます。
 * この機能を使用すれば、動的メニューを行ごとに条件を変えて作成することが
 * 可能になります。
 * 例：select KEY,LABEL from xx where KUBUN='$2' and CDK='$3'
 * さらに、元の文字列"AAA:BBB:CCC:DDD"は、$0 に割り当てられます。割り当てがない
 * 変数は、""(ゼロ文字列)として、扱われます。
 *
 * このエディタはeventColumnに対応していません。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.rev 4.3.3.6 (2008/11/15) 新規作成
 * @og.rev 6.2.2.0 (2015/03/27) SelectionCellEditor I/Fを追加
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Hiroki Nakamura
 * @since    JDK5.0,
 */
// public class Editor_DBRADIO extends AbstractEditor {
public class Editor_DBRADIO extends AbstractEditor implements SelectionCellEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.2.0 (2015/03/27)" ;

	private final String query ;
	private final String dbid ;
	private final String lang;
	private final boolean writable ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 */
	public Editor_DBRADIO() {
		// 4.3.4.4 (2009/01/01)
		query	= null;
		dbid	= null;
		lang	= null;
		writable  = false;
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 6.0.4.0 (2014/11/28) optionAttributes は、コンストラクタで設定します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_DBRADIO( final DBColumn clm ) {
		name  = clm.getName();
		dbid = clm.getDbid();
		lang = clm.getLang();

		query = clm.getEditorParam();
//		if( query == null || query.length() == 0 ) {
		if( query == null || query.isEmpty() ) {
			final String errMsg = "DBRADIO Editor では、編集パラメータは必須です。"
					+ " name=[" + name + "]" + CR ;
			throw new HybsSystemException( errMsg );
		}

		writable = clm.isWritable();

		// 6.1.1.0 (2015/01/17) Attributesの連結記述
		attributes = new Attributes()
					.set( clm.getEditorAttributes() )				// #addAttributes( Attributes ) の代替え
					.add( "class","RADIO" );

//		attributes = new Attributes();
//		attributes.addAttributes( clm.getEditorAttributes() );
//		attributes.add( "class","RADIO" );

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		tagBuffer.add( XHTMLTag.inputAttri( attributes ) )
				 .add( attributes.get( "optionAttributes" ) );		// 6.0.4.0 (2014/11/28)

//		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );
//		final String optAttr = attributes.get( "optionAttributes" );
//		tagBuffer.add( optAttr );								// 6.0.4.0 (2014/11/28)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_DBRADIO( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) $Cのカラム名置換えを追加
	 *
	 * @param	value 入力値
	 *
	 * @return	データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// StringFormat format = new StringFormat( query,value);
		final StringFormat format = new StringFormat( query, value, name ); // 4.3.4.0 (2008/12/01)
		final String newQuery = format.format();
		final String newValue = format.getValue();
		final Selection selection = SelectionFactory.newDBRadioSelection( newQuery,dbid,lang );

		final String radio ;
		if( writable ) {
			radio = selection.getRadio( name,newValue,true );
		}
		else {
			radio = selection.getValueLabel( newValue );
		}

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "pre" )
					.add( tagBuffer.makeTag() )
					.addBody( radio )
					.makeTag();

//		final TagBuffer tag = new TagBuffer( "pre" );
//		tag.add( tagBuffer.makeTag() );
////		tag.add( optAttr );
//		tag.setBody( radio );
//
//		return tag.makeTag();
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 4.3.4.0 (2008/12/01) $Cのカラム名置換えを追加
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// StringFormat format = new StringFormat( query,value);
		final StringFormat format = new StringFormat( query, value, name ); // 4.3.4.0 (2008/12/01)
		final String newQuery = format.format();
		final String newValue = format.getValue();
		final Selection selection = SelectionFactory.newDBRadioSelection( newQuery,dbid,lang );

		final String radio ;
		if( writable ) {
			radio = selection.getRadio( name + HybsSystem.JOINT_STRING + row,newValue,true );
		}
		else {
			radio = selection.getRadioLabel( newValue );
		}

		// 6.1.1.0 (2015/01/17) TagBufferの連結記述
		return new TagBuffer( "pre" )
					.add( tagBuffer.makeTag() )
					.addBody( radio )
					.makeTag( row,value );

//		final TagBuffer tag = new TagBuffer( "pre" );
//		tag.add( tagBuffer.makeTag() );
//		tag.setBody( radio );
////		tag.add( optAttr );
//
//		return tag.makeTag( row,value );
	}
}
