/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import java.util.Calendar;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractDBType;
import org.opengion.hayabusa.db.DBTypeCheckUtil;
import org.opengion.fukurou.util.HybsDateUtil;

/**
 * 文字列の時間属性(時:分:秒)の半角の時間を扱う為の、カラム属性を定義します。
 *
 * HHmmss に対応している必要があります。
 * (HHmmのデータでも利用可能です)
 *
 * ただし、日付の整合性チェックは行いませんが、valueAdd( String value )での
 * 日付の加算時には、正式な日付データにて加算します。
 *
 * タイプチェックとして、以下の条件を判定します。
 * ・文字列長は、直接計算で文字数との比較
 * ・日付使用文字チェック「('0' &gt; c || '9' &lt; c)以外」エラー
 * ・文字パラメータの 正規表現チェック
 *
 * @og.group データ属性
 * @og.rev 5.4.3.6 (2012/01/20)  タイプチェックが抜けているので追加
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class DBType_HMS extends AbstractDBType {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.2.2 (2015/04/03)" ;

	/**
	 * String引数の文字列を＋１した文字列を返します。
	 * これは、英字の場合(A,B,C など)は、B,C,D のように,最終桁の文字コードを
	 * ＋１ します。
	 * 文字列が数字タイプの場合は, 数字に変換して、+1 します。(桁上がりもあり)
	 * 混在タイプの場合は,最後の桁だけを確認して ＋１します。
	 * 引数が null の場合と、ゼロ文字列("")の場合は,物理的初期設定値(String getDefault())
	 * の値を返します。
	 * 24時間を超えた場合は、25時となります。
	 *
	 * @og.rev 6.2.2.2 (2015/04/03) 桁数可変に対応
	 *
	 * @param	value	String引数の文字列
	 *
	 * @return  String引数の文字列を＋１した文字列(１時間加算)
	 */
	@Override
	public String valueAdd( final String value ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }

//		int hour = Integer.parseInt( value.substring( 0,2 ) );
//
//		// 先頭に 00 を付加する為に、数字の 100 を加算している。
//		if( hour < 24 ) { hour = 101 + hour; }
//		else			{ hour = 100; }
//
//		final String rtn = hour + value.substring( 2 );
//
//		return rtn.substring( 1 );

		// 6.2.2.2 (2015/04/03) ４桁より小さいとHHmm、それ以外は、HHmmss
		final boolean isHM = value.length() <= 4 ;

		//                        HHmm    HHmmss
		final int hms = ( isHM ? 10100 : 1010000 ) + Integer.parseInt( value );

		// 元の形式に準拠して返す。
		return String.valueOf( hms ).substring( 1 );
	}

	/**
	 * String引数の文字列に、第２引数に指定の文字列(数字、日付等)を加算して返します。
	 *
	 * ここでは、HHmmss 形式のデータに、時間を加算します。
	 *
	 * この HMS は、引数に、日付単位を指定できます。単位は、HHmmss 形式の
	 * １文字を指定します。大文字、小文字も識別します。value="5H" とすれば、５時間、
	 * value="5m"とすれば、５分 追加します。
	 * 指定しない場合は、時を加算します。
	 *
	 * ここのデータは、時間が繰り越しても、日付に影響しません。
	 * また、２４時間を超えた場合は、00 時に戻ります。
	 *
	 * @og.rev 5.6.0.3 (2012/01/24) ADD に、引数の値を加算する機能を追加します。
	 * @og.rev 5.6.1.0 (2013/02/01) 加算する引数に、日付単位('H','m','s')を指定可能にします。
	 * @og.rev 6.2.2.2 (2015/04/03) 桁数可変に対応
	 *
	 * @param   value  String引数
	 * @param   add    加算する時間文字列(単位付き:['H','m','s'])
	 *
	 * @return  引数の文字列に時間を加算します。
	 */
	@Override
	public String valueAdd( final String value,final String add ) {
		if( value == null || value.isEmpty() ) { return getDefault(); }
		// 日付文字列にダミーの年月日を追加しておく。
//		return HybsDateUtil.getDatePlus( ("20100101" + value),add,Calendar.HOUR_OF_DAY,"HHmmss" );

		final boolean isHM = value.length() <= 4 ;

		// ※ yyyymmd + 1HHmmss で、日付データのダミーの年月日を追加
		final int hms = 1000000 + Integer.parseInt( isHM ? value + "00" : value );

		// 元の形式に準拠して返す。
		return HybsDateUtil.getDatePlus( ( "2010010" + hms ),add,Calendar.HOUR_OF_DAY, isHM ? "HHmm" : "HHmmss" );
	}

	/**
	 * エディターで編集されたデータを登録する場合に、データそのものを
	 * 変換して、実登録データを作成します。
	 * 例えば,大文字のみのフィールドなら、大文字化します。
	 * 実登録データの作成は、DBType オブジェクトを利用しますので,
	 * これと Editor とがアンマッチの場合は、うまくデータ変換
	 * されない可能性がありますので、注意願います。
	 *
	 * @og.rev 6.2.2.2 (2015/04/03) 桁数可変に対応
	 *
	 * @param	value	(一般に編集データとして登録されたデータ)
	 *
	 * @return  修正後の文字列(一般にデータベースに登録するデータ)
	 */
	@Override
	public String valueSet( final String value ) {
		if( value == null || value.isEmpty() ) { return ""; }
//		return StringUtil.lTrim0( StringUtil.deleteChar( value,':' ) );

		// 6.2.2.2 (2015/04/03) ４桁より小さいとHHmm、それ以外は、HHmmss
		final String val = StringUtil.deleteChar( value,':' );
		final boolean isHM = value.length() <= 4 ;

		//                        HHmm    HHmmss
		final int hms = ( isHM ? 10000 : 1000000 ) + Integer.parseInt( value );

		// 元の形式に準拠して返す。
		return String.valueOf( hms ).substring( 1 );
	}

	/**
	 * データが登録可能かどうかをチェックします。
	 * データがエラーの場合は、そのエラー内容を返します。
	 *
	 * @og.rev 5.2.3.6 (2012/01/20) 数値のみに限定するために追加
	 * @og.rev 5.6.0.3 (2012/01/24) ADD に、引数の値を加算する機能を追加します。
	 * @og.rev 6.2.2.2 (2015/04/03) 桁数可変に対応
	 *
	 * @param   key		キー
	 * @param   value	値
	 * @param   sizeX	整数部分の文字列の長さ
	 * @param   sizeY	小数部分の文字列の長さ
	 * @param   typeParam	dbType パラメータ
	 * @param   isStrict	厳密にチェックするかどうか[true:する/false:標準的]
	 *
	 * @return  エラー内容
	 * @og.rtnNotNull
	 */
	@Override
	public ErrorMessage valueCheck( final String key ,final String value ,
									final int sizeX ,final int sizeY ,final String typeParam ,final boolean isStrict) {
		final String checkVal = valueSet(value); // ：は念のため外しておく

		final ErrorMessage msg = new ErrorMessage();
		if( checkVal == null || checkVal.isEmpty() ) { return msg; }

		final int len = (sizeY == 0) ? sizeX : sizeX + sizeY + 1;
		if( isStrict ) {
			if( len != checkVal.length() ) {
				// 文字列の長さが指定の長さと異なります。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0011", key,value, String.valueOf( value.length() ), String.valueOf( len ) );
			}
		}
		else {
			if( len < checkVal.length() ) {
				// 文字列の長さが指定の長さよりも長いです。
				msg.addMessage( 0,ErrorMessage.NG,"ERR0006",key,value,
										String.valueOf( value.length() ),String.valueOf( len ) );
			}
		}

		// 5.6.0.3 (2012/01/24) 文字の範囲チェック
		String check = DBTypeCheckUtil.rangeCheck( checkVal, '0', '9' );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// 5.6.0.3 (2012/01/24) 時分秒の整合性チェック
		check = DBTypeCheckUtil.hmsFormatCheck( checkVal );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		// 3.6.0.0 (2004/09/22) dbType パラメータを使用したマッチチェック
		check = DBTypeCheckUtil.matcheCheck( checkVal,typeParam );
		if( check != null ) {
			// 指定の文字以外の文字が使われています。
			msg.addMessage( 0,ErrorMessage.NG,"ERR0009", key,check );
		}

		return msg;
	}
}
