/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.view;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.html.TableFormatter;
import org.opengion.hayabusa.html.ViewStackTableParam;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * 積上ガント表示専用のViewFormです。
 * stackParamTagを利用する事でスタックガント用の行を出力する事が可能です。
 * stackParamTagによりstackColumnsが指定された場合は、そのカラム毎にブレークして、
 * stacklink属性により積上げ行の判別が可能なtbody行を出力します。
 * その際、stackColumnsで指定されたカラム以外の[xxx]は処理されません（空白として出力）
 * [xxx]以外で書かれた箇所、例えば&lt;iGantBar&gt;タグの本体部分等は出力されます。
 * 
 * ヘッダの表示にはstackHeaderタグを利用します。
 * 
 * [エンジン内部積上げを行わない場合]
 * 積上の表示はJavaScriptによってiGantBarタグの箇所に作成されます。
 * 積上げそのものもiGantBarによって出力されるガントを利用してJavaScriptで行っているため、
 * 最大検索行数と表示行数に注意して下さい。
 * 
 * [エンジン内部積上げを行う場合]
 * 工数積上げをエンジン内部で行いdivタグとして出力します。
 * その後の描画（位置調整や色等）はJavaScriptで行います。
 * ガント部分は出力されません。
 * スタック部分はbody部分の最後尾に新たにtd作成するため、注意してください。
 * paramタグでの指定で、costColumnが必須です。
 * 
 * 
 * AbstractViewForm により、setter/getterメソッドのデフォルト実装を提供しています。
 * 各HTMLのタグに必要な setter/getterメソッドのみ，追加定義しています。
 *
 * AbstractViewForm を継承している為,ロケールに応じたラベルを出力させる事が出来ます。
 * 
 * @og.rev 5.5.7.0 (2012/10/01) 新規作成
 * @og.rev 5.5.8.3 (2012/11/17) 内部積上げ対応
 * @og.rev 5.6.1.2 (2013/02/22) キャパシティ対応
 * @og.group 画面表示
 *
 * @version  5.0
 * @author	 Takahashi Masakazu
 * @since    JDK5.0,
 */
public class ViewForm_HTMLStackedGanttTable extends ViewForm_HTMLTable	{
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.0.0 (2015/02/27)" ;

	/** ボディーフォーマット最大数 初期値:{@value} */
	protected static final int BODYFORMAT_MAX_COUNT = 10;
	// stack行の判定出力用
	protected static final String STACK_TBODY		= " stackline='true'";
	protected static final String GANTT_TBODY		= " stackline='false'";
	protected static final String STACK_ID_PREFIX	= " id='stack_";
	protected static final String STACK_ROW_PREFIX	= " stackrow='";

	/** ヘッダーフォーマット変数 */
	protected TableFormatter 		headerFormat	;
	/** ボディーフォーマット配列変数 */
	protected TableFormatter[]		bodyFormats		;
	/** フッターフォーマット変数 */
	protected TableFormatter		footerFormat	;
	/** ボディーフォーマット数 */
	protected int					bodyFormatsCount;

	// stack,gantt用
	private int[]  stackCols		;

	// 5.5.8.3 (2012/11/17)
	private int[]	costCols		;		// 工数カラム、開始日カラム、終了日カラム
	private boolean	innerStack		= Boolean.parseBoolean( ViewStackTableParam.INNER_STACK_VALUE );
	private boolean	stackHoliday	= Boolean.parseBoolean( ViewStackTableParam.STACK_HOLIDAY_KEY );

	String[][] calArray				;		// headで作成されたカレンダーデータ
	int capCol = -1					;		// 5.6.1.2 (2013/02/22) 能力値カラム

	/**
	 * DBTableModel から HTML文字列を作成して返します。
	 * startNo(表示開始位置)から、pageSize(表示件数)までのView文字列を作成します。
	 * 表示残りデータが pageSize 以下の場合は,残りのデータをすべて出力します。
	 *
	 *
	 * @og.rev 5.5.8.3 (2012/11/17) 内部積上げ対応
	 * @og.rev 5.6.1.2 (2013/02/22) キャパシティ対応
	 * @og.rev 5.6.2.1 (2013/06/13) 積上不具合修正
	 * @og.rev 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
	 *
	 * @param  sttNo	  表示開始位置
	 * @param  pgSize   表示件数
	 *
	 * @return	DBTableModelから作成された HTML文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String create( final int sttNo, final int pgSize )  {
		// ガントは、キーブレイクがあるため、全件表示します。
		final int startNo  = 0;
		final int pageSize = getRowCount() ;
		if( pageSize == 0 ) { return ""; }	// 暫定処置

		// 4.3.1.0 (2008/09/08)
		if( headerFormat == null ) {
			final String errMsg = "ViewTagで canUseFormat() = true の場合、Formatter は必須です。";
			throw new HybsSystemException( errMsg );
		}

		headerLine	 = null;		// 3.5.3.1 (2003/10/31) キャッシュクリア

		final int lastNo = getLastNo( startNo, pageSize );
		final int blc = getBackLinkCount();
//		int hsc = getHeaderSkipCount();			// 6.0.2.5 (2014/10/31) たぶん、カッコのコメントする位置間違い
//		int hscCnt = 1;							// 6.0.2.5 (2014/10/31) たぶん、カッコのコメントする位置間違い

		// このビューの特有な属性を初期化
		paramInit();

		headerFormat.makeFormat( getDBTableModel() );	// 3.5.6.2 (2004/07/05) 移動
		// 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
		setFormatNoDisplay( headerFormat );

		final StringBuilder out = new StringBuilder( BUFFER_LARGE )
			.append( getCountForm( startNo,pageSize ) )
			.append( getHeader() );

		if( bodyFormatsCount == 0 ) {
			bodyFormats[0] = headerFormat ;
			bodyFormatsCount ++ ;
		}
		else {
			for( int i=0; i<bodyFormatsCount; i++ ) {
				bodyFormats[i].makeFormat( getDBTableModel() );
				// 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
				setFormatNoDisplay( bodyFormats[i] );
			}
		}

		String[] astrOldStackKeys = new String[stackCols.length];
		for( int nIndex =0; nIndex < astrOldStackKeys.length; nIndex++) {
			astrOldStackKeys[nIndex] = "";
		}

		int bgClrCnt = 0;
		int stackRow = 0;

		// 5.5.8.3 (2012/11/17)
		double[] costAry = null;
		Calendar firstCalday = null;
		Calendar fstCalEnd = null;
		String calZoom = null;
		String capacity = null; // 5.6.1.2 (2013/02/22)
		if( innerStack ){
			costAry = new double[calArray.length];
			final String[] firstCal = calArray[0];
				firstCalday = HybsSystem.getCalendar(firstCal[0]);
				fstCalEnd   = HybsSystem.getCalendar(firstCal[2]);

			if( differenceDays(firstCalday.getTime(),fstCalEnd.getTime()) == 1 ){
				calZoom = ViewStackTableParam.STACK_ZOOM_DAY;
			}
			else if( differenceDays(firstCalday.getTime(),fstCalEnd.getTime()) == 7 ){
				calZoom = ViewStackTableParam.STACK_ZOOM_WEEK;
			}
			else{
				calZoom = ViewStackTableParam.STACK_ZOOM_MONTH;
			}
		}

		for( int row=startNo; row<lastNo; row++ ) {
			// データのスキップは行わない

			// ガントのブレイク
		//	if(! isSameGroup(row, astrOldGroupKeys)) {
				if( !isSameStack(row, astrOldStackKeys) && stackCols.length > 0 ) { // 積上のブレイク
					if( !(innerStack && row == startNo) ) { // 5.5.8.3 (2012/11/17) 内部積上げは後から積上げるので、初回は出力しない
						stackRow = row;

//						out.append(makeBodyTable(innerStack ? row -1 : row, stackRow, bgClrCnt, blc, costAry, capacity));	// 5.6.1.2 (2013/02/22)
						makeBodyTable( out,innerStack ? row -1 : row, stackRow, bgClrCnt, blc, costAry, capacity );			// 6.1.0.0 (2014/12/26) refactoring

						if( innerStack ){
							costAry = new double[calArray.length];
						}
					}
				}
				// 6.1.0.0 (2014/12/26) refactoring : Avoid if (x != y) ..; else ..;
				if( innerStack ){	// 5.5.8.3 (2012/11/17) 内部積上げをする場合
					final double costDbl = Double.parseDouble( getValue(row,costCols[0]) ); //工数
					final Calendar startDay = HybsSystem.getCalendar(getValue(row,costCols[1]));
					final Calendar endDay   = HybsSystem.getCalendar(getValue(row,costCols[2]));

					final Date startDayDate = startDay.getTime();
					Date endDayDate = endDay.getTime();

					// 5.6.1.2 (2013/02/22)
					if( capCol > -1 ){
						capacity = getValue(row,capCol);
					}
					else{
						capacity = "1";
					}

					// 枠はそのままで計算
					final int fromCell = calNumber(startDayDate,calZoom,firstCalday.getTime());
					final int toCell   = calNumber(endDayDate,calZoom,firstCalday.getTime());	

					endDay.add(Calendar.DATE, 1); // 終了日は範囲に入るので１つ進める
					endDayDate = endDay.getTime();

					int stackMother = differenceDays(startDayDate,endDayDate);
					if( !stackHoliday ){
						for(int cel = fromCell; cel <= toCell; cel++ ){
							if ("1".equals( calArray[cel][1] ) ){
								stackMother--;
							}
						}
					}

					Date calFrom;
					Date calTo;
					int cellDays = 1;

					for(int cel = fromCell; cel <= toCell; cel++ ){
						calFrom = HybsSystem.getCalendar(calArray[cel][0]).getTime();
						calTo   = HybsSystem.getCalendar(calArray[cel][2]).getTime();
						if( calFrom.compareTo( startDayDate ) < 0 ){
							calFrom = startDayDate;
						}
						if( endDayDate.compareTo( calTo ) < 0 ){
							calTo = endDayDate;
						}
						cellDays = differenceDays( calFrom, calTo );
						if( stackHoliday ){
							costAry[cel] += (costDbl / stackMother) * cellDays;
						}
						else{
							// 休日のみの場合は積上げられない！
							if (!"1".equals( calArray[cel][1] ) ){
								costAry[cel] += (costDbl / stackMother) * cellDays;
							}
						}
					}
				}
				else{	// 5.5.8.3 (2012/11/17) 内部積上げの場合はガント部分は出力せずに積上げだけする。
					for( int i=0; i<bodyFormatsCount; i++ ) {
						final TableFormatter bodyFormat = bodyFormats[i];
						if( ! bodyFormat.isUse( row,getDBTableModel() ) ) { continue; }		// 3.5.4.0 (2003/11/25)
						out.append("<tbody").append( getBgColorCycleClass( bgClrCnt++,row ) );
						if( isNoTransition() ) {	// 4.3.3.0 (2008/10/01)
							out.append( getHiddenRowValue( row ) );
						}
						out.append( GANTT_TBODY )
							.append( STACK_ROW_PREFIX ).append( stackRow ).append("'>")
							.append( bodyFormat.getTrTag() );

						// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
						if( isNumberDisplay() ) {
							final String ckboxTD = "<td" + bodyFormat.getRowspan() + ">";
							out.append( makeCheckbox( ckboxTD,row,blc ) );
						}

						int cl = 0;
						for( ; cl < bodyFormat.getLocationSize(); cl++ ) {
							String fmt = bodyFormat.getFormat(cl);
							final int loc = bodyFormat.getLocation(cl);		// 3.5.5.0
							if( ! bodyFormat.isNoClass() && loc >= 0 ) {	// 3.5.5.7 (2004/05/10)
//								final StringBuilder newtg = new StringBuilder( BUFFER_LARGE )
//									.append("<td class=\"")
//									.append( getColumnDbType(loc) )			// 4.0.0 (2005/01/31)
//									.append("\" ");
////								final String tdclass = newtg.toString();
////								fmt = StringUtil.replace( bodyFormat.getFormat(cl) ,"<td", tdclass );
//								fmt = StringUtil.replace( bodyFormat.getFormat(cl) ,"<td", newtg.toString() );
								final String tdclass = "<td class=\"" + getColumnDbType(loc) + "\" " ;
								fmt = StringUtil.replace( bodyFormat.getFormat(cl) ,"<td", tdclass );
							}
							out.append( fmt );			// 3.5.0.0
							// 3.5.5.7 (2004/05/10) #,$ 対応
							if( loc >= 0 ) {
								switch( bodyFormat.getType(cl) ) {
									case '#' : out.append( getColumnLabel(loc) );		break;
									case '$' : out.append( getRendererValue(row,loc) );	break;
									case '!' : out.append( getValue(row,loc) );			break;
									default  : out.append( getValueLabel(row,loc) );	break;
								}
							}
							else {
								out.append( bodyFormat.getSystemFormat(row,loc) );
							}
						}
						out.append( bodyFormat.getFormat(cl) )
							.append("</tbody>").append( CR );
					}
				}
//			}
		}

		// 6.0.2.5 (2014/10/31) たぶん、カッコのコメントする位置間違いで使われてないようなので、一旦コメントする。
//		// 3.5.2.0 (2003/10/20) ヘッダー繰り返し属性( headerSkipCount )を採用
//			if( hsc > 0 && hscCnt % hsc == 0 ) {
//				out.append("<tbody class=\"row_h\"").append(" >");
//				out.append( getHeadLine() );
//				out.append("</tbody>");
//				hscCnt = 1;
//			}
//			else {
//				hscCnt ++ ;
//			}
	//	} // 5.6.5.2 (2013/06/21) 括弧の位置間違いのため修正

		// 内部積上げ時は最終行の出力を行う
		if( innerStack ){
//			out.append(makeBodyTable(lastNo-1, stackRow, bgClrCnt, blc, costAry, capacity));	// 5.6.1.2 (2013/02/22)
			makeBodyTable( out, lastNo-1, stackRow, bgClrCnt, blc, costAry, capacity );			// 6.1.0.0 (2014/12/26) refactoring
		}

		if( footerFormat != null ) {
			out.append( getTableFoot() );
		}

		out.append("</table>").append( CR )
			.append( getScrollBarEndDiv() );	// 3.8.0.3 (2005/07/15)

		return out.toString();
	}

	/**
	 * 内容をクリア(初期化)します。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17) 内部積上げのための修正
	 * @og.rev 5.6.1.2 (2013/02/22) キャパシティ対応
	 *
	 */
	@Override
	public void clear() {
		super.clear();
		headerFormat			= null;
		bodyFormats				= null;
		footerFormat			= null;
		bodyFormatsCount		= 0;
		stackCols		= null; // 5.5.8.3 (2012/11/17)
		costCols		= null;	// 5.5.8.3 (2012/11/17)
		innerStack 		= Boolean.parseBoolean( ViewStackTableParam.INNER_STACK_VALUE ); // 5.5.8.3 (2012/11/17)
		calArray		= null; // 5.5.8.3 (2012/11/17)
		stackHoliday = Boolean.parseBoolean( ViewStackTableParam.STACK_HOLIDAY_KEY ); // 5.5.8.3 (2012/11/17)
		capCol			= -1; // 5.6.1.2 (2013/02/22)
	}

	/**
	 * このビューに対する特別な初期化を行う。
	 *
	 * @og.rev 5.5.8.3 (2012/11/17)
	 * @og.rev 5.5.9.0 (2012/12/03) objectではなくArrayList化
	 * @og.rev 5.6.1.2 (2013/02/22) キャパシティ対応
	 * @og.rev 5.6.2.1 (2013/03/08) stackHolidayが抜けていたので追加
	 */
	private void paramInit() {
		final String strStackCols	= getParam( ViewStackTableParam.STACK_COLUMNS_KEY,	ViewStackTableParam.STACK_COLUMNS_VALUE	);
		final String costCol		= getParam( ViewStackTableParam.COST_COLUMNS_KEY,	ViewStackTableParam.COST_COLUMNS_VALUE	); // 5.5.8.3 (2012/11/17)
		innerStack					= getBoolParam( ViewStackTableParam.INNER_STACK_KEY	); // 5.5.8.3 (2012/11/17)
		final String capColName		= getParam( ViewStackTableParam.CAP_COLUMN_KEY,	ViewStackTableParam.CAP_COLUMN_VALUE	); // 5.6.1.2 (2013/02/22)
		stackHoliday				= getBoolParam( ViewStackTableParam.STACK_HOLIDAY_KEY	); // 5.6.2.1 (2013/03/08)

		if( innerStack ){
			// 6.1.0.0 (2014/12/26) findBugs: Bug type ITA_INEFFICIENT_TO_ARRAY (click for details)
			// 長さが0の配列の引数で Collection.toArray() を使用しています。
//			calArray = getViewArrayList().toArray(new String[][]{});	// 5.5.9.0 (2012/12/03)
			final List<String[]> lst = getViewArrayList();
			calArray = lst.toArray( new String[lst.size()][3] );
			if( calArray == null || costCol == null){
				final String errMsg = "ヘッダのカレンダデータ、costColumnsの設定は必須です。"+costCol;
				throw new HybsSystemException( errMsg );
			}
		}

		final DBTableModel table = getDBTableModel();

		final String[] stackKeys = StringUtil.csv2Array(strStackCols);
		stackCols = new int[stackKeys.length];
		for( int nIndex = 0; nIndex < stackCols.length ; nIndex++) {
			stackCols[nIndex] = table.getColumnNo( stackKeys[nIndex] );
		}

		final String[] costKeys = StringUtil.csv2Array(costCol);
		costCols = new int[costKeys.length];
		for( int nIndex = 0; nIndex < costCols.length ; nIndex++) {
			costCols[nIndex] = table.getColumnNo( costKeys[nIndex] );
		}

		// 5.6.1.2 (2013/02/22) キャパシティ
		if( capColName != null && capColName.length() > 0 ){
			capCol = table.getColumnNo(capColName);
		}
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @return	テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getTableHead() {
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
			buf.append("<colgroup class=\"X\" /><colgroup class=\"BIT\" /><colgroup class=\"S9\" />")
				.append(CR);
		}

		// 3.5.2.0 (2003/10/20) ヘッダー繰り返し部をgetHeadLine()へ移動
		buf.append("<thead id=\"header\">").append( CR )	// 3.5.6.5 (2004/08/09)
			.append( getHeadLine() )
			.append("</thead>").append( CR );

		return buf.toString();
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 6.1.2.0 (2015/01/24) キャッシュを返すのを、#getHeadLine() に移動。
	 *
	 * @return	テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getHeadLine() {
//		return getHeadLine( "<th" ) ;
		if( headerLine == null ) {					// キャッシュになければ、設定する。
			headerLine = getHeadLine( "<th" + headerFormat.getRowspan() ) ;
		}

		return headerLine ;
	}

	/**
	 * ヘッダー繰り返し部を、getTableHead()メソッドから分離。
	 *
	 * @og.rev 6.1.2.0 (2015/01/24) キャッシュを返すのを、#getHeadLine() に移動。
	 *
	 * @param	thTag タグの文字列
	 *
	 * @return	テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	@Override
	protected String getHeadLine( final String thTag ) {
//		if( headerLine != null ) { return headerLine; }		// キャッシュを返す。

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );

		buf.append( headerFormat.getTrTag() ).append( CR );

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
			// 6.1.2.0 (2015/01/24) thTag に、headerFormat.getRowspan() を含ませて受け取る。
//			// 3.5.4.3 (2004/01/05) 追加分
//			if( isUseCheckControl() && "checkbox".equals( getSelectedType() ) ) {
//				buf.append( thTag ).append( headerFormat.getRowspan() ).append("></th>");
//				buf.append( thTag ).append( headerFormat.getRowspan() );
//				buf.append('>').append( getAllCheckControl() ).append( "</th>");	// 6.0.2.5 (2014/10/31) char を append する。
//				buf.append( thTag ).append( headerFormat.getRowspan() );
//				buf.append('>').append( getNumberHeader() ).append("</th>");		// 6.0.2.5 (2014/10/31) char を append する。
//			}
//			else {
//				buf.append( thTag ).append(" colspan=\"3\"");
//				buf.append( headerFormat.getRowspan() );
//				buf.append('>').append( getNumberHeader() ).append("</th>");	// 6.0.2.5 (2014/10/31) char を append する。
//			}
			if( isUseCheckControl() && "checkbox".equals( getSelectedType() ) ) {
				buf.append( thTag ).append( "></th>" )
					.append( thTag ).append( '>' ).append( getAllCheckControl() ).append( "</th>" )		// 6.0.2.5 (2014/10/31) char を append する。
					.append( thTag ).append( '>' ).append( getNumberHeader()    ).append( "</th>" );	// 6.0.2.5 (2014/10/31) char を append する。
			}
			else {
				buf.append( thTag ).append( " colspan=\"3\">" ).append( getNumberHeader() ).append( "</th>" );	// 6.0.2.5 (2014/10/31) char を append する。
			}
		}

		int cl = 0;
		for( ; cl < headerFormat.getLocationSize(); cl++ ) {
			buf.append( StringUtil.replace( headerFormat.getFormat(cl) ,"td","th" ));
			final int loc = headerFormat.getLocation(cl);
			if( loc >= 0 ) { buf.append( getSortedColumnLabel(loc) ); }
		}
		buf.append( StringUtil.replace( headerFormat.getFormat(cl) ,"td","th" ) ).append( CR );

//		headerLine = buf.toString();
//		return headerLine;
		return buf.toString();				// 6.1.2.0 (2015/01/24)
	}

	/**
	 * DBTableModel から テーブルのタグ文字列を作成して返します。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
	 *
	 * @return	テーブルのタグ文字列
	 * @og.rtnNotNull
	 */
	protected String getTableFoot() {
		footerFormat.makeFormat( getDBTableModel() );
		// 6.2.0.0 (2015/02/27) フォーマット系の noDisplay 対応
		setFormatNoDisplay( footerFormat );

		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append("<tfoot>").append( CR )
			.append( footerFormat.getTrTag() ).append( CR );

		// 3.5.5.0 (2004/03/12) No 欄そのものの作成判断追加
		if( isNumberDisplay() ) {
			buf.append(" <th colspan=\"3\"")
				.append( footerFormat.getRowspan() )
				.append("></th>");
		}

		int cl = 0;
		for( ; cl < footerFormat.getLocationSize(); cl++ ) {
			final int loc = footerFormat.getLocation(cl);
			if( loc >= 0 ) { buf.append( getSortedColumnLabel(loc) ); }
		}
		buf.append( footerFormat.getFormat(cl) ).append( CR )
			.append("</tfoot>").append( CR );

		return buf.toString();
	}

	/**
	 * フォーマットを設定します。
	 *
	 *
	 * @param	list	TableFormatterのリスト
	 */
	@Override
	public void setFormatterList( final List<TableFormatter> list ) {		// 4.3.3.6 (2008/11/15) Generics警告対応
		bodyFormats = new TableFormatter[BODYFORMAT_MAX_COUNT];

		bodyFormatsCount = 0;
		for( int i=0; i<list.size(); i++ ) {
			final TableFormatter format = list.get( i );		// 4.3.3.6 (2008/11/15) Generics警告対応

			switch( format.getFormatType() ) {
				case TYPE_HEAD : headerFormat = format; break;
				case TYPE_BODY : bodyFormats[bodyFormatsCount++] = format; break;
				case TYPE_FOOT : footerFormat = format; break;
				default : final String errMsg = "FormatterType の定義外の値が指定されました。";
				// 4.3.4.4 (2009/01/01)
						  throw new HybsSystemException( errMsg );
			}
		}

		// 3.5.5.5 (2004/04/23) headerFormat が定義されていない場合はエラー
		if( headerFormat == null ) {
			final String errMsg = "h:thead タグの、フォーマットの指定は必須です。";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * フォーマットメソッドを使用できるかどうかを問い合わせます。
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	@Override
	public boolean canUseFormat() {
		return true;
	}

	/**
	 * ビューで表示したカラムの一覧をCSV形式で返します。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規追加
	 *
	 * @return	ビューで表示したカラムの一覧
	 * @og.rtnNotNull
	 */
	@Override
	public String getViewClms() {
		final DBTableModel table = getDBTableModel();
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE );
		for( int i=0; i<headerFormat.getLocationSize(); i++ ) {
			if( buf.length() > 0 ) { buf.append( ',' ); }
			buf.append( table.getColumnName( headerFormat.getLocation( i ) ) );
		}
		return buf.toString();
	}

	/**
	 * 上下行のデータが同じ積上かどうかをチェックする。
	 *
	 * @param   nRowIndex テーブルモデルの行番号
	 * @param   astrOldValues 古いグルプーデータ配列
	 *
	 * @return  使用可能(true)/ 使用不可能 (false)
	 */
	private boolean isSameStack(final int nRowIndex, final String[] astrOldValues) {
		boolean bRet = stackCols.length > 0 ;
		if( bRet ) {
			for( int nIndex = 0; bRet && nIndex < stackCols.length ; nIndex++) {
				bRet = astrOldValues[nIndex].equals( getValue( nRowIndex, stackCols[nIndex] ) ) ;
			}

			// 不一致時に astrOldValues に 新しい値を設定しておきます。
			if(!bRet) {
				for( int nIndex = 0; nIndex < stackCols.length; nIndex++) {
					astrOldValues[nIndex] = getValue(nRowIndex, stackCols[nIndex]);
				}
			}
		}
		return bRet;
	}

	/**
	 * 対象カラムが積上げカラムかどうか。
	 *
	 * @param   loc 列番号
	 *
	 * @return  対象(true)/ 非対象 (false)
	 */
	private boolean isStackClm(final int loc) {
		boolean rtn = false;
		for( int nIndex = 0; nIndex < stackCols.length ; nIndex++) {
			if( stackCols[nIndex] == loc ) {
				rtn = true;
			}
		}
		return rtn;
	}

	/**
	 * 2つの日付の差を求めます。
	 * java.util.Date 型の日付 date1 - date2 が何日かを返します。
	 * 
	 * @og.rev 5.5.8.3 (2012/11/17) 新規
	 * 
	 * @param date1    日付 
	 * @param date2    日付 
	 * @return    2つの日付の差(日数 2-1) 同日なら０
	 */
	public static int differenceDays(final Date date1,final Date date2) {
		final long datetime1 = date1.getTime();
		final long datetime2 = date2.getTime();
		final long one_date_time = 1000 * 60 * 60 * 24L;
		return (int)((datetime2 - datetime1) / one_date_time);
	}

	/**
	 * 日付から枠番号を返す。
	 * 
	 * @og.rev 5.5.8.3 (2012/11/17) 新規
	 *
	 * @param   date 日付(YYYY/MM/DD)
	 * @param 	zoom Zoom設定値
	 * @param	calFD ヘッダ初日
	 *
	 * @return  枠番号
	 */
	private int calNumber(final Date date, final String zoom, final Date calFD ) {
		int rtn = 0;
		if( zoom.equals( ViewStackTableParam.STACK_ZOOM_MONTH ) ){
			// 月だけは別の計算が必要
			final Calendar cal1 = Calendar.getInstance();
			cal1.setTime( calFD );
			final Calendar cal2 = Calendar.getInstance();
			cal2.setTime( date );
			rtn = ( cal2.get( Calendar.YEAR )-cal1.get( Calendar.YEAR ) ) * 12
					+ ( cal2.get( Calendar.MONTH ) - cal1.get( Calendar.MONTH ) );
		}
		else{
			final int diff = differenceDays( calFD, date );
			if( zoom.equals( ViewStackTableParam.STACK_ZOOM_WEEK )){
				rtn = diff/7;
			}
			else{
				rtn = diff;
			}
		}
		return rtn;
	}

	/**
	 * テーブル本体の作成。
	 * 
	 * @og.rev 5.5.8.3 (2012/11/17) 繰り返し利用するため分離
	 * @og.reb 5.6.1.2 (2013/02/22) td終了が抜けていたので追加、キャパシティ対応
	 * @og.reb 6.1.0.0 (2014/12/26) 引数に StringBuffer を追加し、それに、データを追記していく。
	 *
	 * @param   outBuf	データを追加するStringBuffer
	 * @param   row		テーブルモデルのrow
	 * @param 	stackRow スタック行保存用
	 * @param	bgClrCnt 背景色カウンタ
	 * @param	blc		チェックボックス用
	 * @param	costAry コスト集計配列
	 * @param	cap		能力
	 *
	 * @return  テーブル本体のHTML(入力の out オブジェクトそのもの)
	 * @og.rtnNotNull
	 */
//	private StringBuffer makeBodyTable( final int row, final int stackRow, final int bgClrCnt, final int blc, final double[] costAry, final String cap){
	private StringBuilder makeBodyTable( final StringBuilder outBuf, final int row, final int stackRow,
											 final int bgClrCnt, final int blc, final double[] costAry, final String cap ) {
//		final StringBuffer outBuf = new StringBuffer( 100 );		// 6.1.0.0 (2014/12/26) 引数に append するように変更。
		int bcCnt = bgClrCnt;		// 6.0.0.1 (2014/04/25) 引数を直接変更できなくする。
		for( int i=0; i<bodyFormatsCount; i++ ) {
			final TableFormatter bodyFormat = bodyFormats[i];
			if( ! bodyFormat.isUse( row,getDBTableModel() ) ) { continue; }	
			outBuf.append("<tbody").append( getBgColorCycleClass( bcCnt++,row ) );
			if( isNoTransition() ) { 
				outBuf.append( getHiddenRowValue( row ) );
			}
			outBuf.append( STACK_TBODY )
//				.append( STACK_ROW_PREFIX ).append( stackRow ).append( "'" )
				.append( STACK_ROW_PREFIX ).append( stackRow ).append( '\'' )
				.append( STACK_ID_PREFIX  ).append( stackRow ).append( "'>" )
				.append( bodyFormat.getTrTag() );

			//  No 欄そのものの作成判断追加
			if( isNumberDisplay() ) {
				final String ckboxTD = "<td" + bodyFormat.getRowspan() + ">";
				outBuf.append( makeCheckbox( ckboxTD,row,blc ) );
			}

			int cl = 0;
			for( ; cl < bodyFormat.getLocationSize(); cl++ ) {
				String fmt = bodyFormat.getFormat(cl);
				final int loc = bodyFormat.getLocation(cl);	// 3.5.5.0
				if( ! bodyFormat.isNoClass() && loc >= 0 ) {	// 3.5.5.7 (2004/05/10)
//					final StringBuilder newtg = new StringBuilder( BUFFER_LARGE )
//					newtg.append("<td class=\"").append( getColumnDbType(loc) ).append("\" ");	// 4.0.0 (2005/01/31)
//					final String tdclass = newtg.toString();
					final String tdclass = "<td class=\"" + getColumnDbType(loc) + "\" " ;
					fmt = StringUtil.replace( bodyFormat.getFormat(cl) ,"<td", tdclass );
				}
				outBuf.append( fmt );

				// locがstackに入っていれば出力
				if ( isStackClm(loc) ){
					if( loc >= 0 ) {
						switch( bodyFormat.getType(cl) ) {
							case '#' : outBuf.append( getColumnLabel(loc) );		break;
							case '$' : outBuf.append( getRendererValue(row,loc) );	break;
							case '!' : outBuf.append( getValue(row,loc) );			break;
							default  : outBuf.append( getValueLabel(row,loc) );	break;
						}
					}
					else {
						outBuf.append( bodyFormat.getSystemFormat(row,loc) );
					}
				}
			}
			// 5.5.8.3 (2012/11/17)内部積上げの結果は出力場所の特定が難しいため一番最後尾にtd付きで出力しておきます
			if( innerStack ){
				outBuf.append("</td><td><div class='stackDivParent' capacity='"+ cap + "' style='width:100%; position:relative;'>"); // 5.6.1.2 (2013/02/22) td終了追加
				for( int cs = 0; cs < costAry.length; cs++ ){
					outBuf.append("<div class='stackDiv' style='position:absolute; top:0px;' num='")
						.append( cs)
						.append("' stackedCost='")
						.append( costAry[cs] )
						.append( "'></div>");
				}
				outBuf.append("</div>");
			}

			outBuf.append( bodyFormat.getFormat(cl) )
				.append("</tbody>").append( CR );
		}
		return outBuf;
	}
}
