/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.io;

// import java.io.BufferedReader;
import java.io.File;										// 6.2.0.0 (2015/02/27)
import java.util.Map;

import org.opengion.fukurou.util.HybsConst ;				// 6.1.0.0 (2014/12/26)
import org.opengion.fukurou.util.CSVTokenizer;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.resource.CodeData;

/**
 * 指定の区切り記号(初期値:タブ区切り)ファイルの読み取りクラスです。
 *
 * 名前，データの入力部のみオーバーライドすれば，各種入力フォーマットに合わせた
 * サブクラスを実現する事が可能です。
 *
 * @og.group ファイル入力
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public abstract class AbstractTableReader implements TableReader {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.2.0.0 (2015/02/27)" ;

	/** システムの改行コードを設定します。*/
	protected static final String CR		 = HybsConst.CR;			// 6.1.0.0 (2014/12/26) refactoring
	/** StringBilderなどの初期値を設定します。	{@value} */
	protected static final int BUFFER_MIDDLE = HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

//	private String	separator	= TAB_SEPARATOR;		// 項目区切り文字
	protected String	separator	= TAB_SEPARATOR;	// 6.2.0.0 (2015/02/27) 項目区切り文字。protected化
	private ResourceManager resource	;				// 4.0.0 (2005/01/31)
	private int		maxRowCount	= HybsSystem.sysInt( "DB_MAX_ROW_COUNT" ) ;

	protected DBTableModel	table		;
	protected DBColumn[]	dbColumn	;
	private   int[]			clmNo		;			// 6.1.0.0 (2014/12/26) 出力対象のカラム番号配列

	// 3.5.4.5 (2004/01/23) カラム名の外部指定を出来る様にする。
	protected String	columns		;			// 外部指定のカラム名 ( 4.3.4.7 (2009/01/22) protectedに変更 )
	// 6.1.0.0 (2014/12/26) 読み取り対象外のカラム列を、外部(タグ)より指定する。
	private String		omitNames	;			// 6.1.0.0 (2014/12/26)

//	private String	  encode		;			// 6.2.0.0 (2015/02/27) 廃止
	private boolean	  useNumber		= true;		// 3.7.0.5 (2005/04/11)

	private int		  skipRowCount	;			// 5.1.6.0 (2010/05/01) データの読み飛ばし設定
	private boolean	  useRenderer	;			// 5.2.1.0 (2010/10/01)

	// 5.2.1.0 (2010/10/01) コードリソース毎のラベル逆引きマップ
	private Map<?,?>[] maps			;			// 5.5.1.7 (2012/04/16) ワイルドカードを指定

	// 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)
	protected String  sheetName		;		// 3.5.4.2 (2003/12/15)
	protected String  sheetNos		;		// 5.5.7.2 (2012/10/09)

	protected String  constKeys		;		// 5.5.8.2 (2012/11/09) 固定値となるカラム名(CSV形式)
	protected String  constAdrs		;		// 5.5.8.2 (2012/11/09) 固定値となるアドレス(行-列,行-列,・・・)
	protected String  nullBreakClm	;		// 5.5.8.2 (2012/11/09) 取込み条件/Sheet BREAK条件

	private boolean	  useDebug		;			// 5.5.7.2 (2012/10/09) デバッグ情報の出力するかどうか

//	/**
//	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
//	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
//	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
//	 * このメソッドは、EXCEL 読み込み時に使用します。
//	 *
//	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。廃止
//	 *
//	 * @see #isExcel()
//	 */
//	abstract public void readDBTable();

//	/**
//	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
//	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
//	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
//	 *
//	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。廃止
//	 *
//	 * @param   reader BufferedReaderオブジェクト
//	 */
//	abstract public void readDBTable( final BufferedReader reader );

	/**
	 * DBTableModel から 各形式のデータを作成して,BufferedReader より読み取ります。
	 * コメント/空行を除き、最初の行は、必ず項目名が必要です。
	 * それ以降は、コメント/空行を除き、データとして読み込んでいきます。
	 *
	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。新規
	 *
	 * @param   filename 読み取り元ファイル名
	 * @param   enc ファイルのエンコード文字列
	 */
	abstract public void readDBTable( final File filename , final String enc );

	/**
	 * リソースマネージャーをセットします。
	 * これは、言語(ロケール)に応じた DBColumn をあらかじめ設定しておく為に
	 * 必要です。
	 * リソースマネージャーが設定されていない、または、所定のキーの DBColumn が
	 * リソースに存在しない場合は、内部で DBColumn オブジェクトを作成します。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) lang ⇒ ResourceManager へ変更
	 *
	 * @param  resource リソースマネージャー
	 */
	public void setResourceManager( final ResourceManager resource ) {
		this.resource = resource;
	}

//	/**
//	 * DBColumn オブジェクトをDBTable に設定します。
//	 *
//	 * @og.rev 3.5.4.2 (2003/12/15) private を protected に変更。
//	 * @og.rev 3.5.4.5 (2004/01/23) DBColumn 配列に値をセットします。
//	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer対応(コードリソース毎のラベル逆引き)
//	 *
//	 * @param names カラム名配列(可変長引数)
//	 */
////	protected void setTableDBColumn( final String[] names ) {
//	protected void setTableDBColumn( final String... names ) {
//		dbColumn = new DBColumn[names.length] ;	 // 3.5.4.5 追加
//		for( int i=0; i<names.length; i++ ) {
//			final DBColumn clm = resource.makeDBColumn( names[i] );
//			table.setDBColumn( i,clm );
//			dbColumn[i] = clm;		  // 3.5.4.5 追加
//		}
//
//		if( useRenderer ) {
//			maps = new Map<?,?>[names.length];		// 5.5.1.7 (2012/04/16) ワイルドカードを指定
//			for( int i=0; i<names.length; i++ ) {
//				final CodeData cd = dbColumn[i].getCodeData();
//				if( cd != null ) { maps[i] = cd.makeLabelMap(); }
//				else             { maps[i] = null; }
//			}
//		}
//	}

	/**
	 * DBColumn オブジェクトをDBTable に設定します。
	 *
	 * ここでは、omitNames 属性を加味した、カラム名を作成します。
	 * その際、ゼロ文字列のカラム名は、omit します。
	 * 戻り値は、新しいカラム数(omitNames を加味、ゼロ文字列のカラムも除去)です。
	 *
	 * ※ 6.1.0.0 (2014/12/26) で、omitNames 属性を追加します。
	 *    これに伴い、従来は、EXCELのみ、#NAME で、ゼロ文字列のカラム名や
	 *    columns 指定で、a,b,,,,,,x のようなカラム名指定で、カラム飛ばしを
	 *    実装していましたが、その他の Reader でも、対応します。
	 *    これは、互換性に影響しますので、ご注意ください。
	 *    (読み込んでも、カラム名が無いので、使えなかったと思いますけど…)
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) private を protected に変更。
	 * @og.rev 3.5.4.5 (2004/01/23) DBColumn 配列に値をセットします。
	 * @og.rev 5.2.1.0 (2010/10/01) useRenderer対応(コードリソース毎のラベル逆引き)
	 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
	 *
	 * @param names カラム名配列
	 * @return 新しいカラム数(omitNames を加味、ゼロ文字列のカラムも除去)
	 */
	protected int setTableDBColumn( final String[] names ) {
		clmNo = new int[names.length];			// 全カラムのカラム番号配列を作成(-1 が読み飛ばし)

		int size = 0;
		// omitするカラム名や、カラム名が、ゼロ文字列の場合は、セットしない。

		// omit するカラム名を見つける処理
		if( omitNames != null ) {
			final String omit = "," + omitNames + "," ;			// 前後にキーをマッチさせるための "," を追加
			for( int i=0; i<names.length; i++ ) {
				if( omit.indexOf( "," + names[i] + "," ) < 0 ) {
					clmNo[i] = size++;		// 不一致 → names として残る。
				}
				else {
					clmNo[i] = -1;			// 一致 → omitするカラム
				}
			}
		}
		else {
			size = names.length;
			for( int i=0; i<names.length; i++ ) {
				clmNo[i] = i;
			}
		}

		// TableModelの作成
		final String[] nms = new String[size];
		table.init( size );
		dbColumn = new DBColumn[size] ;
		if( useRenderer ) { maps = new Map<?,?>[names.length]; }		// 5.5.1.7 (2012/04/16) ワイルドカードを指定
		int no = 0;
		for( int i=0; i<names.length; i++ ) {
			if( clmNo[i] >= 0 ) {
				final DBColumn clm = resource.makeDBColumn( names[i] );	// 指定の名前の DBColumn を作成する。
				table.setDBColumn( no,clm );
				dbColumn[no++] = clm;

				if( useRenderer ) {
					final CodeData cd = clm.getCodeData();
					if( cd != null ) { maps[i] = cd.makeLabelMap(); }
					else             { maps[i] = null; }
				}
			}
		}

		return size ;

//		dbColumn = new DBColumn[names.length] ;	 // 3.5.4.5 追加
//		for( int i=0; i<names.length; i++ ) {
//			final DBColumn clm = resource.makeDBColumn( names[i] );
//			table.setDBColumn( i,clm );
//			dbColumn[i] = clm;		  // 3.5.4.5 追加
//		}
//
//		if( useRenderer ) {
//			maps = new Map<?,?>[names.length];		// 5.5.1.7 (2012/04/16) ワイルドカードを指定
//			for( int i=0; i<names.length; i++ ) {
//				final CodeData cd = dbColumn[i].getCodeData();
//				if( cd != null ) { maps[i] = cd.makeLabelMap(); }
//				else             { maps[i] = null; }
//			}
//		}
	}

	/**
	 * DBTableModelオブジェクトに、１行分のデータを追加します。
	 * これ自体は、メソッドの共通化による 拡張をしやすくするために用意しました。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 * @og.rev 6.2.0.0 (2015/02/27) コードリソースの逆引き処理の前に、Key:Val 分解を実行する。
	 *
	 * @param values １行分のデータ配列
	 */
	protected void setTableColumnValues( final String[] values ) {

		// コードリソースの逆引き対応
		if( useRenderer ) {
			final String[] vals = new String[table.getColumnCount()];
			int no = 0;
			for( int i=0; i<values.length; i++ ) {
				if( clmNo[i] >= 0 ) {
					String val = values[i];
					if( maps[i] != null ) {					// != null でコードリソース判定も兼ねる。
						if( val == null ) { val = ""; }
						else {
							// 6.2.0.0 (2015/02/27) 逆引き処理の前に、Key:Val 分解を実行する。
							final int sp = val.indexOf( ':' );			// Key:Val で、分解してみる。
							if( sp >= 0 ) {
								val = val.substring( 0,sp );
							}
							else {
								final String tmp = (String)maps[i].get( val );
								if( tmp != null ) { val = tmp; }				// Map にあれば、ラベル→コードの逆引成功
							}

//							final String tmp = (String)maps[i].get( val );
//							if( tmp != null ) { val = tmp; }				// Map にあれば、ラベル→コードの逆引成功
//							else {
//								final int sp = val.indexOf( ':' );			// なければ、Key:Val で、分解してみる。
//								if( sp >= 0 ) {
//									val = val.substring( 0,sp );
//								}
//							}
						}
					}
					vals[no++] = val;
				}
			}
			table.addColumnValues( vals );
		}
		else {
			// omitNames 対応
			if( omitNames != null ) {				// カラムと値の対応が異なる。
				final String[] vals = new String[table.getColumnCount()];
				int no = 0;
				for( int i=0; i<clmNo.length; i++ ) {
					if( clmNo[i] >= 0 ) { vals[no++] = values[i]; }
				}
				table.addColumnValues( vals );
			}
			// 通常の設定
			else {
				table.addColumnValues( values );
			}
		}

//		if( useRenderer ) {
//			final int clmSize = values.length;
//			for( int clmNo=0; clmNo<clmSize; clmNo++ ) {
//				if( maps[clmNo] != null ) {
//					String val = values[clmNo];
//					if( val == null ) { val = ""; }
//					else {
//						final String tmp = (String)maps[clmNo].get( val );
//						if( tmp != null ) { values[clmNo] = tmp; }
//						else {
//							final int sp = val.indexOf( ':' );
//							if( sp >= 0 ) {
//								values[clmNo] = val.substring( 0,sp );
//							}
//						}
//					}
//				}
//			}
//		}
//
//		table.addColumnValues( values );
	}

	/**
	 * １行のデータを テーブルモデルにセットするように分割します。
	 *
	 * なお、読込みは，NAME項目分を読み込みます。データ件数が少ない場合は、
	 * "" をセットしておきます。
	 * データの分割は、separator文字を使用します。
	 *
	 * @og.rev 3.3.3.1 (2003/07/18) ファイルリード/ライト時に後ろスペースの除去を行います。
	 * @og.rev 3.7.0.5 (2005/04/11) useNumber 属性を考慮します。
	 *
	 * @param	data	１行のデータ
	 * @param	clmSize	カラム数
	 *
	 * @return	分割された文字列配列
	 */
	protected String[] readData( final String data,final int clmSize ) {
		String[] rtnData = new String[ clmSize ];
		final CSVTokenizer token = new CSVTokenizer( data,separator.charAt(0) );
		// 超イレギュラー処理 最初の separator 以前の文字は無視する。
		// 3.7.0.5 (2005/04/11)
		if( useNumber ) { token.nextToken(); }	  // 先頭は行番号のため無視する。

		int clmNo = 0;
		while( token.hasMoreTokens() ) {
			final String val = StringUtil.csvOutQuote( token.nextToken() );
			if( val != null && val.startsWith( "'0" ) ) {
				rtnData[clmNo++] = StringUtil.rTrim( val.substring( 1 ) );
			}
			else {
				rtnData[clmNo++] = StringUtil.rTrim( val );
			}
			if( clmNo >= clmSize ) { break; }	// 3.7.0.5 (2005/04/11) 多い場合は、以降を無視する。
		}
		// EXCEL が、終端TABを削除してしまうため、少ない場合は埋める。
		for( int i=clmNo; i<clmSize; i++ ) {
			rtnData[i] = "";
		}

		return rtnData;
	}

	/**
	 * 内部の DBTableModel を返します。
	 *
	 * @return	テーブルモデル
	 */
	public DBTableModel getDBTableModel() {
		return table;
	}

	/**
	 * データを読み込む場合の,区切り文字をセットします。
	 *
	 * なお，このメソッドは,サブクラスによっては,使用しない場合があります。
	 * もし，使用しないサブクラスを作成する場合は, UnsupportedOperationException
	 * を throw するように,サブクラスで実装して下さい。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 *
	 * @param   sep 区切り文字
	 */
	public void setSeparator( final String sep ) {
		if( sep != null ) { this.separator = sep; }
	}

//	/**
//	 * データを書き込む場合の,区切り文字を返します。
//	 *
//	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。protected化
//	 * @og.rev 6.2.0.0 (2015/02/27) 廃止
//	 *
//	 * @return	区切り文字
//	 */
////	public String getSeparator() {
//	protected String getSeparator() {
//		return separator;
//	}

	/**
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します
	 *		(初期値:DB_MAX_ROW_COUNT[={@og.value org.opengion.hayabusa.common.SystemData#DB_MAX_ROW_COUNT}])。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @return  最大検索件数
	 */
	public int getMaxRowCount() {
		return maxRowCount;
	}

	/**
	 * DBTableModelのデータとして登録する最大件数をこの値に設定します
	 *		(初期値:DB_MAX_ROW_COUNT[={@og.value org.opengion.hayabusa.common.SystemData#DB_MAX_ROW_COUNT}])。
	 * サーバーのメモリ資源と応答時間の確保の為です。
	 *
	 * @og.rev 3.1.1.0 (2003/03/28) 同期メソッド(synchronized付き)を非同期に変更する。
	 * @og.rev 5.5.8.5 (2012/11/27) 0を無制限として処理します。
	 *
	 * @param   maxRowCount 最大検索件数
	 */
	public void setMaxRowCount( final int maxRowCount ) {
		this.maxRowCount = ( maxRowCount > 0 ) ? maxRowCount : Integer.MAX_VALUE ;
	}

	/**
	 * DBTableModelのデータとしてEXCELファイルを読み込むときのシート名を設定します。
	 * これにより、複数の形式の異なるデータを順次読み込むことや、シートを指定して
	 * 読み取ることが可能になります。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * のでご注意ください。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 3.5.4.2 (2003/12/15) 新規追加
	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)
	 *
	 * @param   sheetName シート名
	 */
	public void setSheetName( final String sheetName ) {
//		final String errMsg = "このメソッドは、EXCEL追加機能ですので、使用できません。";
//		throw new UnsupportedOperationException( errMsg );
		this.sheetName = sheetName;
	}

	/**
	 * EXCELファイルを読み込むときのシート番号を指定します(初期値:0)。
	 *
	 * EXCEL読み込み時に複数シートをマージして取り込みます。
	 * シート番号は、0 から始まる数字で表します。
	 * ヘッダーは、最初のシートのカラム位置に合わせます。（ヘッダータイトルの自動認識はありません。）
	 * よって、指定するシートは、すべて同一レイアウトでないと取り込み時にカラムのずれが発生します。
	 * 
	 * シート番号の指定は、CSV形式で、複数指定できます。また、N-M の様にハイフンで繋げることで、
	 * N 番から、M 番のシート範囲を一括指定可能です。また、"*" による、全シート指定が可能です。
	 * これらの組み合わせも可能です。（ 0,1,3,5-8,10-* ）
	 * ただし、"*" に関しては例外的に、一文字だけで、すべてのシートを表すか、N-* を最後に指定するかの
	 * どちらかです。途中には、"*" は、現れません。
	 * シート番号は、重複(1,1,2,2)、逆転(3,2,1) での指定が可能です。これは、その指定順で、読み込まれます。
	 * sheetNos と sheetName が同時に指定された場合は、sheetNos が優先されます。エラーにはならないのでご注意ください。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 * 
	 * 初期値は、0（第一シート） です。
	 *
	 * ※ このクラスでは実装されていません。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規追加
	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)
	 *
	 * @param   sheetNos EXCELファイルのシート番号（0から始まる）
	 * @see		#setSheetName( String ) 
	 */
	public void setSheetNos( final String sheetNos ) {
//		final String errMsg = "このメソッドは、EXCEL追加機能ですので、使用できません。";
//		throw new UnsupportedOperationException( errMsg );
		this.sheetNos = sheetNos;
	}

	/**
	 * 固定値となるカラム名(CSV形式)と、constAdrs 固定値となるアドレス(行-列,行-列,・・・)を設定します。
	 *
	 * アドレスは、EXCEL上の行-列をCSV形式で指定します。
	 * 行列は、EXCELオブジェクトに準拠するため、０から始まる整数です。
	 * 0-0 ⇒ A1 , 1-0 ⇒ A2 , 0-1 ⇒ B1 になります。
	 * これにより、シートの一か所に書かれている情報を、DBTableModel のカラムに固定値として
	 * 設定することができます。
	 * 例として、DB定義書で、テーブル名をシートの全レコードに設定したい場合などに使います。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * 5.7.6.3 (2014/05/23) より、
	 *   ①EXCEL表記に準拠した、A1,A2,B1 の記述も処理できるように対応します。
	 *     なお、A1,A2,B1 の記述は、必ず、英字1文字＋数字 にしてください。(A～Zまで)
	 *   ②処理中のEXCELシート名をカラムに割り当てるために、"SHEET" という記号に対応します。
	 * 例えば、sheetConstKeys="CLM,LANG,NAME" とし、sheetConstAdrs="0-0,A2,SHEET" とすると、
	 * NAMEカラムには、シート名を読み込むことができます。
	 * これは、内部処理の簡素化のためです。
	 *
	 * 例として、DB定義書で、テーブル名をシートの全レコードに設定したい場合などに使います。
	 * このメソッドは、isExcel() == true の場合のみ利用されます。
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 新規追加
	 *
	 * @param   constKeys 固定値となるカラム名(CSV形式)
	 * @param   constAdrs 固定値となるアドレス(行-列,行-列,・・・)
	 */
	public void setSheetConstData( final String constKeys,final String constAdrs ) {
//		final String errMsg = "このメソッドは、EXCEL追加機能ですので、使用できません。";
//		throw new UnsupportedOperationException( errMsg );
		this.constKeys = constKeys;
		this.constAdrs = constAdrs;
	}

	/**
	 * ここに指定されたカラム列に NULL が現れた時点で読み取りを中止します。
	 *
	 * これは、指定のカラムは必須という事を条件に、そのレコードだけを読み取る処理を行います。
	 * 複数Sheetの場合は、次のSheetを読みます。
	 * 現時点では、Excel の場合のみ有効です。
	 *
	 * @og.rev 5.5.8.2 (2012/11/09) 新規追加
	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)
	 *
	 * @param   clm カラム列
	 */
	public void setNullBreakClm( final String clm ) {
//		final String errMsg = "このメソッドは、EXCEL追加機能ですので、使用できません。";
//		throw new UnsupportedOperationException( errMsg );
		nullBreakClm = clm;
	}

//	/**
//	 * このクラスが、EXCEL対応機能を持っているかどうかを返します。
//	 *
//	 * EXCEL対応機能とは、シート名のセット、読み込み元ファイルの
//	 * Fileオブジェクト取得などの、特殊機能です。
//	 * 本来は、インターフェースを分けるべきと考えますが、taglib クラス等の
//	 * 関係があり、問い合わせによる条件分岐で対応します。
//	 *
//	 * @og.rev 3.5.4.3 (2004/01/05) 新規追加
//	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。廃止
//	 *
//	 * @return	EXCEL対応機能を持っているかどうか
//	 */
//	public boolean isExcel() {
//		return false;
//	}

//	/**
//	 * 読み取り元ファイル名をセットします。(DIR + Filename)
//	 * これは、EXCEL追加機能として実装されています。
//	 * ※ このクラスでは実装されていません。
//	 *
//	 * @og.rev 3.5.4.3 (2004/01/05) 新規作成
//	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。廃止
//	 *
//	 * @param   filename 読み取り元ファイル名
//	 */
//	public void setFilename( final String filename ) {
//		final String errMsg = "このメソッドは、EXCEL追加機能ですので、使用できません。";
//		throw new UnsupportedOperationException( errMsg );
//	}

	/**
	 * 読み取り元ファイルのカラム列を、外部(タグ)より指定します。
	 * ファイルに記述された #NAME より優先して使用されます。
	 *
	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
	 *
	 * @param   clms 読み取り元ファイルのカラム列(CSV形式)
	 */
	public void setColumns( final String clms ) {
		columns = clms ;
	}

	/**
	 * 読み取り対象外のカラム列を、外部(タグ)よりCSV形式で指定します。
	 *
	 * 指定するカラム名に対して、読み取り処理を行いません。
	 * ここで指定するカラム名は、ファイルの #NAME または、columns で
	 * 指定するカラム名に対して、含まれている必要はありません。
	 * その場合は、ここでの指定は無視されます。
	 *
	 * @og.rev 6.1.0.0 (2014/12/26) omitNames 属性を追加
	 *
	 * @param   clms 読み取り対象外のカラム列(CSV形式)
	 */
	public void setOmitNames( final String clms ) {
		omitNames = clms;
	}

//	/**
//	 * 読み取り元ファイルのエンコード文字列を指定します。
//	 * ファイルは、BufferedReader で受け取る為、本来は、エンコードは不要ですが、
//	 * 固定長ファイルの読み取り時のバイトコード分割時に、指定のエンコードで
//	 * 分割する必要があります。(例えば、半角文字は、Shift_JIS では、１バイト)
//	 *
//	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
//	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。廃止
//	 *
//	 * @param   enc ファイルのエンコード文字列
//	 */
//	public void setEncode( final String enc ) {
//		encode = enc;
//	}

//	/**
//	 * 読み取り元ファイルのエンコード文字列を取得します。
//	 * ファイルは、BufferedReader で受け取る為、本来は、エンコードは不要ですが、
//	 * 固定長ファイルの読み取り時のバイトコード分割時に、指定のエンコードで
//	 * 分割する必要があります。(例えば、半角文字は、Shift_JIS では、１バイト)
//	 *
//	 * @og.rev 3.5.4.5 (2004/01/23) 新規作成
//	 * @og.rev 6.2.0.0 (2015/02/27) TableReader クラスの呼び出し元メソッドの共通化(EXCEL,TEXT)。廃止
//	 *
//	 * @return	ファイルのエンコード文字列
//	 */
//	protected String getEncode() {
//		return encode;
//	}

	/**
	 * 行番号情報を指定[true:使用している/false:していない]します(初期値:true)。
	 *
	 * 通常のフォーマットでは、各行の先頭に行番号が出力されています。
	 * 読み取り時に、#NAME 属性を使用する場合は、この行番号を無視しています。
	 * #NAME 属性を使用せず、columns 属性でカラム名を指定する場合(他システムの
	 * 出力ファイルを読み取るケース等)では、行番号も存在しないケースがあり、
	 * その様な場合に、useNumber="false" を指定すれば、データの最初から読み取り始めます。
	 * この場合、出力データのカラムの並び順が変更された場合、columns 属性も
	 * 指定しなおす必要がありますので、できるだけ、#NAME 属性を使用するように
	 * してください。
	 * なお、EXCEL 入力には、この設定は適用されません。(暫定対応)
	 * 初期値は、true(使用する) です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 *
	 * @param	useNumber	行番号情報  [true:使用する/false:使用しない]
	 */
	public void setUseNumber( final boolean useNumber ) {
		this.useNumber = useNumber ;
	}

	/**
	 * データの読み始めの初期値を取得します。
	 *
	 * TAB区切りテキストやEXCEL等のデータの読み始めの初期値を指定します。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 読み飛ばしは、コメント行などは、無視しますので、実際の行数分読み飛ばします。
	 * ＃NAME属性や、columns 属性は、有効です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @return	読み始めの初期値
	 */
	public int getSkipRowCount() {
		return skipRowCount ;
	}

	/**
	 * データの読み飛ばし件数を設定します。
	 *
	 * TAB区切りテキストやEXCEL等のデータの読み始めの初期値を指定します。
	 * ファイルの先頭行が、０行としてカウントしますので、設定値は、読み飛ばす
	 * 件数になります。(１と指定すると、１件読み飛ばし、２行目から読み込みます。)
	 * 読み飛ばしは、コメント行などは、無視しますので、実際の行数分読み飛ばします。
	 * ＃NAME属性や、columns 属性は、有効です。
	 *
	 * @og.rev 5.1.6.0 (2010/05/01) 新規作成
	 *
	 * @param	count 読み始めの初期値
	 */
	public void setSkipRowCount( final int count ) {
		skipRowCount = count;
	}

	/**
	 * 読取処理でラベルをコードリソースに逆変換を行うかどうかを指定します。
	 *
	 * TableWriter_Renderer 系のクラスで出力した場合は、コードリソースがラベルで出力されます。
	 * そのファイルを読み取ると、当然、エラーになります。
	 * ここでは、コードリソースのカラムに対して、ラベルからコードを求める逆変換を行うことで、
	 * Renderer 系で出力したファイルを取り込むことができるようにします。
	 *
	 * ここでは、TableWriter 系と同様に、TableReader_Renderer 系のクラスを作るのではなく、
	 * 属性値のフラグで、制御します。
	 * 将来的には、TableWriter 系も廃止して、同様のフラグで制御するように変更する予定です。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @param	useRenderer	コードリソースのラベル変換を行うかどうかを指定
	 */
	public void setUseRenderer( final boolean useRenderer ) {
		this.useRenderer = useRenderer;
	}

	/**
	 * 行番号情報を、使用している(true)/していない(false)を返します。
	 *
	 * 通常のフォーマットでは、各行の先頭に行番号が出力されています。
	 * 読み取り時に、#NAME 属性を使用する場合は、この行番号を無視しています。
	 * #NAME 属性を使用せず、columns 属性でカラム名を指定する場合(他システムの
	 * 出力ファイルを読み取るケース等)では、行番号も存在しないケースがあり、
	 * その様な場合に、useNumber="false" を指定すれば、データの最初から読み取り始めます。
	 * この場合、出力データのカラムの並び順が変更された場合、columns 属性も
	 * 指定しなおす必要がありますので、できるだけ、#NAME 属性を使用するように
	 * してください。
	 * なお、EXCEL 入力には、この設定は適用されません。(暫定対応)
	 * 初期値は、true(使用する) です。
	 *
	 * @og.rev 3.7.0.5 (2005/04/11) 新規追加
	 * @og.rev 4.0.0.0 (2007/07/20) メソッド名変更(getUseNumber() ⇒  isUseNumber())
	 *
	 * @return	行番号情報を、使用している(true)/していない(false)を指定
	 */
	protected boolean isUseNumber() {
		return useNumber ;
	}

	/**
	 * デバッグ情報を出力するかどうか[true:する/false:しない]を指定します。
	 *
	 * EXCELなどを読み取る場合、シートマージで読み取ると、エラー時の行番号が、連番になるため、
	 * どのシートなのか、判らなくなります。
	 * そこで、どうしてもわからなくなった場合に備えて、デバッグ情報を出力できるようにします。
	 * 通常は使用しませんので、設定を無視します。
	 * 初期値は、false:デバッグ情報を出力しない です。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @param	useDebug	デバッグ出力するか [true:する/false:しない]
	 */
	public void setDebug( final boolean useDebug ) {
		this.useDebug = useDebug;
	}

	/**
	 * デバッグ情報を出力するかどうか[true:する/false:しない]を取得します。
	 *
	 * EXCELなどを読み取る場合、シートマージで読み取ると、エラー時の行番号が、連番になるため、
	 * どのシートなのか、判らなくなります。
	 * そこで、どうしてもわからなくなった場合に備えて、デバッグ情報を出力できるようにします。
	 *
	 * @og.rev 5.5.7.2 (2012/10/09) 新規作成
	 *
	 * @return	デバッグ出力 [true:する/false:しない]
	 */
	protected boolean isDebug() {
		return useDebug ;
	}
}
