/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.taglet;

import com.sun.tools.doclets.Taglet;
import com.sun.javadoc.Tag;
import java.util.Map;

import org.opengion.fukurou.util.StringUtil;
// import static org.opengion.fukurou.util.HybsConst.BUFFER_MIDDLE;	// 6.1.0.0 (2014/12/26) refactoring

/**
 * ソースコメントから、Javadoc を作成する場合のカスタムタグ情報を作成する
 * Taglet インターフェースの実装クラスを作成します。
 * og.rev タグ(変更履歴)を処理します。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class TagletRev extends AbstractTaglet {

	private static final String NAME   = "og.rev";
	private static final String HEADER = "変更履歴:";

	/**
	 * 実行時にドックレットがタグレットを読み込んで使用するには、
	 * そのタグレットが、次のシグニチャでマップ を引数として受け取る、
	 * レジスタ と呼ばれる static メソッドをもっている必要があります。
	 * このメソッドは、タグレット名をキーとして、カスタムタグレットの
	 * インスタンスをマップに追加します。 タグレットをオーバーライドする場合、
	 * 名前の競合を避けるため、新しいタグレットのインスタンスをマップに
	 * 追加する前に、オーバーライドされる側のタグレットをマップから
	 * 削除する必要があります。
	 *
	 * @param tagletMap タグレットマップ
	 */
	public static void register( final Map<String,Taglet> tagletMap ) {
	   final TagletRev tagRev = new TagletRev();
	   final Taglet tag = tagletMap.get(NAME);
	   if(tag != null) {
		   tagletMap.remove(NAME);
	   }
	   tagletMap.put(NAME, tagRev);
	}

	/**
	 * このカスタムタグの名前を返します。
	 *
	 * @return カスタムタグの名前
	 * @og.rtnNotNull
	 */
	@Override
	public String getName() {
		return NAME;
	}

	/**
	 * このカスタムタグのタグ表現を受け取り、
	 * 文字列としての表現を返し、生成されたページに出力します。
	 *
	 * @param tagTag このカスタムタグのタグ表現
	 *
	 * @return このタグの文字列としての表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString( final Tag tagTag ) {
		return "<li />"
			   + StringUtil.htmlFilter( tagTag.text() ) ;
	}

	/**
	 * このカスタムタグのタグ表現の配列を受け取り、
	 * 文字列としての表現を返し、生成されたページに出力します。
	 * このタグレットがインラインタグを表す場合、
	 * このメソッドは null を返します。
	 *
	 * @og.rev 5.1.9.0 (2010/08/01) dtタグをきちんと記述する。
	 * @og.rev 6.0.2.0 (2014/09/19) revは、linkタグなど使用しないので、削除
	 *
	 * @param tagTags	このカスタムタグを表すタグの配列
	 *
	 * @return このタグの文字列としての表現
	 * @see		com.sun.tools.doclets.Taglet#toString( Tag[] )
	 */
	@Override
	public String toString( final Tag[] tagTags ) {
		if(tagTags.length == 0) {
			return null;			// インラインタグを表す
		}
		final StringBuilder buf = new StringBuilder( BUFFER_MIDDLE )
			.append( "<dt><b>" ).append( HEADER ).append( "</b></dt>" )		// 5.1.9.0 (2010/08/01)
			.append( "<dd><table cellpadding=\"2\" cellspacing=\"0\">" );
		for(int i = 0; i < tagTags.length; i++) {
			buf.append( "<tr><td>" )
				.append( tagTags[i].text() )
				.append( "</td></tr>" );
		}
		buf.append( "</table></dd>\n" );
		return buf.toString();
	}
}
