/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.ErrMsg;
import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.hayabusa.html.ViewForm;
import org.opengion.hayabusa.html.ViewFormFactory;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.hayabusa.db.DBColumn;

import java.util.Arrays;
import java.util.Comparator;

/**
 *  Taglib で共通的に使用される ユーティリティクラスです。
 *
 * 全てのメソッドは、 <del>protected</del>パッケージプライベートな
 *  static 宣言されています。
 *
 * @og.rev 3.1.1.0 (2003/03/28) 新規作成
 * @og.group その他部品
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
final class TaglibUtil  {
	static final String VIEW_ID = "HYBS_VIEW" ;		// 3.5.6.4 (2004/07/16)
	static final String LINK_ID = "HYBS_LINK" ;		// 3.5.6.4 (2004/07/16)
	static final String MARK_ID = "HYBS_MARK" ;		// 3.5.6.4 (2004/07/16)

	private static int tagId = 0;

	/**
	 *	デフォルトコンストラクターをprivateにして、
	 *	オブジェクトの生成をさせないようにする。
	 *
	 */
	private TaglibUtil() {}

	/**
	 * チェックボックスのラベルを関連つけるための id 値を返します。
	 *
	 * InputTag と OrderByTag 全体で、呼び出される都度に連番をつけて返します。
	 * ユニークな番号になる様にする事が目的なので、99999 で、0 にクリアします。
	 * 別スレッドによる同時アクセスで整合性が破綻(同じ番号が返る)しても、
	 * 問題ありません。(単独のスレッド内でユニークになればよい。)
	 *
	 * @return  ユニークな id 値
	 */
	static String getTagId() {
		String id = "ID" + ( tagId++ );

		if( tagId >= 100000 ) { tagId = 0; }
		return ( id );
	}

	/**
	 * ErrorMessage オブジェクトと、ResourceManager より、表示用 HTMLテーブルを作成します。
	 * これは、従来，各Taglibパッケージで使用していました、ErrorMessage オブジェクトを、
	 * HTMLテーブル形式で表示する為に、DBUtilクラスや、ViewFormクラスなど、複数のクラスを
	 * 複雑に組み合わせて使用していましたが、一つの static メソッドにまとめたものです。
	 *
	 * @og.rev 3.5.5.2 (2004/04/02) 新規追加
	 * @og.rev 3.5.6.1 (2004/06/25) lang 言語コード 属性を削除します。
	 * @og.rev 3.5.6.2 (2004/07/05) setBgColorCycle の引数定義を変更
	 * @og.rev 3.8.0.4 (2005/08/08) setUseScrollBar の追加(ヘッダー固定のスクロールバーを出さない)
	 * @og.rev 3.8.9.5 (2007/09/12) PG名、ステップ名追加(開発ガイドライン対応)
	 * @og.rev 4.0.0.0 (2005/01/31) ResourceManager getMessage 関数を使用
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 * @og.rev 4.3.2.0 (2008/09/11) 行番号順ソート、メッセージのラベル(短)化、コメント出力
	 * @og.rev 4.3.4.5 (2009/01/08) 対応方法(概要説明)が存在すれば表示する
	 * @og.rev 4.3.7.6 (2009/07/15) 対応方法のパラメータ対応
	 *
	 * @param   errMsg  ErrorMessageオブジェクト
	 * @param   resource ResourceManager リソースマネージャー
	 *
	 * @return  HTML のテーブルタグに変換後の文字列(無ければ、ゼロストリング)
	 */
	static String makeHTMLErrorTable( final ErrorMessage errMsg,final ResourceManager resource ) {
		if( errMsg == null ) { return ""; }

		final String[] names ;
		if( errMsg.isSetPgStep() ) {
			// names = new String[] { "ROW_NO","KEKKA","MSGID","MESSAGE","PG","STEP" }; // 3.8.9.5 (2007/09/12)
			names = new String[] { "ROW_NO","KEKKA","MSGID","MESSAGE", "PG","STEP" }; // 3.8.9.5 (2007/09/12)
		}
		else {
			names = new String[] { "ROW_NO","KEKKA","MSGID","MESSAGE" };
		}

		int clmSize = names.length;

		DBTableModel errTable = DBTableModelUtil.newDBTable();
		errTable.init( clmSize );

		for( int i=0; i<clmSize; i++ ) {
			DBColumn dbColumn = resource.makeDBColumn( names[i] );
			errTable.setDBColumn( i,dbColumn );
		}

	 	// 4.3.2.0 (2008/09/11) 行番号順ソート
		ErrMsg[] errMsgs = errMsg.toArray();
		Arrays.sort( errMsgs, errComp );

		int tableRow = 0; // 4.3.4.5 (2009/01/08) テーブルモデルの行番号

		for( int i=0; i<errMsg.size(); i++ ) {
			ErrMsg err = errMsgs[i];
			String[] clmVals = new String[clmSize];
			clmVals[0] = String.valueOf( err.getNo() );
			clmVals[1] = String.valueOf( err.getKekka() );
			clmVals[2] = err.getId();
			clmVals[3] = resource.getLabel( err );
			if( errMsg.isSetPgStep() ) {
				clmVals[4] = err.getPg();
				clmVals[5] = err.getStep();
			}
			errTable.addColumnValues( clmVals );
			// errTable.setRowWritable( i,false );	// 書込み禁止
			errTable.setRowWritable( tableRow, false );	// 4.3.4.5 (2009/01/08) 書込み禁止

			// 対応方法(概要説明)が存在すれば直下に一行追加する 4.3.4.5 (2009/01/08)
			// String desc = resource.getDescription( clmVals[2] );
			String desc = resource.getDescription( err ); // 4.3.7.6 (2009/07/15)
			if( desc != null && desc.length() > 0 ){
				tableRow++;
				String[] descVals = new String[clmSize];
				descVals[3] = desc;
				errTable.addColumnValues( descVals );
				errTable.setRowWritable( tableRow, false );
			}

			tableRow++;
		}

//		for( int i=0; i<errMsg.size(); i++ ) {
//			String[] clmVals = new String[clmSize];
//			clmVals[0] = String.valueOf( errMsg.getNo(i) );
//			clmVals[1] = String.valueOf( errMsg.getKekka(i) );
//			clmVals[2] = errMsg.getId(i);
//			ErrMsg err = errMsg.getErrMsg( i );				// 4.0.0 (2005/01/31)
////			clmVals[3] = resource.getMessage( err );		// 4.0.0 (2005/01/31)
//			clmVals[3] = resource.getLabel( err );			// 4.0.0 (2005/01/31)
//			if( errMsg.isSetPgStep() ) {
//				clmVals[4] = errMsg.getPg(i); // 3.8.9.5 (2007/09/12)
//				clmVals[5] = errMsg.getStep(i); // 3.8.9.5 (2007/09/12)
//			}
//			errTable.addColumnValues( clmVals );
//			errTable.setRowWritable( i,false );	// 書込み禁止
//		}

		// ここまでが、DBTableModel の作成。以下は、View での表示

		ViewForm form = ViewFormFactory.newInstance( "HTMLTable" );

		form.init( errTable );
		if( ! errMsg.isOK() )  { 	// 正常以外
			form.setBgColorCycle( -1 * errMsg.getKekka() );	// エラー処理時のゼブラ模様
			form.setNumberType( "delete" );			// 番号欄を出さない。
			form.setUseTableSorter( false );		// ソート用リンクは出さない。
//			form.setUseScrollBar( false,null );		// 4.0.0 (2005/01/31) スクロールバー用のDIV要素を出力しない。
			form.setUseScrollBar( false );			// 4.0.0 (2005/01/31) スクロールバー用のDIV要素を出力しない。
		}

		return form.create();
	}

	/**
	 * ErrorMessage オブジェクトと、ResourceManager より、表示用 HTMLテーブルを作成します。
	 * このメソッドでは、エラーメッセージ及びエラーコードを一覧形式で表示します。
	 *
	 * @param   errMsg  ErrorMessageオブジェクト
	 * @param   resource ResourceManager リソースマネージャー
	 *
	 * @return  HTML のテーブルタグに変換後の文字列(無ければ、ゼロストリング)
	 */
	static String makeHTMLSimpleErrorList( final ErrorMessage errMsg,final ResourceManager resource ) {
		if( errMsg == null ) { return ""; }

		String[] names = new String[] { "MESSAGE" };

		int clmSize = names.length;

		DBTableModel errTable = DBTableModelUtil.newDBTable();
		errTable.init( clmSize );

		for( int i=0; i<clmSize; i++ ) {
			DBColumn dbColumn = resource.makeDBColumn( names[i] );
			errTable.setDBColumn( i,dbColumn );
		}

		ErrMsg[] errMsgs = errMsg.toArray();
		Arrays.sort( errMsgs, errComp );

		for( int i=0; i<errMsg.size(); i++ ) {
			ErrMsg err = errMsgs[i];
			String[] clmVals = new String[clmSize];
			clmVals[0] = resource.getShortLabel( err.getId() ) + "(" + err.getId() + ")";
			errTable.addColumnValues( clmVals );
			errTable.setRowWritable( i,false );
		}

		// ここまでが、DBTableModel の作成。以下は、View での表示
		ViewForm form = ViewFormFactory.newInstance( "HTMLSimpleList" );
		form.init( errTable );

		if( ! errMsg.isOK() )  { 	// 正常以外
			form.setBgColorCycle( -1 * errMsg.getKekka() );	// エラー処理時のゼブラ模様
		}

		return form.create();
	}

	/**
	 * 指定の文字列から、数字部のみ取り出して、変換して返します。
	 *
	 * 例えば、"200px" を 200 に変換します。
	 *
	 * @og.rev 4.0.0.0 (20065/12/05) 新規作成
	 *
	 * @param	inStr 	入力する文字列
	 *
	 * @return  変換後の数字
	 */
	static int changeInt( final String inStr ) {
		if( inStr == null || inStr.length() == 0 ) { return 0; }

		int adrs;
		for( adrs=0; adrs<inStr.length(); adrs++ ) {
			char ch = inStr.charAt(adrs);
			if( ch < '0' || ch > '9' ) { break; }
		}

		if( adrs == 0 ) { return 0; }

		return Integer.parseInt( inStr.substring( 0,adrs ) );
	}

 	// 4.3.2.0 (2008/09/11) 行番号順ソートのための、Comparator オブジェクト
	private static final Comparator<? super ErrMsg> errComp =
		new Comparator<ErrMsg>() {
			public int compare( final ErrMsg err1,final ErrMsg err2 ) {
				return err1.getNo() - err2.getNo() ;
			}
		} ;

}
