/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.db.DBTableModelSorter;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBTableModelUtil;
import org.opengion.fukurou.util.StringUtil;

import static org.opengion.fukurou.util.StringUtil.nval ;

import java.util.Arrays;
import java.io.File;
import java.io.FileFilter;
import java.io.ObjectOutputStream;
import java.io.ObjectInputStream;
import java.io.IOException;

/**
 * ファイルを検索し、DBTableModel にセットするタグです。
 *
 * ファイルの検索結果は、"LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH"
 * のカラムを持つ DBTableModel にセット されます。このカラムは、固定です。
 * 並び替えについては、このタグで指定しますが、ファイルの選別(where 条件)は、
 * BODY 部に記述する fileWhere タグで指定します。(複数指定可能))
 *
 * [カラム名]      検索するファイルの属性は、以下のカラム名で作成されます。
 *      LEVEL           ディレクトリを展開する場合のレベル。
 *      FILE_TYPE       ファイル(F)かディレクトリ(D)であるか判定。
 *      PARENT          この抽象パス名の親のパス名文字列を返します。
 *      NAME            この抽象パス名が示すファイルまたはディレクトリの名前を返します。
 *      LASTMODIFIED    最後に変更された時刻を返します。
 *      FILE_LENGTH     ファイルの長さを返します。
 *      RWH             読み込み、書き込み、隠し属性をそれぞれ、ｒ,w,h で表します。
 *
 * [from 属性]      検索を開始するファイルまたはディレクトリの名前
 *
 * [multi 属性]     ディレクトリを下位展開するかどうか？
 *      true            下位展開する
 *      false           下位展開しない(初期値)
 *
 * [tableId 属性]       settion に登録する時のID
 * [scope 属性]         settion に登録する時のスコープ "request","page","session","applicaton"
 * [maxRowCount 属性]   検索時の最大検索件数(初期値:DB_MAX_ROW_COUNT[=1000])(0:[無制限])
 * [displayMsg 属性]    検索終了時に表示する メッセージリソースのID
 * [overflowMsg 属性]   オーバーフロー時に表示する メッセージリソースのID
 * [command 属性]       実行を制御するコマンド "NEW" と "RENEW" 時のみ実行する(初期値:NEW)
 * [orderBy 属性]       ソートするカラムを指定します(一つのみ)
 * [desc    属性]       逆順でソートする場合に、true を指定します。
 * [addClms 属性]       検索結果のカラム列に追加するカラム名を、カンマ区切り文字で指定します。
 * [useUpdateClms 属性] trueをセットすると、fileUpdateで使用する TO_PARENT、TO_NAMEカラムを追加します(初期値:false)
 * [fileType属性]       選択対象(FILE,DIR)を指定します。下位展開は考慮(multi 属性準拠)されます。
 * [addFrom属性]        from属性で指定された基準ファイル/フォルダ自体をリストに追加するかどうか(初期値:true)
 *
 * @og.formSample
 * ●形式：&lt;og:fileQuery from="…" multi="true/false" &gt;
 *             &lt;og:fileWhere … /&gt;
 *              …
 *         &lt;/og:fileQuery&gt;
 * ●body：あり(EVAL_BODY_BUFFERED:BODYを評価し、{&#064;XXXX} を解析します)
 *
 * ●Tag定義：
 *   &lt;og:fileQuery
 *       from             ○【TAG】ファイルの検索元となるディレクトリを指定します。(必須)
 *       multi              【TAG】多段階展開するか、１レベル展開するかどうか[true/false]を指定します(初期値:false)
 *       tableId            【TAG】(通常は使いません)結果のDBTableModelを、sessionに登録するときのキーを指定します
 *       level              【TAG】多段階展開するレベルを指定します(初期値:100)
 *       orderBy            【TAG】ソートするカラム名を指定します(一つのみ)
 *       desc               【TAG】表示順を逆転するかどうか[true/false]を指定します(初期値:false)
 *       addClms            【TAG】検索結果のカラム列に追加するカラム名を、カンマ区切り文字で指定します
 *       useUpdateClms      【TAG】TO_PARENT、TO_NAMEカラムを追加するかどうか[true/false]を指定します(初期値:false)
 *       fileType           【TAG】ファイル名が、指定されたファイルタイプと一致した場合、スルー(選択)されます
 *       addFrom            【TAG】from属性で指定された基準ファイル/フォルダ自体をリストに追加するかどうか[true/false]を指定します(初期値:true)
 *       command            【TAG】コマンド(NEW,RENEW)をセットします(PlsqlUpdateTag,UpdateTag の場合は、ENTRY)
 *       maxRowCount        【TAG】(通常は使いません)データの最大読み込み件数を指定します (初期値:DB_MAX_ROW_COUNT[=1000])
 *       displayMsg         【TAG】検索結果を画面上に表示するメッセージリソースIDを指定します (初期値:VIEW_DISPLAY_MSG[=])
 *       notfoundMsg        【TAG】検索結果がゼロ件の場合に表示するメッセージリソースIDを指定します(初期値:MSG0077[対象データはありませんでした])
 *       scope              【TAG】キャッシュする場合のスコープ[request/page/session/applicaton]を指定します(初期値:session)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:fileQuery&gt;
 *
 * ●使用例
 *    ・一般的な属性でファイルの検索を行います。
 *         &lt;og:fileQuery
 *                from    = "d:/webapps/dbdef/jsp/"
 *                multi   = "true"
 *                command = "{&#064;command}"  &gt;
 *            &lt;og:fileWhere endWith=".jsp" /&gt;
 *      &lt;/og:fileQuery&gt;
 *
 *    ・最終変更日で逆順ソートする。対象は、2002/10/01 以降に変更されたファイル。
 *        &lt;og:fileQuery
 *                from    = "d:/webapps/dbdef/jsp/"
 *                multi   = "true"
 *                orderBy = "LASTMODIFIED"
 *                desc    = "true"
 *                command = "{&#064;command}"  &gt;
 *            &lt;og:fileWhere lastModified="20021001000000" /&gt;
 *        &lt;/og:fileQuery&gt;
 *
 * @og.rev 4.0.0.0 (2005/01/31) 内部ロジック改定
 * @og.group その他入力
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class FileQueryTag extends QueryTag {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.3.9.0 (2011/09/01)" ;

	private static final long serialVersionUID = 539020110901L ;

	private static final String[] SELECT =
				new String[] { "LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH" };

	private static final int LEVEL			= 0;
	private static final int FILE_TYPE		= 1;
	private static final int PARENT			= 2;
	private static final int NAME			= 3;
	private static final int LASTMODIFIED	= 4;
	private static final int FILE_LENGTH	= 5;
	private static final int RWH			= 6;

	private static final String[] USE_UPDATE_CLM = new String[] { "TO_PARENT","TO_NAME" };	// 5.3.4.0 (2011/04/01)

	private transient FileFilter filter	= null;							// FileWhere で指定したフィルター

	private boolean		multi			= false;						// 下位層展開ﾌﾗｸﾞ
	private int			level			= 100;							// 展開ﾚﾍﾞﾙ
	private String      from			= HybsSystem.sys( "FILE_URL" );	// 検索起点ﾌｧｲﾙ
	private String		orderBy			= null;							// 5.3.4.0 (2011/04/01) ソートカラム
	private boolean		desc			= false;						// 5.3.4.0 (2011/04/01) ソートの方向(true:逆順)
	private String[]	addClms			= new String[0];				// 5.3.4.0 (2011/04/01) 追加カラム配列
	private String[]	valClms			= null;							// 5.3.4.0 (2011/04/01) 追加カラム配列の初期値
	private String		fileType		= null;							// 5.3.4.0 (2011/04/01) 選択対象を指定(FILE,DIR)
	private boolean		useUpdateClms	= false;						// 5.3.4.0 (2011/04/01) TO_PARENT、TO_NAMEカラムを追加(true:追加)
	private boolean		addFrom			= true;							// 5.3.9.0 (2011/09/01) from属性で指定された基準ファイル/フォルダ自体をリストに追加するかどうか

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定カラムのソート処理機能追加
	 * @og.rev 5.3.5.0 (2011/05/01) 最初のファイルが存在する場合のみ、実行する。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		executeCount = 0;

		table = initDBTable();
		if( maxRowCount < 0 ) {
			maxRowCount	= sysInt( "DB_MAX_ROW_COUNT" ) ;
		}

//		execute( new File( from ),0 ) ;	// 5.3.5.0 (2011/05/01) 廃止

		// 5.3.5.0 (2011/05/01) 最初のファイルが存在する場合のみ、実行する。
		File fin = new File( from );
		if( fin.exists() ) {
			execute( fin,0 ) ;

			// 5.3.4.0 (2011/04/01) 指定カラムのソート処理
			if( orderBy != null ) {
				int clmNo = table.getColumnNo( orderBy );
				DBTableModelSorter temp = new DBTableModelSorter();
				temp.setModel( table );
				temp.sortByColumn( clmNo,!desc );	// 注意 desc の値と ソート正逆は、反対です。
				table = temp;
			}
		}

		return(SKIP_BODY);
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定カラムのソート処理機能、カラム追加機能、fileType追加
	 * @og.rev 5.3.9.0 (2011/09/01) addFrom属性追加
	 */
	@Override
	protected void release2() {
		super.release2();
		multi			= false;
		level			= 100;
		from			= HybsSystem.sys( "FILE_URL" );
		filter			= null;
		orderBy			= null;				// 5.3.4.0 (2011/04/01) ソートカラム
		desc			= false;			// 5.3.4.0 (2011/04/01) 降順ﾌﾗｸﾞ
		addClms			= new String[0];	// 5.3.4.0 (2011/04/01) 追加カラム配列
		valClms			= null;				// 5.3.4.0 (2011/04/01) 追加カラム配列の初期値
		fileType		= null;				// 5.3.4.0 (2011/04/01) 選択対象を指定(FILE,DIR,ALL)
		useUpdateClms	= false;			// 5.3.4.0 (2011/04/01) TO_PARENT、TO_NAMEカラムを追加(true:追加)
		addFrom			= true;				// 5.3.9.0 (2011/09/01) addFrom属性追加
	}

	/**
	 * FileQuery を実行します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) fileType の条件に合致する場合だけ、データを作成する。
	 * @og.rev 5.3.7.0 (2011/07/01) フォルダにアクセスできない場合は、null となるのでその対応
	 * @og.rev 5.3.9.0 (2011/09/01) addFrom属性追加
	 *
	 * @param	fin	検索を開始するファイル/ディレクトリ
	 * @param	lvl	階層展開レベル
	 */
	protected void execute( final File fin,final int lvl ) {
		if( ( !multi && lvl > 1 ) || lvl > level ) { return; }	// 階層展開する、しない
		if( executeCount > maxRowCount ) { table.setOverflow( true ); return; }

		boolean isDIR = fin.isDirectory();

		if( fileType == null ||
			(  isDIR &&  "DIR".equalsIgnoreCase( fileType ) ) ||
			( !isDIR && "FILE".equalsIgnoreCase( fileType ) ) ) {
				if( addFrom || ( !addFrom && lvl > 0 ) ) {
					addFileData( executeCount++,lvl,fin );
				}
		}
//		if( fin.isDirectory() ) {
		if( isDIR ) {
			File[] list = fin.listFiles( filter );
			// 5.3.7.0 (2011/07/01) フォルダにアクセスできない場合は、null となる。
			if( list != null ) {
				for( int i = 0; i < list.length; i++ ) {
					execute( list[i],lvl+1 );
				}
			}
		}
	}

	/**
	 * 初期化された DBTableModel を返します。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定カラム追加機能追加
	 *
	 * @return	 テーブルモデル
	 */
	private DBTableModel initDBTable() {
		DBTableModel tbl = DBTableModelUtil.newDBTable();

		// TO_PARENT、TO_NAMEカラムを追加
		int clmSize = addClms.length ;
		if( useUpdateClms ) {
			if( clmSize == 0 ) {
				addClms = USE_UPDATE_CLM ;
			}
			else {
				addClms = Arrays.copyOf( addClms ,clmSize + USE_UPDATE_CLM.length );	// JDK1.6
				for( int i=0; i<USE_UPDATE_CLM.length; i++ ) {
					addClms[clmSize+i] = USE_UPDATE_CLM[i];
				}
			}
		}
		valClms = new String[addClms.length];

		tbl.init( SELECT.length + addClms.length );
		int ad=0;
		for( int i=0; i<SELECT.length; i++ ) {
			DBColumn dbColumn = getDBColumn( SELECT[i] );
			tbl.setDBColumn( ad++,dbColumn );
		}

		for( int i=0; i<addClms.length; i++ ) {
			DBColumn dbColumn = getDBColumn( addClms[i] );
			valClms[i] = dbColumn.getDefault();				// 初期値を指定しておく
			tbl.setDBColumn( ad++,dbColumn );
		}

		return tbl ;
	}

	/**
	 * DBTableModel に、ファイル情報をセットします。
	 * ファイル情報は、"LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH" です。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 指定カラム追加機能追加
	 *
	 * @param	rowNo	セットする行番号
	 * @param	lvl  	セットするレベル
	 * @param	fin		ファイル情報の元となるファイルオブジェクト
	 */
	private void addFileData( final int rowNo,final int lvl,final File fin ) {
		try {
			File file = fin.getCanonicalFile();

			String rwh = ((file.canRead())?"r":"-" ) +
							((file.canWrite())?"w":"-" ) +
							((file.isHidden())?"h":"-" ) ;

			String lastModified = HybsSystem.getDate( file.lastModified(),"yyyyMMddHHmmss" );

			String[] data = new String[ SELECT.length + addClms.length ];	// 5.3.4.0 (2011/04/01) 指定カラム追加機能
			data[LEVEL			] = String.valueOf( lvl ) ;
			data[FILE_TYPE		] = (file.isFile())?"F":"D" ;
			data[PARENT			] = file.getParent() ;
			data[NAME			] = file.getName() ;
			data[LASTMODIFIED	] = lastModified ;
			data[FILE_LENGTH	] = String.valueOf( file.length() ) ;
			data[RWH			] = rwh ;

			// これ以降の data 配列には、カラムオブジェクトの初期値を設定しておく。
			int ad = SELECT.length;
			for( int i=0; i<addClms.length; i++ ) {
				data[ad++] = valClms[i];
			}

			table.addColumnValues( data );
		}
		catch( IOException ex ) {
			String errMsg = "正式なファイル名の取得に失敗しました。[" + fin + "]"
						+ " ROW=[" + rowNo + "]"
						+ HybsSystem.CR + ex.getMessage();
			throw new HybsSystemException( errMsg,ex );
		}
	}

	/**
	 * 【TAG】ファイルの検索元となるディレクトリを指定します。
	 *
	 * @og.tag ファイルの検索元となるディレクトリを指定します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/20) 指定されたディレクトリ名の最後が"\"or"/"で終わっていない場合に、"/"を付加する。
	 *
	 * @param	url ファイルの検索元となるディレクトリ
	 */
	public void setFrom( final String url ) {
		String furl = nval( getRequestParameter( url ),null );
		if( furl != null ) {
			char ch = furl.charAt( furl.length()-1 );
			if( ch != '/' && ch != '\\' ) { furl = furl + "/"; }
		}
		furl = StringUtil.urlAppend( from,furl );
		furl = StringUtil.urlAppend( furl,"." );

		from = HybsSystem.url2dir( furl );
	}

	/**
	 * 【TAG】多段階展開するか、１レベル展開するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 初期値は、false (１レベル) です。
	 *
	 * @param	mlti 多段階展開するか [true:する/false:１レベル]
	 */
	public void setMulti( final String mlti ) {
		multi = nval( getRequestParameter( mlti ),multi );
	}

	/**
	 * 【TAG】多段階展開するレベルを指定します(初期値:100)。
	 *
	 * @og.tag
	 *
	 * @param	lvl 多段階展開するレベル
	 */
	public void setLevel( final String lvl ) {
		level = nval( getRequestParameter( lvl ),level );
	}

	/**
	 * 【TAG】ソートするカラム名を指定します(一つのみ)。
	 *
	 * @og.tag
	 * ソートするカラム名を、"LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH"
	 * から一つ選びます。
	 * これは、複数カラムでのソートはできません。
	 * 逆順にソートする場合は、desc属性を true にセットください。
	 * + をつけても、無効(カラム名がないということでエラーになります。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	clm ソートするカラム名(一つのみ)、逆順は、マイナスを付ける。
	 * @see		#setDesc( String )
	 */
	public void setOrderBy( final String clm ) {
		orderBy = nval( getRequestParameter( clm ),orderBy );

		if( orderBy != null && ! check( orderBy, SELECT ) ) {
			String errMsg = "指定の orderBy は、指定できません。" + HybsSystem.CR
							+ "orderBy=[" + orderBy + "] "   + HybsSystem.CR
							+ StringUtil.array2csv( SELECT ) + HybsSystem.CR ;
//			throw new HybsSystemException( errMsg.toString() );
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】表示順を逆転するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * orderBy 属性で指定した表示順を、逆順にするかどうかを指定できます。
	 * 初期値は、false (昇順) です。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	flag 表示順を逆転するかどうか [逆順:true/正順:false]
	 * @see		#setOrderBy( String )
	 */
	public void setDesc( final String flag ) {
		desc = nval( getRequestParameter( flag ),desc );
	}

	/**
	 * 【TAG】検索結果のカラム列に追加するカラム名を、カンマ区切り文字で指定します。
	 *
	 * @og.tag
	 * デフォルトのカラム名、"LEVEL","FILE_TYPE","PARENT","NAME","LASTMODIFIED","FILE_LENGTH","RWH"
	 * 以外に、指定のカラム名を追加することが可能です。
	 * これは、ファイル検索結果以外の項目を追加して、データベースに書き込む場合に、利用できます。
	 * 並び順は、デフォルトカラムの後ろに、指定のカラムの順番で付きます。
	 * ここで追加したカラムには、値はセットされていません。よって、ソート対象のカラムにも指定できません。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	clms 検索結果のカラム列に追加するカラム名を、カンマ区切り文字で指定します。
	 * @see		#setUseUpdateClms( String )
	 */
	public void setAddClms( final String clms ) {
		String tmpClms = nval( getRequestParameter( clms ),null );

		if( tmpClms != null && tmpClms.length() > 0 ) {
			addClms = StringUtil.csv2Array( tmpClms );
		}
	}

	/**
	 * 【TAG】TO_PARENT、TO_NAMEカラムを追加するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * fileUpdateタグでは、ファイルのCOPYやMOVEが出来ますが、そのコピー先、移動先の
	 * ファイルを行ごとに指定する場合、TO_PARENT、TO_NAMEカラムという固定名のカラムが
	 * 必要です。
	 * これを、addClms 属性で指定する代わりに、この属性で、true をセットすることで、
	 * 自動的に追加されます。
	 * 初期値は、false (追加しない) です。
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) 新規追加
	 *
	 * @param	flag TO_PARENT、TO_NAMEカラムを追加するかどうか [true:追加する/false:追加しない]
	 * @see		#setAddClms( String )
	 */
	public void setUseUpdateClms( final String flag ) {
		useUpdateClms = nval( getRequestParameter( flag ),useUpdateClms );
	}

	/**
	 * 【TAG】ファイル名が、指定されたファイルタイプと一致した場合、スルー(選択)されます。
	 * @og.tag
	 * 大文字小文字は区別しません。
	 * ファイルタイプ は、DIR,FILE が指定できます。
	 * DIR は、ディレクトリのみ検索します。(階層下がりも行います)
	 * FILEは、ファイルのみ検索します。(階層下がりも行います)
	 * 引数が null の場合は、追加しません。(つまり、すべてスルーされます。)
	 *
	 * @og.rev 5.3.4.0 (2011/04/01) fileType メソッドで選択対象指定の追加
	 *
	 * @param    str 指定するファイルタイプ(DIR,FILE,null)
	 */
	public void setFileType( final String str ) {
		String tmp = nval( getRequestParameter( str ),fileType );
		if( tmp == null						||
			"DIR".equalsIgnoreCase( tmp )	||
			"FILE".equalsIgnoreCase( tmp ) ) {
				fileType = tmp;
		}
		else {
			// ファイルタイプに不正な値が設定された場合は、エラーになる。
			String errMsg = "この、fileType 属性には、DIR,FILE 以外は指定できません。["
						+ tmp + "]";
			throw new HybsSystemException( errMsg );
		}
	}

	/**
	 * 【TAG】from属性で指定された基準ファイル/フォルダ自体をリストに追加するかどうか[true/false]を指定します(初期値:true)。
	 * @og.tag
	 * 初期値はtrue(追加する)です。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) 新規作成
	 *
	 * @param    flg 基準ファイル/フォルダ自体をリストに追加するかどうか true:追加する/false:追加しない]
	 */
	public void setAddFrom( final String flg ) {
		addFrom = nval( getRequestParameter( flg ),addFrom );
	}

	/**
	 * FileFilterオブジェクトをセットします。
	 * これは、BODY 部に登録した、FileWhereタグによって設定された
	 * ファイルフィルターです。
	 *
	 * @param	filter	オブジェクト
	 */
	protected void setFileFilter( final FileFilter filter ) {
		this.filter = filter;
	}

	/**
	 * シリアライズ用のカスタムシリアライズ書き込みメソッド
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectOutputStreamオブジェクト
	 */
	private void writeObject( final ObjectOutputStream strm ) throws IOException {
		strm.defaultWriteObject();
	}

	/**
	 * シリアライズ用のカスタムシリアライズ読み込みメソッド
	 *
	 * ここでは、transient 宣言された内部変数の内、初期化が必要なフィールドのみ設定します。
	 *
	 * @og.rev 4.0.0.0 (2006/09/31) 新規追加
	 * @serialData
	 *
	 * @param	strm	ObjectInputStreamオブジェクト
	 * @see #release2()
	 */
	private void readObject( final ObjectInputStream strm ) throws IOException , ClassNotFoundException {
		strm.defaultReadObject();
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "multi"		,multi	)
				.println( "level"		,level	)
				.println( "from"		,from	)
				.fixForm().toString()
			+ HybsSystem.CR
			+ super.toString() ;
	}
}
