/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.db;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.opengion.fukurou.db.DBUtil;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.resource.LabelData;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * DBTableModelを継承した TableModelの編集設定による変換を行うための実装クラスです。
 *
 * このクラスでは、オブジェクト初期化後は、通常のDBTableModelと同じ振る舞いをします。
 * オブジェクト初期化時(createメソッド呼び出し時)に、検索結果オブジェクトから直接、編集設定に
 * 応じて変換されたDBTableModelを生成します。
 *
 * このような実装を行う理由は、メモリ使用量を節約するためです。
 * この編集設定では、集計機能を備えていますが、一旦DBTableModel作成後に集計処理を行うと、
 * メモリを大量に使用する恐れがあるため、検索結果オブジェクトから直接集計処理を行い、DBTableModelを
 * 生成しています。
 *
 * DBTableModel インターフェースは，データベースの検索結果(Resultset)をラップする
 * インターフェースとして使用して下さい。
 *
 * @og.rev 5.3.6.0 (2011/06/01) 新規作成
 * @og.group テーブル管理
 *
 * @version  5.0
 * @author   Hiroki Nakamura
 * @since    JDK6.0,
 */
public class DBTableModelEditor extends DBTableModelImpl {
	private static final String JS				= HybsSystem.JOINT_STRING;
	private static final DecimalFormat FORMAT	= new DecimalFormat( "0.#########" );

	private int rowCountColumn = -1;
	private DBEditConfig config;

	/**
	 * DBTableModel を設定し、このオブジェクトを初期化します。
	 *
	 * @param	result			検索結果オブジェクト
	 * @param	skipRowCount	読み飛ばし件数
	 * @param	maxRowCount		最大検索件数
	 * @param	resource		ResourceManagerオブジェクト
	 * @param	config			エディット設定オブジェクト
	 * @throws	SQLException
	 */
	public void create( final ResultSet result, final int skipRowCount, final int maxRowCount, final ResourceManager resource, final DBEditConfig config ) throws SQLException {
		if( result == null || config == null || resource == null ) {
			String msg = "DBTableModelまたは、DBEditConfigが設定されていません。";
			throw new HybsSystemException( msg );
		}

		this.config = config;

		/**********************************************************************
		 * 各パラメーターの初期化処理
		 **********************************************************************/
		ResultSetMetaData metaData	= result.getMetaData();
		int colCnt = metaData.getColumnCount();
		if( config.useGroup() || config.useSubTotal() || config.useTotal() || config.useGrandTotal() ) {
			rowCountColumn = colCnt;
			colCnt++;
		}
		init( colCnt );

		DBColumn[] dbColumn = new DBColumn[numberOfColumns];
		int[] types  = new int[numberOfColumns];
		boolean[] sumFilter = new boolean[numberOfColumns];
		boolean[] groupFilter = new boolean[numberOfColumns];
		boolean[] subTotalFilter = new boolean[numberOfColumns];
		boolean[] totalFilter = new boolean[numberOfColumns];
		for( int column=0; column<numberOfColumns; column++ ) {
			String name = null;
			if( column != rowCountColumn ) {
				name = (metaData.getColumnLabel(column+1)).toUpperCase(Locale.JAPAN);
				types[column] = metaData.getColumnType(column+1);
				dbColumn[column] = resource.getDBColumn( name );
				if( dbColumn[column] == null ) {
					LabelData labelData  = resource.getLabelData( name );
					dbColumn[column] = DBTableModelUtil.makeDBColumn( name,labelData,metaData,column,resource.getLang() );
				}
			}
			else {
				name = "rowCount";
				dbColumn[column] = resource.makeDBColumn( name );
			}

			setDBColumn( column,dbColumn[column] );

			sumFilter[column] = config.isSumClm( name );
			groupFilter[column] = config.isGroupClm( name );
			subTotalFilter[column] = config.isSubTotalClm( name );
			totalFilter[column] = config.isTotalClm( name );
		}

		/**********************************************************************
		 * 集計、ソート、合計処理
		 **********************************************************************/
		// 集計キーに基づく集計処理を行いデータを追加します。
		if( config.useGroup() ) {
			addGroupRows( result, types, skipRowCount, maxRowCount, sumFilter, groupFilter  );
		}
		// 通常と同じように結果カーソルからデータを読込みデータを追加します。
		else {
	 		// 5.5.2.4 (2012/05/16) int[] types は使われていないので、削除します。
//			addPlainRows( result, types, skipRowCount, maxRowCount );
			addPlainRows( result, skipRowCount, maxRowCount );
		}

		// ソート処理
		if( getRowCount() > 0 && config.useOrderBy() ) {
			sort();
		}

		// 小計・合計行を追加します。
		if( getRowCount() > 0 && !isOverflow()
			&& ( config.useSubTotal() || config.useTotal() || config.useGrandTotal() ) ) {
			addTotalRows( maxRowCount, resource, sumFilter, groupFilter, subTotalFilter, totalFilter );
		}
	}

	/**
	 * 集計キーの設定に基づき、DBTableModelの行を追加します。
	 * 内部的には、キーブレイクではなく、内部マップにより集計処理を行っているため、
	 * 集計キーが検索順により散在した場合でも1まとまりで集計されます。
	 *
	 * @og.rev 5.3.9.0 (2011/09/01) 値がNULLの場合にエラーになるバグを修正
	 * @og.rev 5.6.1.0 (2013/02/01) doubleをBigDecimalに
	 *
	 * @param result 検索結果オブジェクト
	 * @param types カラムタイプの配列
	 * @param skipRowCount 読み飛ばし件数
	 * @param maxRowCount 最大検索件数
	 * @param sumFilter 集計項目フィルター
	 * @param groupFilter グループキーフィルター
	 * @throws SQLException
	 */
	private void addGroupRows( final ResultSet result, final int[] types, final int skipRowCount, final int maxRowCount
								, final boolean[] sumFilter, final boolean[] groupFilter ) throws SQLException {
		int numberOfRows = 0;
		while( numberOfRows < skipRowCount && result.next() ) {
			// 注意 resultSet.next() を先に判定すると必ず１件読み飛ばしてしまう。
			numberOfRows ++ ;
		}
		numberOfRows = 0;

		Map<String,String[]> groupLinkedMap = new LinkedHashMap<String,String[]>();
		Map<String,Integer> groupCountMap = new HashMap<String,Integer>();
//		Map<String,double[]> sumMap = new HashMap<String,double[]>();
		Map<String,BigDecimal[]> sumMap = new HashMap<String,BigDecimal[]>(); // 5.6.1.0 (2013/02/01)
		while( numberOfRows < maxRowCount && result.next() ) {
			StringBuilder groupKey = new StringBuilder();
//			double[] sumVals = new double[config.getSumClmCount()];
			BigDecimal[] sumVals = new BigDecimal[config.getSumClmCount()]; // 5.6.1.0 (2013/02/01) 
			String[] groupVals = new String[config.getGroupClmCount()];
			int sc = 0;
			int gc = 0;
			for( int column=0; column<numberOfColumns; column++ ) {
				if( column != rowCountColumn ) {
					String val = DBUtil.getValue( result, column, types[column] );
					if( sumFilter[column] ) {
						// 5.3.9.0 (2011/09/01) 値がNULLの場合の対応漏れ
//						sumVals[sc++] = Double.valueOf( val );
						// sumVals[sc++] = ( val != null && val.length() > 0 ? Double.valueOf( val ) : 0 );
						sumVals[sc++] = ( val != null && val.length() > 0 ? new BigDecimal( val ) : new BigDecimal(0) ); // 5.6.1.0 (2013/02/01)
					}
					if( groupFilter[column] ) {
						groupVals[gc++] = val;
						groupKey.append( val ).append( JS );
					}
				}
			}

			String key = groupKey.toString();
			int groupCount = 0;
			if( groupLinkedMap.containsKey( key ) ) {
//				double[] eSumVals = sumMap.get( key );
				BigDecimal[] eSumVals = sumMap.get( key ); // 5.6.1.0 (2013/02/01)
				for( int i=0; i<config.getSumClmCount(); i++ ) {
//					sumVals[i] += eSumVals[i];
					sumVals[i] = sumVals[i] == null ? new BigDecimal(0) : sumVals[i].add( eSumVals[i] ); // 5.6.1.0 (2013/02/01)
				}
				sumMap.put( key, sumVals );
				groupCount = groupCountMap.get( key ).intValue() + 1;
			}
			else {
				groupLinkedMap.put( key, groupVals );
				groupCount = 1;
				numberOfRows++;
			}
			sumMap.put( key, sumVals );
			groupCountMap.put( key, Integer.valueOf( groupCount ) );
		}

		for( Map.Entry<String, String[]> entry : groupLinkedMap.entrySet() ) {
			String key = entry.getKey();
			addRow( groupFilter, entry.getValue(), groupCountMap.get( key ), sumFilter, sumMap.get( key ) );
		}

		// 最大件数が、超えた場合でかつ次のデータがある場合は、オーバーフロー
		if( numberOfRows >= maxRowCount && result.next() ) {
			setOverflow( true );
		}
	}

	/**
	 * 検索結果オブジェクトを順に読み取り、そのままDBTableModelの行を追加します。
	 *
	 * @og.rev 5.5.2.4 (2012/05/16) int[] types は使われていないので、削除します。
	 *
	 * @param result 検索結果オブジェクト
	 * @param skipRowCount 読み飛ばし件数
	 * @param maxRowCount 最大検索件数
	 * @throws	SQLException
	 */
//	private void addPlainRows( final ResultSet result, final int[] types, final int skipRowCount, final int maxRowCount ) throws SQLException {
	private void addPlainRows( final ResultSet result, final int skipRowCount, final int maxRowCount ) throws SQLException {
		int numberOfRows = 0;
		while( numberOfRows < skipRowCount && result.next() ) {
			// 注意 resultSet.next() を先に判定すると必ず１件読み飛ばしてしまう。
			numberOfRows ++ ;
		}
		numberOfRows = 0;

		while( numberOfRows < maxRowCount && result.next() ) {
			numberOfRows++ ;
			String[] columnValues = new String[numberOfColumns];
			for( int column=0; column<numberOfColumns; column++ ) {
				if( column != rowCountColumn ) {
					Object obj = result.getObject(column+1);
					columnValues[column] = ( obj == null ? "" : String.valueOf( obj ) );
				}
				else {
					columnValues[column] = "";
				}
			}
			addColumnValues( columnValues );
		}

		// 最大件数が、超えた場合でかつ次のデータがある場合は、オーバーフロー
		if( numberOfRows >= maxRowCount && result.next() ) {
			setOverflow( true );
		}
	}

	/**
	 * DBTableModelのソート処理を行います。
	 *
	 */
	private void sort() {
		// orderByClmsによる並び替え
		DBTableModelSorter temp = new DBTableModelSorter();
		temp.setModel( this );
		String[] oClms = StringUtil.csv2Array( config.getOrderByClms() );
		for( int i=oClms.length-1; i>=0; i-- ) {
			String oc = oClms[i];
			boolean ascending = true;
			if( oc.startsWith( "!" ) ) {
				oc = oc.substring( 1 );
				ascending = false;
			}
			int clmNo = getColumnNo( oc );
			temp.sortByColumn( clmNo, ascending );
		}
		this.data = temp.data;
		this.rowHeader = temp.rowHeader;
	}

	/**
	 * DBTableModelからデータを読み取り、エディット設定情報を元に合計行の追加処理を行います。
	 * 合計行の追加は、キーブレイクにより行われますので、同じキーが複数回出現した場合は、
	 * それぞれの行に対して、合計行が付加されます。
	 *
	 * @og.rev 5.3.7.0 (2011/07/01) 小計、合計行追加処理でオーバーフローフラグがセットされないバグを修正
	 * @og.rev 5.6.1.0 (2013/02/01) 誤差回避のため、doubleではなくdecimalで計算する
	 *
	 * @param	maxRowCount 最大検索件数
	 * @param	resource リソースマネージャー
	 * @param	sumFilter 集計項目フィルター
	 * @param	groupFilter グループキーフィルター
	 * @param	subTotalFilter 小計キーフィルター
	 * @param	totalFilter 合計キーフィルター
	 *
	 * @return	オーバーフローしたかどうか(最大件数が超えた場合でかつ次のデータがある場合は、true)
	 */
	private boolean addTotalRows( final int maxRowCount, final ResourceManager resource, final boolean[] sumFilter
									,  final boolean[] groupFilter, final boolean[] subTotalFilter,  final boolean[] totalFilter ) {

		String subTotalLabel = ( config.useSubTotal() ? resource.makeDBColumn( "EDIT_SUBTOTAL_VALUE" ).getLongLabel() : null );
		String totalLabel = ( config.useTotal() ? resource.makeDBColumn( "EDIT_TOTAL_VALUE" ).getLongLabel() : null );
		String grandTotalLabel = ( config.useGrandTotal() ? resource.makeDBColumn( "EDIT_GRANDTOTAL_VALUE" ).getLongLabel() : null );

		int numberOfRows = getRowCount();
		int sumClmCount = config.getSumClmCount();
//		double subTotalSum[] = new double[sumClmCount];
//		double totalSum[] = new double[sumClmCount];
//		double grandTotalSum[] = new double[sumClmCount];
		BigDecimal subTotalSum[] = new BigDecimal[sumClmCount]; // 5.6.1.0 (2013/02/01)
		BigDecimal totalSum[] = new BigDecimal[sumClmCount];
		BigDecimal grandTotalSum[] = new BigDecimal[sumClmCount];

		String lastSubTotalKey = null;
		String lastTotalKey = null;

		int subTotalCount = 0;
		int totalCount = 0;
		int grandTotalCount = 0;
		int rowCount =0;

		int tblIdx = 0;
		while( numberOfRows < maxRowCount && tblIdx < getRowCount() ) {
//			double[] sumVals = new double[sumClmCount];
			BigDecimal[] sumVals = new BigDecimal[sumClmCount]; // 5.6.1.0 (2013/02/01)
			StringBuilder groupKey = new StringBuilder();
			StringBuilder subTotalKey = new StringBuilder();
			StringBuilder totalKey = new StringBuilder();

			int sc = 0;
			for( int column=0; column<numberOfColumns; column++ ) {
				String val = getValue( tblIdx, column );
				if( groupFilter[column] )		{ groupKey.append( val ).append( JS ); }
//				if( sumFilter[column] )			{ sumVals[sc++] = ( val != null && val.length() > 0 ? Double.valueOf( val ) : 0 ); }
				if( sumFilter[column] )			{ sumVals[sc++] = ( val != null && val.length() > 0 ? new BigDecimal( val ) : new BigDecimal(0) ); } // 5.6.1.0 (2013/02/01)
				if( subTotalFilter[column] )	{ subTotalKey.append( val ).append( JS ); }
				if( totalFilter[column] )		{ totalKey.append( val ).append( JS ); }
				if( column == rowCountColumn )	{ rowCount = ( val != null && val.length() > 0 ? Integer.valueOf( val ) : 0 ); }
			}

			// 小計キーブレイク処理
			if( numberOfRows < maxRowCount && config.useSubTotal() && lastSubTotalKey != null && lastSubTotalKey.length() > 0
				&& !lastSubTotalKey.equals( subTotalKey.toString() ) ) {
				addRow( subTotalFilter, subTotalLabel, subTotalCount, sumFilter, subTotalSum, tblIdx );
//				subTotalSum = new double[sumClmCount];
				subTotalSum = new BigDecimal[sumClmCount]; // 5.6.1.0 (2013/02/01)
				subTotalCount = 0;
				numberOfRows++;
				tblIdx++;
			}

			// 合計キーブレイク処理
			if( numberOfRows < maxRowCount && config.useTotal() && lastTotalKey != null && lastTotalKey.length() > 0
				&& !lastTotalKey.equals( totalKey.toString() ) ) {
				addRow( totalFilter, totalLabel, totalCount, sumFilter, totalSum, tblIdx );
//				totalSum = new double[sumClmCount];
				totalSum = new BigDecimal[sumClmCount]; // 5.6.1.0 (2013/02/01)
				totalCount = 0;
				numberOfRows++;
				tblIdx++;
			}

			// 小計、合計、総合計単位に集計項目の合計値を加算します。
			for( int cnt=0; cnt<sumClmCount; cnt++ ) {
//				subTotalSum[cnt] += sumVals[cnt];
//				totalSum[cnt] += sumVals[cnt];
//				grandTotalSum[cnt] += sumVals[cnt];
				subTotalSum[cnt] = subTotalSum[cnt] == null ? new BigDecimal(0) : subTotalSum[cnt].add(sumVals[cnt]); // 5.6.1.0 (2013/02/01)
				totalSum[cnt] = totalSum[cnt] == null ? new BigDecimal(0) : totalSum[cnt].add(sumVals[cnt]);
				grandTotalSum[cnt] = grandTotalSum[cnt] == null ? new BigDecimal(0) : grandTotalSum[cnt].add(sumVals[cnt]);
			}

			lastSubTotalKey = subTotalKey.toString();
			lastTotalKey = totalKey.toString();

			// グループ集計時はグルーピングした行数を加算する。
			int gcnt = 1;
			if( config.useGroup() && rowCountColumn >= 0 && rowCount > 0 ) {
				gcnt = rowCount;
			}
			subTotalCount += gcnt;
			totalCount    += gcnt;
			grandTotalCount += gcnt;

			tblIdx++;
		}

		// 最大件数が、超えた場合でかつ次のデータがある場合は、オーバーフロー
		boolean isOverFlow = ( tblIdx < getRowCount() );

		// 小計キー最終行処理
		if( config.useSubTotal() && lastSubTotalKey != null ) {
			if( numberOfRows < maxRowCount ) {
				addRow( subTotalFilter, subTotalLabel, subTotalCount, sumFilter, subTotalSum, tblIdx );
				numberOfRows++;
				tblIdx++;
			}
			else {
				isOverFlow = true;
			}
		}

		// 合計キー最終行処理
		if( config.useTotal() && lastTotalKey != null ) {
			if( numberOfRows < maxRowCount ) {
				addRow( totalFilter, totalLabel, totalCount, sumFilter, totalSum, tblIdx );
				numberOfRows++;
				tblIdx++;
			}
			else {
				isOverFlow = true;
			}
		}

		// 総合計処理
		if( config.useGrandTotal() && numberOfRows > 0 ) {
			if( numberOfRows < maxRowCount ) {
				boolean[] grandTotalFilter = new boolean[numberOfColumns];
				// 総合計のラベル表示廃止
				// grandTotalFilter[0] = true;
				addRow( grandTotalFilter, grandTotalLabel, grandTotalCount, sumFilter, grandTotalSum, tblIdx );
				numberOfRows++;
				tblIdx++;
			}
			else {
				isOverFlow = true;
			}
		}

		if( isOverFlow ) {
			setOverflow( true );
		}

		return isOverFlow;
	}

	/**
	 * キーの値配列、集計値の配列を引数として、追加行を生成し、DBTableModelに追加します。
	 * キー、及び集計値がDBTableModel上のどのカラムに位置するかは、キーフィルタ、集計フィルタで指定します。
	 * 
	 * @og.rev 5.6.1.0 (2013/02/01) doubleをdecimalに
	 *
	 * @param keyFilter キーフィルタ
	 * @param keyVals キーの値配列
	 * @param keyCount 集計した行のカウント
	 * @param sumFilter 集計フィルタ
	 * @param sumVals 集計値配列
	 * @param aRow 挿入する行番号
	 */
//	private void addRow( final boolean[] keyFilter, final String[] keyVals, final int keyCount
//							, final boolean[] sumFilter, final double[] sumVals, final int aRow ) {
	private void addRow( final boolean[] keyFilter, final String[] keyVals, final int keyCount
			, final boolean[] sumFilter, final BigDecimal[] sumVals, final int aRow ) {
		String[] columnValues = new String[numberOfColumns];
		int sc = 0;
		int kc = 0;
		for( int column=0; column<numberOfColumns; column++ ) {
			String val = "";
			if( keyFilter[column] ) {
				val = keyVals[kc++];
			}
			if( sumFilter[column] ) {
				val = FORMAT.format( sumVals[sc++] );
			}
			if( column == rowCountColumn ) {
				val = String.valueOf( keyCount );
			}
			columnValues[column] = val;
		}

		if( aRow < 0 ) {
			addColumnValues( columnValues );
		}
		else {
			addValues( columnValues, aRow, false );
		}
	}

	/**
	 * キーの値配列、集計値の配列を引数として、追加行を生成し、DBTableModelに追加します。
	 * キー、及び集計値がDBTableModel上のどのカラムに位置するかは、キーフィルタ、集計フィルタで指定します。
	 * 
	 * @og.rev 5.6.1.0 (2013/02/01) doubleをbigDecimal
	 *
	 * @param keyFilter キーフィルタ
	 * @param keyVals キーの値配列
	 * @param keyCount 集計した行のカウント
	 * @param sumFilter 集計フィルタ
	 * @param sumVals 集計値配列
	 */
//	private void addRow( final boolean[] keyFilter, final String[] keyVals, final int keyCount
//							, final boolean[] sumFilter, final double[] sumVals ) {
	private void addRow( final boolean[] keyFilter, final String[] keyVals, final int keyCount
			, final boolean[] sumFilter, final BigDecimal[] sumVals ) {
		addRow( keyFilter, keyVals, keyCount, sumFilter, sumVals, -1 );
	}

	/**
	 * キーの値、集計値の配列を引数として、追加行を生成し、DBTableModelに追加します。
	 * キー、及び集計値がDBTableModel上のどのカラムに位置するかは、キーフィルタ、集計フィルタで指定します。
	 * 
	 * @og.rev 5.6.1.0 (2013/02/01) doubleをbigDecimalに
	 *
	 * @param keyFilter キーフィルタ
	 * @param keyVal キーの値
	 * @param keyCount 集計した行のカウント
	 * @param sumFilter 集計フィルタ
	 * @param sumVals 集計値配列
	 * @param aRow 挿入する行番号
	 */
//	private void addRow( final boolean[] keyFilter, final String keyVal, final int keyCount
//							, final boolean[] sumFilter, final double[] sumVals, final int aRow ) {
	private void addRow( final boolean[] keyFilter, final String keyVal, final int keyCount
			, final boolean[] sumFilter, final BigDecimal[] sumVals, final int aRow ) {
		List<String> tmp = new ArrayList<String>();
		for( int column=0; column<numberOfColumns; column++ ) {
			if( keyFilter[column] ) {
				tmp.add( keyVal );
			}
		}
//		addRow( keyFilter, tmp.toArray( new String[0] ), keyCount, sumFilter, sumVals, aRow );
		addRow( keyFilter, tmp.toArray( new String[tmp.size()] ), keyCount, sumFilter, sumVals, aRow );
	}
}
