/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.table;

import java.io.File;
import java.io.PrintWriter;

import org.opengion.fukurou.util.ErrorMessage;
import org.opengion.fukurou.util.FileUtil;
import org.opengion.fukurou.util.FixLengthData;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;
import org.opengion.hayabusa.db.AbstractTableFilter;
import org.opengion.hayabusa.db.DBTableModel;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * TableFilter_DBARG_OUT は、TableFilter インターフェースを継承した、DBTableModel 処理用の
 * 実装クラスです。
 *
 * ここでは、テーブル一覧の検索結果より、GF81 のARG変数チェックテーブルから
 * 必要な情報を取得し、ARG変数作成スクリプトを作成します。
 * 出力ファイルは、オブジェクト名＋".sql" という命名規則で作成します。
 * 検索では、(SYSTEM_ID,TBLSYU,OBJ_NAME,SEQNO,CLM,CLM_NAME,CLS_NAME,USE_LENGTH,MAX_LENGTH,TABLE_NAME)
 * の項目を取得する必要があります。
 *
 * パラメータは、tableFilterタグの keys, vals にそれぞれ記述するか、BODY 部にCSS形式で記述します。
 * 【パラメータ】
 *  {
 *       DIR       : 出力ディレクトリ ; 
 *  }
 *
 * @og.formSample
 * ●形式：
 *      select SYSTEM_ID,TBLSYU,OBJ_NAME,SEQNO,CLM,CLM_NAME,CLS_NAME,USE_LENGTH,MAX_LENGTH,TABLE_NAME from GF81
 *      ① &lt;og:tableFilter classId="DBARG_OUT" keys="DIR" vals="{&#064;BASE_DIR}/sql/07_ARG" /&gt;
 *
 *      ② &lt;og:tableFilter classId="DBARG_OUT" &gt;
 *               {
 *                   DIR       : {&#064;BASE_DIR}/sql/07_ARG ;
 *               }
 *         &lt;/og:tableFilter&gt;
 *
 * @og.rev 5.6.6.0 (2013/07/05) keys の整合性チェックを追加
 *
 * @version  0.9.0  2000/10/17
 * @author   Kazuhiko Hasegawa
 * @since    JDK1.1,
 */
public class TableFilter_DBARG_OUT extends AbstractTableFilter {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.6.6.0 (2013/07/05)" ;

	// 5.6.6.0 (2013/07/05) keys の整合性チェックを行います。
	// keysMap は、AbstractTableFilter で、protected static final 定義しておきます。
	static {
		keysMap.put( "DIR"			, "出力ファイルの基準フォルダ(必須)"	);
	}

	// 5.6.6.0 (2013/07/05) 検索項目を増やしておきます。
//	private static final String[] KEYS = new String[] { "OBJ_NAME","CLM","CLS_NAME","USE_LENGTH","CLM_NAME" };
	private static final String[] KEYS = new String[] { "SYSTEM_ID","TBLSYU","OBJ_NAME","SEQNO","CLM","CLM_NAME","CLS_NAME","USE_LENGTH","MAX_LENGTH","TABLE_NAME" };

	private static final int SYSTEM_ID		= 0;
	private static final int TBLSYU			= 1;
	private static final int OBJ_NAME		= 2;
	private static final int SEQNO			= 3;
	private static final int CLM			= 4;
	private static final int CLM_NAME		= 5;
	private static final int CLS_NAME		= 6;
	private static final int USE_LENGTH		= 7;
	private static final int MAX_LENGTH		= 8;
	private static final int TABLE_NAME		= 9;

	private static final String ENCODE = "Windows-31J" ;
//	private static final String CR     = HybsSystem.CR ;	// 5.1.1.0 (2009/12/01) CR 定義をAbstractTableFilterで行う。

	// 5.6.6.0 (2013/07/05) ヘッダー部作成用
	private static final String CMNT  = "************************************************************************" ;

	private static final int X = FixLengthData.X ;		// type 定数
	private static final int S = FixLengthData.S ;		// type 定数
	private static final int K = FixLengthData.K ;		// type 定数
	private static final int T = FixLengthData.T ;		// addLen 定数
	private static final int T3= FixLengthData.T3 ;		// addLen 定数

	private ResourceManager resource = null;			// 5.6.6.0 (2013/07/05) ヘッダー用のリソース

	/**
	 * DBTableModel処理を実行します。
	 *
	 * @og.rev 4.0.0.0 (2007/11/28) メソッドの戻り値をチェックします。
	 * @og.rev 5.5.2.6 (2012/05/25) protected変数を、private化したため、getterメソッドで取得するように変更
	 *
	 * @return 処理結果のDBTableModel
	 */
	public DBTableModel execute() {
		DBTableModel table = getDBTableModel();		// 5.5.2.6 (2012/05/25) インターフェースにgetterメソッド追加

		int[]  clmNo = getTableColumnNo( KEYS );

		File dir = new File( getValue( "DIR" ) );
		if( ! dir.exists() && !dir.mkdirs() ) {
			String errMsg = "所定のフォルダが作成できませんでした。[" + dir + "]" ;
			// 4.3.4.4 (2009/01/01)
			throw new HybsSystemException( errMsg );
		}

		// カンマ,カラム,クラス,(,桁数,),記号(--),表示順,名称
	 	// 5.6.6.0 (2013/07/05) FixLengthData の簡易コンストラクタを使用
//		FixLengthData fixData = new FixLengthData(6);

	 	// 5.6.6.0 (2013/07/05) カラムの表示順も出力します。
//		int[] addLen = new int[] { 1,5,5,1 };	// 各データ間のスペース
//		int[] type   = new int[] { X,X,X,K };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
//		カンマ,カラム,クラス,(,桁数,),記号(--),表示順,名称
		int[] addLen = new int[] { 0,T,0,0,0,T3,0,1,0 };	// 各データ間のスペース
		int[] type   = new int[] { X,X,X,X,S,X, X,S,K };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
//		fixData.setAddLength( addLen );
//		fixData.setType( type );
		FixLengthData fixData = new FixLengthData( addLen,type );

		String[] data  = null;
		String oldObjName  = null;

		int rowCnt = table.getRowCount();
		for( int row=0; row<rowCnt; row++ ) {
			String objName = null;
			try {
				data = table.getValues( row );
				objName = data[clmNo[OBJ_NAME]];

				// 5.6.6.0 (2013/07/05) 初めに一度だけ作成しておきます。
				if( resource == null ) {
					String systemId = data[clmNo[SYSTEM_ID]];
					resource = ResourceFactory.newInstance( systemId,"ja",false );
				}

				boolean blk = ! objName.equals( oldObjName ) ;
				if( row > 0 && blk ) {
					PrintWriter writer = FileUtil.getPrintWriter( new File( dir,oldObjName + ".sql" ),ENCODE );
					writer.println( makeHeadLine( oldObjName ) );
					writer.print( fixData.getAllFixData() );
					writer.println( makeEndLine( oldObjName ) );
					writer.close();
					fixData.clear();
				}

				String[] outData = makeLineList( clmNo,data,blk );
				fixData.addListData( outData );

				oldObjName = objName ;
			}
			catch( RuntimeException ex ) {
				ErrorMessage errMessage = makeErrorMessage( "TableFilter_DBARG_OUT Error",ErrorMessage.NG );
				errMessage.addMessage( row+1,ErrorMessage.NG,"ARG",ex.getMessage() );
				errMessage.addMessage( row+1,ErrorMessage.NG,"ARG",StringUtil.array2csv( data ) );
				errMessage.addMessage( row+1,ErrorMessage.NG,"ARG","OBJ_NAME=[" + objName + "]" );
			}
		}

		// 常に、一回り遅れてデータ出力している為、最後のデータを出力しておく必要がある。
		PrintWriter writer = FileUtil.getPrintWriter( new File( dir,oldObjName + ".sql" ),ENCODE );
		writer.println( makeHeadLine( oldObjName ) );
		writer.print( fixData.getAllFixData() );
		writer.println( makeEndLine( oldObjName ) );
		writer.close();
		fixData.clear();

		return table;
	}

	/**
	 * ヘッダーとして使用する文字列を作成します。
	 *
	 * @og.rev 5.6.6.0 (2013/07/05) ヘッダーとして各種情報をセットします。
	 *
	 * @param	objName	オブジェクト名
	 *
	 * @return	ヘッダーとして使用する文字列
	 */
	private String makeHeadLine( final String objName ) {
		// objName から、ARG を取り除いた文字列が、画面IDになるはず。リソースから検索
		String gamenId = objName.substring( 0,objName.length()-3 );		// 3文字は、"ARG" 分
		String gamenNm = resource.getLabel( gamenId ) ;

		String LINE1 = objName ;
		String LINE2 = gamenId + " ( " + gamenNm + " )" ;
		String LINE3 = "Created : " + HybsSystem.getDate() ;

		int[] addLen = new int[] { 0,0,0 };	// 各データ間のスペース
		int[] type   = new int[] { X,K,X };	// 各データの種別 X:半角 S:空白前埋め K:全角混在
		FixLengthData fixData = new FixLengthData( addLen,type );

		String[][] outData = new String[][] {
			{ "/**",	CMNT ,	"**/" },
			{ "/* ",	LINE1,	" */" },
			{ "/* ",	LINE2,	" */" },
			{ "/* ",	LINE3,	" */" },
			{ "/**",	CMNT ,	"**/" },
		};

		// 5.6.6.0 (2013/07/05) 簡易メソッドを利用
		fixData.addAllListData( outData );

//		for( int i=0; i<outData.length; i++ ) {
//			fixData.addListData( outData[i] );
//		}

//		StringBuilder buf = new StringBuilder();
//		for( int i=0; i<outData.length; i++ ) {
//			buf.append( fixData.getFixData( i ) ).append( CR );
//		}

		StringBuilder buf = new StringBuilder();
		fixData.getAllFixData( buf );

		buf.append( "DROP TYPE "              ).append( objName ).append( "_ARRAY;"    ).append( CR );
		buf.append( "CREATE OR REPLACE TYPE " ).append( objName ).append( " AS OBJECT" ).append( CR );
		buf.append( " (" );

		return buf.toString() ;
	}

	/**
	 * 各行に相当する文字列の配列を作成します。
	 * カンマ,カラム,クラス,(,桁数,),記号(--),表示順,名称
	 *
	 * @og.rev 5.5.1.9 (2012/04/18) useLen.length=0対応
	 * @og.rev 5.6.6.0 (2013/07/05) カラムの表示順も出力します。
	 *
	 * @param	clmNo	カラム番号配列
	 * @param	data	１行分の入力データ配列
	 * @param	first	最初の行かどうか[true:最初/false:それ以降]
	 *
	 * @return	１行分に相当する結果配列(カンマ,カラム,クラス(桁数),コメント記号,表示順,名称)
	 */
	private String[] makeLineList( final int[] clmNo,final String[] data,final boolean first ) {
		// カンマ,カラム,クラス(桁数),記号(--),表示順,名称
//		String[] outData = new String[4];
		String[] outData = new String[9];					// 5.6.6.0 (2013/07/05) カラムの表示順追加

		outData[0] = ( first ) ? "   " : " , " ;			// 0:カンマ
		outData[1] = data[clmNo[CLM]] ;						// 1:カラム

		String clsName = data[clmNo[CLS_NAME]];
		if( clsName.startsWith( "CLOB" ) || clsName.startsWith( "DATE" ) ) {
			data[clmNo[USE_LENGTH]] = null;
		}
		String useLen = data[clmNo[USE_LENGTH]];
//		if( useLen != null && ! useLen.equals( "0" ) ) {
		if( useLen != null && ! useLen.equals( "0" ) && useLen.length() > 0 ) { // 5.5.1.9 (2012/04/19)
//			outData[2] = clsName + "(" + useLen + ")" ;		// 2:クラス(桁数)
			outData[2] = clsName ;							// 2:クラス
			outData[3] = " ( " ;							// 3:(
			outData[4] = useLen ;							// 4:桁数
			outData[5] = " )" ;								// 5:)
		}
		else {
			outData[2] = clsName ;							// NUMBER型桁数オープン対応。ARGで使えるか不明。
		}

		String seqno  = data[clmNo[SEQNO]] ;				// 表示順
		String nameJA = data[clmNo[CLM_NAME]] ;				// 名称
		if( nameJA != null ) {
			outData[6] = "-- " ;							// 3:コメント記号
			outData[7] = seqno ;							// 4:表示順
			outData[8] = nameJA ;							// 5:名称
		}

		return outData ;
	}

	/**
	 * 最後の行に相当する文字列を作成します。
	 *
	 * @param	objName	オブジェクト名
	 *
	 * @return	最後の行に相当する文字列
	 */
	private String makeEndLine( final String objName ) {

		StringBuilder buf = new StringBuilder();
		buf.append( "  ) ;" ).append( CR );
		buf.append( "/"     ).append( CR );
		buf.append( CR );
		buf.append( "CREATE OR REPLACE TYPE " ).append( objName ).append( "_ARRAY" );
		buf.append( " AS VARRAY(1000) OF "    ).append( objName ).append( ";"      ).append( CR );
		buf.append( "/"     );

		return buf.toString() ;
	}
}
