/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.common.HybsSystemException;

import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.hayabusa.db.DBColumnConfig;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager;

/**
 * 動的カラムのデータを編集する場合に使用するエディタークラスです。
 *
 * 値の文字列より、DBColumn を作成して、そのエディター文字列を動的に作成します。
 *
 * このエディタはeventColumnに対応していません。
 *
 * カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 * @og.group データ編集
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Editor_COLUMN extends AbstractEditor {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private final String	lang ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 *
	 */
	public Editor_COLUMN() {
		// 4.3.4.4 (2009/01/01)
//		super();
		lang = null;
	}

	/**
	 * デフォルトコンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	private Editor_COLUMN( final DBColumn clm ) {
	//	super( clm );
		name = clm.getName();
		lang = clm.getLang();
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_COLUMN( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 3.5.2.0 (2003/10/20) 値で動的カラムが作成できない場合に、エラーメッセージを追加。
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		ResourceManager resource = ResourceFactory.newInstance( lang ) ;
		DBColumn dbColumn = resource.getDBColumn( value );

		if( dbColumn == null ) {
			String errMsg = "指定のカラムの値[]で、動的カラムは作成できません。"
						+ HybsSystem.CR
						+ "  name=[" + name + "]"
						+ "  value=[" + value + "]";
			throw new HybsSystemException( errMsg );
		}

		return dbColumn.getEditorValue( null );
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @param	row   行番号
	 * @param	value 入力値
	 *
	 * @return	データ表示/編集用の文字列
	 */
	@Override
	public String getValue( final int row,final String value ) {
		ResourceManager resource = ResourceFactory.newInstance( lang ) ;
		DBColumn dbColumn = resource.getDBColumn( value );

		String name2 = name + HybsSystem.JOINT_STRING + row ;
		final DBColumnConfig config ;
		if( dbColumn == null ) {
			config = new DBColumnConfig( name2 ) ;
		}
		else {
			config = dbColumn.getConfig() ;
			config.setName( name2 );
		}
		DBColumn newColmn = new DBColumn( config );

		return newColmn.getEditorValue( null );
	}
}
