/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.db.AbstractRenderer;
import org.opengion.hayabusa.db.CellRenderer;
import org.opengion.hayabusa.db.DBColumn;

/**
 * MONEY レンデラーは、カラムのデータを金額表示する場合に使用するクラスです。
 *
 * マイナス時の表示は、id="minus" をキーに CSSファイルで指定しています。
 * 通貨は、標準では、￥ですが、値:記号 という形式で指定すれば、各値ごとに
 * 通貨を指定できます。(ただし、通貨変換は、サポートしていません。)
 * 負数の場合はspanタグclass="minus"を付けて出力します。
 *
 *  カラムの表示に必要な属性は, DBColumn オブジェクト より取り出します。
 * このクラスは、DBColumn オブジェクト毎に１つ作成されます。
 *
 * @og.group データ表示
 * @og.rev 5.4.3.6 (2012/01/19) コメント修正
 *
 * @version  4.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class Renderer_MONEY extends AbstractRenderer {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "4.0.0 (2005/08/31)" ;

	private static final CellRenderer[] dbCell = {
							new Renderer_MONEY(),
							new Renderer_MONEY(1),
							new Renderer_MONEY(2)
	};

	private final int	 minFraction;
	private final String noDisplayVal ;		// 5.6.2.3 (2013/03/22)

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.3.0.0 (2003/06/23) 初期値設定追加。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数初期化
	 *
	 */
	public Renderer_MONEY() {
		minFraction  = 0;
		noDisplayVal = null;		// 5.5.1.0 (2012/04/03)
	}

	/**
	 * コンストラクター。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.3.0.0 (2003/06/23) 初期値設定追加。NumberFormatクラスは、廃止します。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
	 *
	 * @param	size  	小数点
	 */
	private Renderer_MONEY( final int size ) {
		minFraction  = size ;
		noDisplayVal = null;		// 5.5.1.0 (2012/04/03)
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 3.1.1.1 (2003/04/03) 各オブジェクトから自分のインスタンスを返すファクトリメソッドを追加。
	 * @og.rev 3.1.2.1 (2003/04/10) synchronized を、削除します。
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellRendererオブジェクト
	 */
	public CellRenderer newInstance( final DBColumn clm ) {
		int size = clm.getSizeY();
		if( size < dbCell.length ) { return dbCell[size]; }
		return new Renderer_MONEY( size );
	}

	/**
	 * データの表示用文字列を返します。
	 *
	 * 引数の値が、『数字型文字列:通貨』という値を渡すことで、通貨を
	 * 頭につけて通貨ごとに異なる値を表示させることができる。
	 *
	 * @og.rev 3.1.0.0 (2003/03/20) 内部に、DBColumn オブジェクトをキープしないように変更
	 * @og.rev 3.3.0.0 (2003/06/23) NumberFormatクラスは、廃止します。
	 * @og.rev 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
	 *
	 * @param	value 入力値
	 *
	 * @return	データの表示用文字列
	 */
	@Override
	public String getValue( final String value ) {
		// 5.6.2.3 (2013/03/22) noDisplayVal 変数追加
		if( noDisplayVal != null && noDisplayVal.equalsIgnoreCase( value ) ) { return "" ; }

		String rtn ;
		if( value == null || (value.trim()).length() == 0 ) {
			rtn = "0" ;
		}
		else {
			rtn = value;
		}

		boolean minus = false ;
		if( rtn.charAt( 0 ) == '-' ) {
			minus = true;
			rtn   = rtn.substring( 1 );
		}

		String tuuka = "￥";
		int taniPos = rtn.indexOf( ':' );
		if( taniPos >= 0 ) {
			tuuka = rtn.substring( taniPos+1 );
			rtn   = rtn.substring( 0,taniPos );
		}

		rtn = tuuka + StringUtil.numberFormat( rtn,minFraction ) + "-";

		if( minus ) {
			rtn = "<span class=\"minus\">-" + rtn + "</span>";
		}
		return rtn;
	}
}
