/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.fukurou.util.Cleanable;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.hayabusa.common.HybsSystem ;
import org.opengion.hayabusa.common.SystemManager ;
import org.opengion.hayabusa.common.HybsSystemException ;
import org.opengion.hayabusa.common.BuildNumber ;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.hayabusa.resource.ResourceManager ;
import org.opengion.hayabusa.resource.UserInfoFactory ;
import org.opengion.hayabusa.resource.UserInfo ;
import org.opengion.hayabusa.resource.GUIInfo ;

import static org.opengion.fukurou.util.StringUtil.nval ;

import javax.servlet.http.HttpSession ;
import javax.servlet.http.HttpServletRequest ;
import javax.servlet.http.HttpServletResponse ;
import javax.servlet.ServletException;
import javax.servlet.jsp.tagext.Tag ;

import java.io.UnsupportedEncodingException;
import java.io.IOException;
import java.io.File;
import java.util.Map;
import java.util.HashMap;


/**
 * HTMLファイルの head タグに相当する機能を付加します。
 * レイアウトを構築する場合に、HTMLファイルの構造に準拠した形式で記述できるように
 * なります。JavaScript や body タグを 各JSPファイル内で独自に設定できるように
 * なる為、カスタマイズが容易になります。
 *
 * 機能的には、旧バージョンの
 *    common/session_init.jsp
 *    common/HTML.jsp
 *    &lt;og:responseHeader cacheControl = "max-age=0" /&gt;
 * を組み込んだ、common/jsp_init.jsp の代わりに使用できるタグになります。
 *
 * 属性に、下記の値を設定できます。それ以外は、BODY部に直接記述することで追加できます。
 * BODY部、および、属性の引数には、{&64;XXXX}が使用できます。
 * 例えば、{&64;SYS.CONTEXT_NAME}を使用すれば、相対アドレスでの指定ではなく、
 * 絶対アドレスで指定可能になるため、ブラウザのキャッシュが利用できます。
 *
 * @og.formSample
 * ●形式：&lt;og:head /&gt;
 * ●body：あり
 *
 *
 * ●Tag定義：
 *   &lt;og:head
 *       title              【TAG】HTML の タイトルを設定します(初期値:session に"GUI_TITLE" キーで登録されている値)
 *       headType           【TAG】ヘッダータイプを設定します(初期値:null)
 *       useScrollBar       【TAG】ScrollBar.js JavaScript を使用するかどうか[true/false]を指定します(初期値:true)
 *       useTabstrip        【TAG】tabstrip.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)
 *       useGantt           【TAG】option/gantt.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)
 *       useGantt2          【TAG】option/adjustGantTable.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)
 *       useTreeBOM         【TAG】option/treeBOM.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)
 *       useGuest           【TAG】ゲストユーザを利用するかどうか[true/false]を指定します(初期値false)
 *       useAjaxTreeTable   【TAG】Ajaxを利用したツリー形式のテーブル表示を使用するかを指定します
 *       noTransition       【TAG】画面遷移を行わない形式の登録方法を使用するかを指定します
 *       useTabLink         【TAG】og:tabLinkタグで使用される common/tagLink.js をインクルードするかどうか[true/false]を指定します(初期値:false)
 *       useRealTimeCheck   【TAG】カラムチェックをリアルタイムで行うかどうか[true/false]を指定します(初期値:true)
 *       useIE7Header       【TAG】IE8でIE7エミュレートモードで動作させるタグを出力するかどうか[true/false]を指定します (初期値:USE_IE7_HEADER[=true])
 *       useAjaxSubmit      【TAG】フォームのSubmit処理をJavaScriptを利用して行うか(ポップアップエラー表示)
 *       useFocusHilight    【TAG】フォーカスが当っている入力ボックスの色を変更するかどうか[true/false]を指定します(初期値:true)
 *       addEvent			【TAG】javaScript を初期ロードし、起動するメソッドを指定します(初期値:null)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   &gt;   ... Body ...
 *   &lt;/og:head&gt;
 * ●使用例
 *       &lt;og:head
 *          headType        = "[xml|init|query|result|menu]"
 *          title           = "HEAD タグの TITLE属性"
 *      //  useTimeSet      = "[true/false]"    timeSet.js , timeCheck.js
 *      //  useToggleFrame  = "[true/false]"    toggleFrame.js      5.2.1.0 (2010/10/01) 廃止
 *      //  useInputFocus   = "[true/false]"    inputFocus.js       4.3.5.0 (2009/02/01) 廃止
 *      //  useHilightRow   = "[true/false]"    hilightRow.js       5.2.1.0 (2010/10/01) 廃止
 *          useScrollBar    = "[true/false]"    scrollBar2.js
 *      //  usePrintLine    = "[true/false]"    printLine.js        5.2.1.0 (2010/10/01) 廃止
 *          useTabstrip     = "[false/true]"    tabstrip.js
 *          useTabLink      = "[false/true]"    tabLink.js
 *          useGantt        = "[false/true]"    option/gantt.js
 *          useGantt2       = "[false/true]"    option/adjustGantTable.js  5.6.2.1 (2013/03/08) 新規追加
 *          useTreeBOM      = "[false/true]"    option/treeBOM.js
 *          useAjaxTreeTable= "[false/true]"    option/ajaxTreeTable.js
 *          noTransition    = "[false/true]"    option/noTransition.js
 *          useTabLink      = "[false/true]"    tabLink.js
 *      //  useRealTimeCheck= "[true/false]"    realTimeCheck.js    5.2.1.0 (2010/10/01) 廃止
 *      //  useAutoConvert  = "[true/false]"    autoConvert.js      5.2.1.0 (2010/10/01) 廃止
 *      //  useFocusHilight = "[true/false]"    focusHilight.js     5.2.1.0 (2010/10/01) 廃止
 *      //  useFocusFirstElm= "[true/false]"    focusFirstElm.js    5.2.1.0 (2010/10/01) 廃止
 *      //  useAdjustFrameHeight = "[true/false]" adjustFrameHeight 5.2.1.0 (2010/10/01) 廃止
 *          // 5.4.5.0 (2012/03/01) 追加(復活)
 *          useRealTimeCheck= "[true/false]"    realTimeCheck
 *          useFocusHilight = "[true/false]"    focusHilight
 *       &gt; ... Body ...
 *       &lt;/og:head&gt;
 *
 * @og.rev 4.0.0 (2007/04/16) 新規作成
 * @og.group その他
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class HeadTag extends CommonTagSupport {
	//* このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "5.2.1.0 (2010/10/01)" ;

	private static final long serialVersionUID = 5210 ;	// 5.2.1.0 (2010/10/01)

	private static final String CR		 = HybsSystem.CR;
	private static final String JSP_ADRS = "/" + HybsSystem.getContextName() + "/jsp/" ;
	private static final String JSP_DIR  = HybsSystem.sys( "REAL_PATH" ) + "jsp/" ;			// 5.2.1.0 (2010/10/01)

	// headType が null の場合、使用されるJavaScript,stylesheet (XMLタイプ以外すべて)
	private static final String DEFAULT_CSS		= "common/default.css";
	private static final String CALENDAR_CSS	= "common/calendar.css";
	private static final String CUSTOM_CSS		= "custom/custom.css";

	// init,query,result,menu 両タイプで使用されるJavaScript,stylesheet (null でなく、XML でないすべて)
	private static final String JQUERY_JS		= "common/jquery/jquery.js";
	private static final String JQUERY_OG_JS	= "common/jqog.js";
	private static final String DEFAULT_JS		= "common/default.js";					// 5.2.1.0 (2010/10/01)
//		private static final String DEFAULT_JS		= "common/default.js";
//		private static final String INPUTMENU_JS	= "common/inputMenu.js";
//		private static final String OGPOPUP_JS		= "common/ogPopup.js";
	private static final String TABLINK_JS		= "common/option/tabLink.js";			// useTabLink

	// query,result 両タイプで使用されるJavaScript
	private static final String EVENT_SCRIPT_JS		= "common/eventScript.js";			// 5.2.1.0 (2010/10/01)
//		private static final String FOCUS_HILIGHT_JS	= "common/focusHilight.js";			// 4.3.5.0 (2009/02/01)
//		private static final String AUTO_CONVERT_JS		= "common/autoConvert.js";			// 4.3.5.0 (2009/02/01)
//		private static final String EVENT_COLUMN_JS		= "common/eventColumn.js";			// 4.3.6.0 (2009/04/01)
//		private static final String REAL_TIME_CHECK_JS	= "common/realTimeCheck.js";		// 4.3.5.0 (2009/02/01)
//		private static final String BIND_EVENTS_JS		= "common/bindEvents.js";			// 5.2.0.0 (2010/09/01)
	private static final String REAL_TIME_CHECK_FLG	= "USE_REAL_TIME_CHECK";	// 5.4.5.0 (2012/03/01)
	private static final String FOCUS_HILIGHT_FLG	= "USE_FOCUS_HILIGHT";	// 5.4.5.0 (2012/03/01)

	private static final String JQUERY_FORM_JS		= "common/jquery/jquery.form.js";	// noTransition || useAjaxSubmit
	private static final String JQUERY_DRAG_JS		= "common/jquery/jqDnR.js";			// noTransition || useAjaxSubmit
	private static final String JQUERY_MODAL_JS		= "common/option/jqModal.js";		// noTransition || useAjaxSubmit
	private static final String JQUERY_MODAL_CSS	= "common/option/jqModal.css";		// noTransition || useAjaxSubmit

	// query タイプで使用されるJavaScript
	private static final String QUERY_SCRIPT_JS			= "common/queryScript.js";		// 5.2.1.0 (2010/10/01)
//	private static final String QUERYPRE_SCRIPT_JS		= "common/queryScriptPre.js";	// 5.6.0.3 (2013/01/24)
	private static final String QUERYCONV_SCRIPT_JS		= "common/queryConvert.js";		// 5.6.2.3 (2013/03/22) QUERYPRE_SCRIPT_JS   → QUERYCONV_SCRIPT_JS に変更
//		private static final String TIME_SET_JS				= "common/timeSet.js";
//		private static final String TOGGLE_FRAME_JS			= "common/toggleFrame.js";
//		private static final String ADJUST_FRAME_HEIGHT_JS	= "common/adjustFrameHeightEvent.js";
//		private static final String FOCUS_FIRST_ELM			= "common/focusFirstElm.js";

	// result タイプで使用されるJavaScript,stylesheet
	private static final String SCROLL_BAR_JS		= HybsSystem.sys( "SCROLL_BAR_TYPE" ) + ".js";	// 4.3.5.0 (2009/02/01)
	private static final String SCROLL_BAR_CSS		= HybsSystem.sys( "SCROLL_BAR_TYPE" ) + ".css";	// 4.3.5.0 (2009/02/01)

	private static final String RESULT_SCRIPT_JS	= "common/resultScript.js";		// 5.2.1.0 (2010/10/01)
//		private static final String HILIGHT_ROW_JS		= "common/hilightRow.js";
//		private static final String BULK_SET_JS			= "common/bulkSet.js";		// 5.2.1.0 (2010/10/01)
//		private static final String PRINT_LINE_JS		= "common/printLine.js";
//		private static final String TIME_CHECK_JS		= "common/timeCheck.js";
//		private static final String ROW_COUNT_REPLACE	= "common/rowCountReplace.js";			// 4.3.7.6 (2009/07/15)

	private static final String GANTT_JS			= "common/option/gantt.js";				// useGantt
	private static final String TREEBOM_JS			= "common/option/treeBOM.js";			// useTreeBOM
	private static final String AJAX_TREETABLE_JS	= "common/option/ajaxTreeTable.js";		// useAjaxTreeTable
	private static final String AJAX_HYBS_FORM		= "common/option/ajaxHybsForm.js";		// noTransition || useAjaxSubmit
	private static final String NO_TRANSITION_JS	= "common/option/noTransition.js";		// noTransition
	private static final String AJAX_SUBMIT_JS		= "common/option/ajaxSubmit.js";		// useAjaxSubmit

	// 5.6.2.1 (2013/03/08) useGantt2="true" で組み込む script,css 群
	private static final String GANTT2_JS			= "common/option/adjustGantTable.js";			// 5.6.2.1 (2013/03/08) useGantt2
	private static final String GANTT2_CSS			= "common/option/adjustGantTable.css";			// 5.6.2.1 (2013/03/08) useGantt2
//	private static final String GANTT2_SCRIPT		= "addEvent( this, \"load\", adjustGant );";	// 5.6.2.1 (2013/03/08) useGantt2   5.6.3.0 (2013/04/01) 廃止

	// 4.2.1.0 (2008/04/16) htmlタグにクラスを付けるため、HEAD1と2に分離
	// class名は"headtype"+headtype (IDの先頭は数字では駄目なのでheadtypeを付ける)
	private static final String HEAD1 =
		"<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Frameset//EN\""	+ CR
		+ "  \"http://www.w3.org/TR/html4/frameset.dtd\">"				+ CR
		+ "<html class=\"headtype-";

	private static final String HEAD2 =
		"\">"				+ CR
		+ "<head>"			+ CR
		+ "  <meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />" + CR
		+ "  <meta http-equiv=\"Content-Style-Type\" content=\"text/css\" />"			+ CR
		+ "  <meta http-equiv=\"Content-Script-Type\" content=\"text/javascript\" />"	+ CR;

	private static final String XML_HEAD =
		"<?xml version=\"1.0\" encoding=\"UTF-8\" ?>"	+ CR;

	private static final String TAB_HEAD =
		"<?xml version=\"1.0\" encoding=\"UTF-8\" ?>"	+ CR
		+ "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\""	+ CR
		+ "    \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">"	+ CR
		+ "<html xmlns=\"http://www.w3.org/1999/xhtml\" xmlns:mp xmlns:ts >"	+ CR
		+ "<head>"																+ CR
		+ "  <meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />" + CR
		+ "  <meta http-equiv=\"Content-Style-Type\" content=\"text/css\" />"			+ CR
		+ "  <meta http-equiv=\"Content-Script-Type\" content=\"text/javascript\" />"	+ CR
		+ "  <?IMPORT NAMESPACE=\"mp\" IMPLEMENTATION=\""
		+ JSP_ADRS + "common/option/multipage.htc\" />"				+ CR
		+ "  <?IMPORT NAMESPACE=\"ts\" IMPLEMENTATION=\""
		+ JSP_ADRS + "common/option/tabstrip.htc\" />"					+ CR;

	// 4.3.7.0 (2009/06/01) IE8対応。IE7用Metaタグ追加
	private static final String IE7_HEAD =
		"  <meta http-equiv=\"X-UA-Compatible\" content=\"IE=EmulateIE7\" />" + CR;

	// 5.5.8.0 (2012/11/01) タブレット対応
	private static final String MOBILE_HEAD =
		"  <meta content=\"width=device-width, initial-scale=1\" name=\"viewport\"/>" + CR;
	
	// 5.2.1.0 (2010/10/01) 予め合成できる文字列は合成しておく。⇒ 廃止
//	private static final String VERSION_PREFIX = "v"; // 4.3.5.3 (2008/02/22)
//	private static final String VERSION_PREFIX = "?v=" + BuildNumber.BUILD_ID ;

	// 5.2.1.0 (2010/10/01) キャッシュは、各ファイルのタイムスタンプベースで行う。
	private static final Map<String,String> srcAdrs = new HashMap<String,String>();

	// 5.2.1.0 (2010/10/01) キャッシュのクリアは、adminメニューの[ﾌﾟｰﾙ削除]で可能
	static {
		final Cleanable clr = new Cleanable() {
			public void clear() {
				srcAdrs.clear();
			}
		};
		SystemManager.addCleanable( clr );
	}

	//enum化
	// 5.4.4.2 (2012/02/03) simple追加
	private enum HEAD_TYPE_ENUM { xml , init , query , result , menu , simple  };

	private String	body		= null;
	private String	title		= null;
	private HEAD_TYPE_ENUM headType = null;

//	private boolean useTimeSet		= true;		// 5.2.1.0 (2010/10/01) 廃止
//	private boolean useToggleFrame	= true;		// 5.2.1.0 (2010/10/01) 廃止
//	private boolean useInputFocus	= true;		// 5.2.1.0 (2010/10/01) 廃止
//	private boolean useHilightRow	= true;		// 5.2.1.0 (2010/10/01) 廃止
	private boolean useScrollBar	= true;		// 5.2.1.0 (2010/10/01) 廃止
//	private boolean usePrintLine	= true;		// 5.2.1.0 (2010/10/01) 廃止
	private boolean useTabstrip		= false;
	private boolean useGantt		= false;
	private boolean useGantt2		= false;	// 5.6.2.1 (2013/03/08) 追加
	private boolean useTreeBOM		= false;
	private boolean useGuest		= false;		// 4.2.2.0 (2008/05/28)
	private boolean useAjaxTreeTable= false;		// 4.3.1.0 (2008/09/02)
	private boolean noTransition		= HybsSystem.sysBool( "NO_TRANSITION_MODE" );		// 4.3.1.0 (2008/09/02)
	private boolean useAjaxSubmit		= HybsSystem.sysBool( "USE_AJAX_SUBMIT" );
//	private boolean useRealTimeCheck	= HybsSystem.sysBool( "USE_REAL_TIME_CHECK" );		// 4.3.3.0 (2008/10/01)
	private boolean useRealTimeCheck	= true;		// 5.4.5.0 (2012/03/01)
	private boolean useTabLink			= false;	// 4.3.3.0 (2008/10/01)
//	private boolean useAutoConvert		= HybsSystem.sysBool( "USE_INPUT_AUTO_CONVERT" );	// 4.3.5.0 (2009/02/01)
//	private boolean useFocusHilight		= true;		// 4.3.5.0 (2009/02/01)
	private boolean useFocusHilight		= true;		// 5.4.5.0 (2012/03/01)
//	private boolean useFocusFirstElm	= true;		// 4.3.5.0 (2009/02/01)
//	private boolean useAdjustFrameHeight = true;	// 4.3.5.0 (2009/02/01)
//	private boolean useEventColumn		= true;		// 4.3.6.0 (2009/04/01)
	private boolean useIE7Header		= HybsSystem.sysBool( "USE_IE7_HEADER" );			// 4.3.7.0 (2009/06/01) IE8対応
//	private boolean useRowCountReplace	= HybsSystem.sysBool( "USE_ROWCNT_REPLACE" );		// 4.3.8.0 (2009/08/01)
	private boolean useViewPort			= HybsSystem.sysBool( "USE_VIEWPORT_HEADER" );		// 5.5.8.0 (2012/11/01)
	private String  addEvent			= null;		// 5.6.3.0 (2013/04/01) addEvent 追加

	/**
	 * このタグハンドラの親(最も近くを囲んでいるタグハンドラ)を設定する
	 * setParent( Tag ) を オーバーライドします。
	 * この、オーバーライドは、カスタムタグのライフサイクルに基づき、
	 * 各種セッターメソッドの前に呼ばれる メソッドとして、利用しています。
	 *
	 * 目的は、sessionInit() メソッドの実行により、{&64;SYS.XXXX} 変数を利用可能にすることです。
	 *
	 * @og.rev 4.3.4.1 (2008/12/08) 新規作成
	 *
	 * @param  tag タグハンドラの親
	 */
	@Override
	public void setParent( final Tag tag ) {
		super.setParent( tag );
		sessionInit() ;
	}

	/**
	 * Taglibの開始タグが見つかったときに処理する doStartTag() を オーバーライドします。
	 *
	 * @og.rev 4.2.1.0 (2008/04/16) headTypeのenum化
	 * @og.rev 4.3.4.1 (2008/12/08) sessionInit() を、setParent( Tag ) メソッドへ移動、title 属性の設定
	 * @og.rev 5.2.3.0 (2010/12/01) 画面アクセス履歴管理対応
	 *
	 * @return	後続処理の指示( EVAL_BODY_BUFFERED )
	 */
	@Override
	public int doStartTag() {
		// 4.3.5.7 (2009/03/22) アクセスカウントの不具合対応。sessionInitから移動
		if( headType == HEAD_TYPE_ENUM.result ) {
			GUIInfo guiInfo = (GUIInfo)getSessionAttribute( HybsSystem.GUIINFO_KEY );
			guiInfo.addAccessCount();
			// 5.2.3.0 (2010/12/01) 画面アクセス履歴管理対応
			UserInfo userInfo = (UserInfo)getSessionAttribute( HybsSystem.USERINFO_KEY );
			userInfo.setAccessGui( guiInfo );
		}

		HttpServletResponse response = (HttpServletResponse)pageContext.getResponse();
		// if( headType == XML_TYPE ) { // 4.2.1.0 (2008/04/16)
		if( headType == HEAD_TYPE_ENUM.xml ) {
			response.setHeader( "contentType","text/xml; charset=UTF-8" );
		}
		response.setHeader( "Cache-Control","max-age=0" );

		// if( headType == QUERY_TYPE ) { // 4.2.1.0 (2008/04/16)
		if( headType == HEAD_TYPE_ENUM.query ){
			removeSessionAttribute( HybsSystem.TBL_MDL_CONKEY );
		}

		// 4.3.4.1 (2008/12/08) title 属性を sessionInit() でなく、doStartTag() メソッドで処理
		// title 属性を最優先で設定する。
		if( title != null ) {
			setSessionAttribute( "GUI_TITLE", title );
		}
		else {
			title = String.valueOf( getSessionAttribute( "GUI_TITLE" ) );
		}

		return( EVAL_BODY_BUFFERED );	// Body を評価する。( extends BodyTagSupport 時)
	}

	/**
	 * Taglibのタグ本体を処理する doAfterBody() を オーバーライドします。
	 *
	 * @return	後続処理の指示(SKIP_BODY)
	 */
	@Override
	public int doAfterBody() {
		body = nval( getBodyString(),null );
		return(SKIP_BODY);
	}

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @og.rev 4.2.1.0 (2008/04/16) headTypeのenum化、HTMLタグにclass属性を出力
	 * @og.rev 4.3.1.0 (2008/09/02) AJaxツリーテーブル、画面遷移なしモードの対応
	 * @og.rev 4.3.3.0 (2008/10/01) リアルタイムチェック時にjQueryをインクルード・tabLinkのJS追加
	 * @og.rev 4.3.5.0 (2009/02/01) JavaScript整理に伴う変更
	 * @og.rev 4.3.6.0 (2009/04/01) eventColumn追加
	 * @og.rev 4.3.8.0 (2009/08/01) useAjaxSubmit追加
	 * @og.rev 5.1.3.0 (2010/02/01) CSSは、headTypeがnullでもinclude(forward.jspでエラーが発生した場合への対応)
	 * @og.rev 5.1.3.0 (2010/02/01) noTransition,ajaxSubmitのコントロールは、requestで行う。
	 * @og.rev 5.2.0.0 (2010/09/01) Javaスクリプトで登録したイベントをイベントハンドラにバインドする
	 * @og.rev 5.2.1.0 (2010/10/01) bulkSet.js追加 、JavaScript の統合、廃止
	 * @og.rev 5.3.6.0 (2011/06/01) QUERYのFocusの制御を追加
	 * @og.rev 5.4.4.2 (2012/02/03) type simple追加
	 * @og.rev 5.4.5.0 (2012/03/01) realTimeCheck,focusHilight制御追加
	 * @og.rev 5.5.8.0 (2012/11/01) viewport
	 * @og.rev 5.6.0.3 (2013/01/24) EventScriptの前に呼ぶqueryScriptPre追加
	 * @og.rev 5.6.2.1 (2013/03/08) useGantt2="true" の条件追加
	 * @og.rev 5.6.2.3 (2013/03/22) QUERYPRE_SCRIPT_JS   → QUERYCONV_SCRIPT_JS に変更
	 * @og.rev 5.6.3.0 (2013/04/01) addEvent 追加。useGantt2="true" 時の処理内容と分離
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)

		// 4.3.3.0 (2008/10/01) デフォルト値として、リクエストパラメーターを使用
		noTransition = nval( getRequest().getParameter( HybsSystem.NO_TRANSITION_MODE_KEY ), noTransition ) ;
		useAjaxSubmit = nval( getRequest().getParameter( HybsSystem.USE_AJAX_SUBMIT_KEY ), useAjaxSubmit ) ;
		// 4.3.8.0 (2009/08/01) useAjaxSubmit追加 noTransitionがtrueの場合はajaxSubmitはfalse
		if( noTransition ) { useAjaxSubmit = false; }

		setNoTransitionRequest( noTransition );
		setAjaxSubmitRequest( useAjaxSubmit );

		// if( headType == XML_TYPE ) { // 4.2.1.0 (2008/04/16)
		if( headType == HEAD_TYPE_ENUM.xml ){
			jspPrint( XML_HEAD );
			return(EVAL_PAGE);		// ページの残りを評価する。
		}

		StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );

		if( useTabstrip ) { buf.append( TAB_HEAD ); }	// tabstrip 機能を使用する場合
		else			  { buf.append( HEAD1 );
							buf.append( headType );
							buf.append( HEAD2  );
		}

		// 4.3.7.0 (2009/06/01) IE8対応
		if ( useIE7Header ){
			buf.append ( IE7_HEAD );
		}
		
		// 5.5.8.0 (2012/11/01) 
		if( useViewPort ){
			buf.append( MOBILE_HEAD );
		}

		if( title.startsWith( "<span" ) ) {
			int adrs1 = title.indexOf( '>' );
			int adrs2 = title.length() - 7;		// </span> で終了しているはず。
			buf.append( title.substring( 0,adrs1+1 ) ).append( "<title>" );
			buf.append( title.substring( adrs1+1,adrs2 ) ).append( "</title>" );
			buf.append( title.substring( adrs2 ) );
		}
		else {
			buf.append( "  <title>" ).append( title ).append( "</title>" ).append( CR );
		}

		// 5.4.5.0 (2012/03/01) イベント制御用JSフラグ出力
		if( headType == HEAD_TYPE_ENUM.query || headType == HEAD_TYPE_ENUM.result ){
			buf.append("<script type=\"text/javascript\">").append( CR );
			addScriptFlg(REAL_TIME_CHECK_FLG ,useRealTimeCheck ,buf);
			addScriptFlg(FOCUS_HILIGHT_FLG ,useFocusHilight ,buf);
			buf.append( "</script>" ).append( CR );
		}

		// 5.1.3.0 (2010/02/01) CSSは、headTypeがnullでもinclude
		// 5.4.4.2 (2012/02/03) CSSを読み込まないタイプsimpleを追加
		if( headType == null || ( headType != HEAD_TYPE_ENUM.xml && headType != HEAD_TYPE_ENUM.simple ) ) {
			addStyleTag(  DEFAULT_CSS	,buf );
			addStyleTag(  CALENDAR_CSS	,buf );
			addStyleTag(  CUSTOM_CSS	,buf );
		}

		// init,query,result,menu タイプで使用されるJavaScript
		// 5.4.4.2 (2012/02/03) simple追加
		if ( headType != null && headType != HEAD_TYPE_ENUM.xml && headType != HEAD_TYPE_ENUM.simple ){ // 4.2.1.0 (2008/04/21) or連結を反転
			addScriptTag( JQUERY_JS		,buf ); // 4.3.5.0 (2009/02/01)
			addScriptTag( JQUERY_OG_JS  ,buf ); // 4.3.5.0 (2009/02/01)
			addScriptTag( DEFAULT_JS	,buf );
//			addScriptTag( INPUTMENU_JS	,buf );		// 5.2.1.0 (2010/10/01) DEFAULT_JS に統合
//			addScriptTag( OGPOPUP_JS	,buf );		// 5.2.1.0 (2010/10/01) DEFAULT_JS に統合
			if( useTabLink ) { addScriptTag( TABLINK_JS	,buf );	} // 4.3.3.0 (2008/10/01)
		}

		// 5.6.0.3 (2013/01/24) EventScrpt前に呼び出すquery専用スクリプト
		if( headType == HEAD_TYPE_ENUM.query ){
//			addScriptTag( QUERYPRE_SCRIPT_JS ,buf );			// 5.6.2.3 (2013/03/22) QUERYPRE_SCRIPT_JS   → QUERYCONV_SCRIPT_JS に変更
			addScriptTag( QUERYCONV_SCRIPT_JS ,buf );
		}
		
		// query,result タイプで使用されるJavaScript
		if( headType == HEAD_TYPE_ENUM.query || headType == HEAD_TYPE_ENUM.result ){
			addScriptTag( EVENT_SCRIPT_JS, buf );	// 5.2.1.0 (2010/10/01) JavaScript新規統合
//			if( useFocusHilight ) { addScriptTag( FOCUS_HILIGHT_JS, buf ); }	// 4.3.5.0 (2009/02/01)
//			if( useAutoConvert ) { addScriptTag( AUTO_CONVERT_JS, buf ); }		// 4.3.5.0 (2009/02/01)
//			if( useEventColumn ) { addScriptTag( EVENT_COLUMN_JS, buf ); }		// 4.3.6.0 (2009/04/01)
			// 4.3.3.0 (2008/10/01)
//			if( noTransition || useAjaxTreeTable || useRealTimeCheck || useAjaxSubmit ) { // 4.3.8.0 (2009/08/01)
				if( noTransition || useAjaxSubmit ){ // 4.3.8.0 (2009/08/01)
					addScriptTag( JQUERY_FORM_JS, buf );
					addScriptTag( JQUERY_DRAG_JS, buf );
					addScriptTag( JQUERY_MODAL_JS, buf );
					addStyleTag( JQUERY_MODAL_CSS, buf );
				}
//				if( useRealTimeCheck ){ addScriptTag( REAL_TIME_CHECK_JS, buf ); } // 4.3.4.2 (2008/12/15)
//			}

			// 5.2.0.0 (2010/09/01) Javaスクリプトで登録したイベントをイベントハンドラにバインド
//			addScriptTag( BIND_EVENTS_JS ,buf );
		}

		// 4.3.5.0 (2008/02/01) 位置を両タイプで利用される前のインクルードから後に変更
		// query タイプで使用されるJavaScript
		if( headType == HEAD_TYPE_ENUM.query ){
			addScriptTag( QUERY_SCRIPT_JS ,buf );	// 5.2.1.0 (2010/10/01) JavaScript新規統合

			// 5.3.6.0 (2011/06/01) QUERYのFocusの制御を追加
			boolean isFocus = StringUtil.nval( getRequestValue( HybsSystem.QUERY_FOCUS_KEY ), true );
			// 機能を利用しない場合のみ変数を出力
			if( !isFocus ) { addScriptValue( HybsSystem.QUERY_FOCUS_KEY, "false", buf ); }
//			if( useTimeSet		) { addScriptTag( TIME_SET_JS	 ,buf ); }
//			if( useToggleFrame	) { addScriptTag( TOGGLE_FRAME_JS,buf ); }
//			if( useAdjustFrameHeight ) { addScriptTag( ADJUST_FRAME_HEIGHT_JS,buf ); } // 4.3.5.0 (2009/02/01)
//			if( useFocusFirstElm ) { addScriptTag( FOCUS_FIRST_ELM,buf ); }	// 4.3.5.0 (2009/02/01)
		}

		// result タイプで使用されるJavaScript
		if( headType == HEAD_TYPE_ENUM.result ) {
			addScriptTag( RESULT_SCRIPT_JS ,buf );	// 5.2.1.0 (2010/10/01) JavaScript新規統合

//			if( useHilightRow	) { addScriptTag( HILIGHT_ROW_JS	,buf ); }
			if( useScrollBar	) {
									//  4.3.5.0 (2009/02/01) CSSも読み込む
									addScriptTag( SCROLL_BAR_JS, buf );
									addStyleTag ( SCROLL_BAR_CSS, buf );

					// ※ 取り合えず、専用の制御フラグを持たないので、useScrollBar に間借りします。
//					addScriptTag( BULK_SET_JS, buf );	// 5.2.1.0 (2010/10/01)
			}
//			if( usePrintLine	) { addScriptTag( PRINT_LINE_JS		,buf ); }
//			if( useTimeSet		) { addScriptTag( TIME_CHECK_JS		,buf ); }
			if( useGantt		) { addScriptTag( GANTT_JS			,buf ); }
			if( useTreeBOM		) { addScriptTag( TREEBOM_JS		,buf ); }
			if( useAjaxTreeTable) { addScriptTag( AJAX_TREETABLE_JS ,buf ); }		// 4.3.1.0 (2008/09/02)
			if( noTransition	) {
									addScriptTag( AJAX_HYBS_FORM	,buf );			// 4.3.8.0 (2009/08/01)
									addScriptTag( NO_TRANSITION_JS	,buf );
								  } // 4.3.1.0 (2008/09/02)
			if( useAjaxSubmit	) {
									addScriptTag( AJAX_HYBS_FORM	,buf );
									addScriptTag( AJAX_SUBMIT_JS	,buf );
								  } // 4.3.8.0 (2009/08/01)
//			if ( useRowCountReplace ){ addScriptTag( ROW_COUNT_REPLACE ,buf ); }	// 4.3.8.0 (2009/08/01)
			if( useGantt2		) {													// 5.6.2.1 (2013/03/08) useGantt2 追加
									addStyleTag ( GANTT2_CSS	,buf );
									addScriptTag( GANTT2_JS		,buf );
//									buf.append( "  <script type=\"text/javascript\">" );
//									buf.append( GANTT2_SCRIPT );
//									buf.append( "  </script>" ).append( CR );
			}

			// 5.6.3.0 (2013/04/01) addEvent 追加
			if( addEvent != null && addEvent.length() > 0 ) {
									buf.append( "  <script type=\"text/javascript\"> addEvent( this, \"load\", " );
									buf.append( addEvent );
									buf.append( " ); </script>" ).append( CR );
			}
		}

		// 注意：body の直前には、改行は入れないほうが良い。(Tomcat4.1.30 では)
		if( body != null ) {	buf.append( body ); }
		buf.append( "</head>" );

		jspPrint( buf.toString() );
		return(EVAL_PAGE);		// ページの残りを評価する。
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 * @og.rev 5.4.5.0 (2012/03/01) realTimeCheck,focusHilight
	 * @og.rev 5.5.8.0 (2012/11/01) useViewPort  追加
	 * @og.rev 5.6.2.1 (2013/03/08) useGantt2  追加
	 * @og.rev 5.6.3.0 (2013/04/01) addEvent 追加
	 */
	@Override
	protected void release2() {
		super.release2();
		body			= null;
		title			= null ;
		headType		= null;
//		useTimeSet		= true;		// 5.2.1.0 (2010/10/01) 統合
//		useToggleFrame	= true;		// 5.2.1.0 (2010/10/01) 統合
//		useInputFocus	= true;		// 5.2.1.0 (2010/10/01) 統合
//		useHilightRow	= true;		// 5.2.1.0 (2010/10/01) 統合
		useScrollBar	= true;
//		usePrintLine	= true;		// 5.2.1.0 (2010/10/01) 統合
		useTabstrip		= false;
		useGantt		= false;
		useGantt2		= false;	// 5.6.2.1 (2013/03/08) 追加
		useTreeBOM		= false;
		useGuest		= false;		// 4.2.2.0 (2008/05/28)
		useAjaxTreeTable= false;		// 4.3.1.0 (2008/09/02)
		noTransition	= HybsSystem.sysBool( "NO_TRANSITION_MODE" );	// 4.3.1.0 (2008/09/02)
		useAjaxSubmit	= HybsSystem.sysBool( "USE_AJAX_SUBMIT" );
//		useRealTimeCheck= HybsSystem.sysBool( "USE_REAL_TIME_CHECK" );	// 4.3.3.0 (2008/10/01) ,// 5.2.1.0 (2010/10/01) 統合
		useRealTimeCheck= true;	// 5.4.5.0 (2012/03/01) 復活
		useTabLink		= false;		// 4.3.3.0 (2008/10/01)
//		useAutoConvert	= HybsSystem.sysBool( "USE_INPUT_AUTO_CONVERT" ); // 4.3.5.0 (2009/02/01) , // 5.2.1.0 (2010/10/01) 統合
//		useFocusHilight	= true;			// 4.3.5.0 (2009/02/01) , // 5.2.1.0 (2010/10/01) 統合
		useFocusHilight	= true;			// 5.4.5.0 (2012/03/01) 復活
//		useFocusFirstElm= true;			// 4.3.5.0 (2009/02/01) , // 5.2.1.0 (2010/10/01) 統合
//		useAdjustFrameHeight = true;	// 4.3.5.0 (2009/02/01) , // 5.2.1.0 (2010/10/01) 統合
//		useEventColumn	= true;			// 4.3.6.0 (2009/04/01) , // 5.2.1.0 (2010/10/01) 統合
		useIE7Header    = HybsSystem.sysBool( "USE_IE7_HEADER" ); // 4.3.7.0 (2009/06/01) IE8対応
//		useRowCountReplace = HybsSystem.sysBool( "USE_ROWCNT_REPLACE" ); // 4.3.8.0 (2009/08/01) , // 5.2.1.0 (2010/10/01) 統合
		useViewPort		= HybsSystem.sysBool( "USE_VIEWPORT_HEADER" ); // 5.5.8.0 (2012/11/01)
		addEvent		= null;			// 5.6.3.0 (2013/04/01) addEvent 追加
	}

	/**
	 * スクリプトを制御するための変数文字列を作成します。
	 * JavaScriptのファイル統合によりファイルを行うか否かでは、機能のON/OFFを
	 * 行うことができません。
	 * このため、機能のON/OFFを制御するための変数を書き出して、JavaScript内で
	 * これをコントロールします。
	 * &lt;script type="text/javascript" &lt;!-- var (key)="(value)"; --&gt;&lt;/script&gt;
	 *
	 * @og.rev 5.3.6.0 (2011/06/01) 新規作成
	 *
	 * @param	key		JavaScript変数名
	 * @param	value	JavaScriptに渡す値
	 * @param	buf	指定文字列を追加する StringBuilder
	 *
	 * @return	追加されたStringBuilder
	 */
	private StringBuilder addScriptValue( final String key, final String value, final StringBuilder buf ) {
		if( value != null && value.length() > 0 ) {
			String tag = "  <script type=\"text/javascript\">"
				+ " var " + key + "=\"" + value + "\"; </script>"
				+ CR ;
			buf.append( tag );
		}
		return buf ;
	}

	/**
	 * スクリプト指定文字列を作成します。
	 * JSPアドレスと、スクリプトファイル名を元に、スクリプト指定文字列を作成します。
	 * &lt;script type="text/javascript" <del>language="JavaScript"</del>
	 *       src="(jspAdrs)(filename)" &gt;&lt;!-- --&gt;&lt;/script&gt;
	 *
	 * @og.rev 4.3.5.3 (2008/02/22) Tomcatキャッシュ対応(CSSとJavaScriptのパラメーターにバージョン番号を付加)
	 * @og.rev 5.2.1.0 (2010/10/01) 予め合成できる文字列は合成しておく。
	 * @og.rev 5.2.1.0 (2010/10/01) キャッシュのパラメータを、各ファイルの最終更新日を使用する。
	 * @og.rev 5.6.2.1 (2013/03/08) ファイルの最終更新日が取得できない場合は、BuildNumber.BUILD_ID を使用する。
	 *
	 * @param	filename	スクリプトファイル名(common/default.js)
	 * @param	buf			指定文字列を追加する StringBuilder
	 *
	 * @return	追加されたStringBuilder
	 */
//	private StringBuilder addScriptTag( final String filename, final StringBuilder buf ) {
//		buf.append( "  <script type=\"text/javascript\" src=\"" );
//		buf.append( JSP_ADRS );
////		buf.append( XHTMLTag.addUrlEncode( filename, XHTMLTag.urlEncode( VERSION_PREFIX, BuildNumber.BUILD_ID ) ) ); // 4.3.5.3 (2008/02/22)
//		buf.append( filename ).append( VERSION_PREFIX );	// 5.2.1.0 (2010/10/01)
//		buf.append( "\" ><!-- --></script>" ).append( CR );
//
//		return buf ;
//	}
	private StringBuilder addScriptTag( final String filename, final StringBuilder buf ) {
		String tag = srcAdrs.get( filename );
		if( tag == null ) {
			File file = new File( JSP_DIR + filename );
			// 5.6.2.1 (2013/03/08) ファイルの最終更新日が取得できない場合は、BuildNumber.BUILD_ID を使用
			long ts = file.lastModified();
			String tsv = (ts != 0L) ? String.valueOf( ts ) : BuildNumber.BUILD_ID ;

			tag = "  <script type=\"text/javascript\" src=\""
					+ JSP_ADRS
					+ filename
					+ "?v="
//					+ String.valueOf( file.lastModified() )
//					+ file.lastModified()						// 5.3.0.0 (2010/12/01) String.valueOf は不要
					+ tsv										// 5.6.2.1 (2013/03/08) 
					+ "\" ><!-- --></script>"
					+ CR ;
			srcAdrs.put( filename,tag );
		}
		buf.append( tag );
		return buf ;
	}

	/**
	 * var xxx = true;のような文字列を作成します。
	 * sxxとtrue/falseを外部から与えます。
	 *
	 * @og.rev 5.4.5.0 (2012/03/01) 新規作成
	 *
	 * @param	var	変数名
	 * @param	flg	true/false
	 * @param	buf	指定文字列を追加する StringBuilder
	 *
	 * @return	追加されたStringBuilder
	 */
	private StringBuilder addScriptFlg( final String var, final boolean flg, final StringBuilder buf ) {
		buf.append( "var ")
			.append( var )
			.append( " = ")
			.append( flg )
			.append( ";" )
			.append( CR );
		return buf ;
	}

	/**
	 * スタイルシート指定文字列を作成します。
	 * JSPアドレスと、スタイルシートファイル名を元に、スタイルシート指定文字列を作成します。
	 * &lt;link rel="stylesheet" type="text/css"
	 *       href="(jspAdrs)(filename)" /&gt;
	 *
	 * @og.rev 4.3.5.3 (2008/02/22) Tomcatキャッシュ対応(CSSとJavaScriptのパラメーターにバージョン番号を付加)
	 * @og.rev 5.2.1.0 (2010/10/01) 予め合成できる文字列は合成しておく。
	 * @og.rev 5.2.1.0 (2010/10/01) キャッシュのパラメータを、各ファイルの最終更新日を使用する。
	 * @og.rev 5.3.0.0 (2010/11/22) linkタグの終了がscriptで終了しているバグを修正。
	 * @og.rev 5.6.2.1 (2013/03/08) ファイルの最終更新日が取得できない場合は、BuildNumber.BUILD_ID を使用する。
	 *
	 * @param	filename	スタイルシートファイル名(common/default.js)
	 * @param	buf			指定文字列を追加する StringBuilder
	 *
	 * @return	追加されたStringBuilder
	 */
//	private StringBuilder addStyleTag( final String filename, final StringBuilder buf ) {
//		buf.append( "  <link rel=\"stylesheet\" type=\"text/css\" href=\"" );
//		buf.append( JSP_ADRS );
////		buf.append( XHTMLTag.addUrlEncode( filename, XHTMLTag.urlEncode( VERSION_PREFIX, BuildNumber.BUILD_ID ) ) ); // 4.3.5.3 (2008/02/22)
//		buf.append( filename ).append( VERSION_PREFIX );	// 5.2.1.0 (2010/10/01)
//		buf.append( "\" />" ).append( CR );
//
//		return buf ;
//	}
	private StringBuilder addStyleTag( final String filename, final StringBuilder buf ) {
		String tag = srcAdrs.get( filename );
		if( tag == null ) {
			File file = new File( JSP_DIR + filename );
			// 5.6.2.1 (2013/03/08) ファイルの最終更新日が取得できない場合は、BuildNumber.BUILD_ID を使用
			long ts = file.lastModified();
			String tsv = (ts != 0L) ? String.valueOf( ts ) : BuildNumber.BUILD_ID ;

			tag = "  <link rel=\"stylesheet\" type=\"text/css\" href=\""
					+ JSP_ADRS
					+ filename
					+ "?v="
//					+ String.valueOf( file.lastModified() )
//					+ file.lastModified()						// 5.3.0.0 (2010/12/01) String.valueOf は不要
					+ tsv										// 5.6.2.1 (2013/03/08) 
//					+ "\" ><!-- --></script>"
					+ "\" />"
					+ CR ;
			srcAdrs.put( filename,tag );
		}
		buf.append( tag );
		return buf ;
	}

	/**
	 * セッションの初期処理を行います。
	 * common/session_init.jsp をタグに組み込みました。
	 *
	 * @og.rev 3.8.0.0 (2005/06/07) 同一セッションでのマルチ起動対策を行います。
	 * @og.rev 3.8.0.4 (2005/08/08) custom を呼び出せるように追加
	 * @og.rev 3.8.1.2 (2005/12/19) 認証無しの場合のユーザー制御修正
	 * @og.rev 4.0.0.0 (2007/10/18) メッセージリソース統合( getResource().getMessage > getResource().getLabel )
	 * @og.rev 4.0.0.0 (2007/10/31) ロール指定でのログイン機能追加
	 * @og.rev 4.2.1.0 (2008/04/16) headTypeのenum化
	 * @og.rev 4.2.2.0 (2008/05/28) GUESTユーザの復活
	 * @og.rev 4.2.4.1 (2008/07/08) pageContext.include時にflushしないように修正
	 * @og.rev 4.3.3.1 (2008/10/08) システムパラメータの初期値設定は userInfo 作成ブロックの外で行うように変更します。
	 * @og.rev 4.3.4.1 (2008/12/08) title 属性を使用しなくします。(doStartTag() メソッドで処理)
	 * @og.rev 5.1.4.0 (2010/03/01) lang を書き換え可能とする。
	 * @og.rev 5.1.8.0 (2010/07/01) useGuest属性が有効にならないバグを修正。
	 * @og.rev 5.1.8.0 (2010/07/01) ユーザーIDの大文字・小文字を区別する(大文字化を廃止)
	 * @og.rev 5.2.0.0 (2010/09/01) アクセス禁止アドレスによる不正アクセス防止機能追加
	 */
	private void sessionInit() {
		HttpSession session = pageContext.getSession();
		HttpServletRequest request = (HttpServletRequest)pageContext.getRequest();

		// 3.5.3.0 (2003/10/27) encode.jsp 廃止
		try {
			request.setCharacterEncoding( "UTF-8" );
		}
		catch (UnsupportedEncodingException ex) {
			throw new HybsSystemException( ex );
		}

		// 3.8.0.0 (2005/06/07) 同一セッションでのマルチ起動対策を行います。
		String cntR = request.getParameter( HybsSystem.MULTI_SESSION_CHECK );
		if( cntR != null ) {
			String cntS = (String)session.getAttribute( HybsSystem.MULTI_SESSION_CHECK );
			if( cntS != null && !cntR.equals( cntS ) ) {
				ResourceManager resource = ResourceFactory.newInstance( request.getLocale().getLanguage() );

				StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
				buf.append( resource.getLabel( "ERR0035.T" ) ).append( HybsSystem.CR );
				buf.append( resource.getLabel( "ERR0035.1" ) ).append( HybsSystem.CR );
				buf.append( resource.getLabel( "ERR0035.2" ) ).append( HybsSystem.CR );
				buf.append( resource.getLabel( "ERR0035.3" ) ).append( HybsSystem.CR );

				throw new HybsSystemException( buf.toString() );
			}
		}

		// システムパラメータの初期値設定
		// 4.3.3.1 (2008/10/08) 戻るリンクの不具合対応で、後方の userInfo 作成ブロックの外に出してくる。
		if( ! HybsSystem.isParamSet() ) {
			String	serverName	= request.getServerName();
			String	scheme		= request.getScheme();
			int		port		= request.getServerPort();
			String	contextPath = request.getContextPath();

			HybsSystem.setRequestParam( serverName,scheme,port,contextPath );
		}

		// 3.3.0.0 (2003/06/23) 操作途中以外でのセッション切断に対応。
		// ログイン時の初期化を行います。
		UserInfo userInfo = (UserInfo)getSessionAttribute( HybsSystem.USERINFO_KEY );
		String userID = request.getRemoteUser() ;

		// 3.8.1.2 (2005/12/19)
		if( userInfo == null || ( userID != null && ! userID.equalsIgnoreCase( userInfo.getUserID() ))) {
		// セッションタイムアウトの確認 command が、null,"NEW","INIT" 以外は、エラー
			String command = request.getParameter( "command" );
			if( command == null || "NEW".equals( command ) || "INIT".equals( command ) ) {
				if( userID == null ) {
					// 4.2.2.0 (2008/05/28) GUESTユーザ復活(useGuestで制御)
					// 5.1.8.0 (2010/07/01) useGuest属性が有効にならないバグを修正。
					userID = "GUEST" ;		// 認証されていない場合は、強制的に GUESTユーザー
				}
				userInfo = UserInfoFactory.newInstance( userID,request.getRemoteAddr(),request.getParameter( "ROLES" ) );

				// 5.1.4.0 (2010/03/01) lang を書き換え可能とする。
				String newLang = request.getParameter( "NEWLANG" ) ;
				if( newLang != null ) { userInfo.setLang( newLang ); }

				session.setAttribute( HybsSystem.USERINFO_KEY, userInfo );

				// セッションタイムアウトを変更
				session.setMaxInactiveInterval( HybsSystem.sysInt( "MAX_INACTIVE_INTERVAL" ) );
				SystemManager.addSession( session );

				try {
					// 4.2.4.1 (2008/07/08)
					pageContext.include( "/jsp/custom/init.jsp", false );
				} catch ( IOException ex ) {
					String errMsg = "/jsp/custom/init.jsp の include に失敗しました。 " + toString();
					throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
				} catch ( ServletException ex ) {
					String errMsg = "/jsp/custom/init.jsp の include に失敗しました。 " + toString();
					throw new HybsSystemException( errMsg,ex );		// 3.5.5.4 (2004/04/15) 引数の並び順変更
				}
			}
			else {
				ResourceManager resource = ResourceFactory.newInstance( request.getLocale().getLanguage() );

				// エラー表示ページを表示する。4.0.0 (2005/01/31)
				String msg = String.valueOf( HybsSystem.sysInt( "MAX_INACTIVE_INTERVAL" ) ) ;

				// ERR0002.T=ユーザーログインエラー
				// ERR0002.1=セッションのタイムアウトか，またはユーザーログインエラーが発生しました。
				// ERR0002.2=現在のタイムアウトは {0} 秒にセットされています。
				// ERR0002.3=長時間使用しない場合は,画面を閉じる事により,サーバー側の負荷を軽減する事ができます。
				// ERR0002.4=<h2>もう一度，ログインし直して下さい。</h2>

				StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
				buf.append( resource.getLabel( "ERR0002.T" ) ).append( HybsSystem.CR );
				buf.append( resource.getLabel( "ERR0002.1" ) ).append( HybsSystem.CR );
				buf.append( resource.getLabel( "ERR0002.2",new String[] { msg } ) ).append( HybsSystem.CR );
				buf.append( resource.getLabel( "ERR0002.3" ) ).append( HybsSystem.CR );
				buf.append( resource.getLabel( "ERR0002.4" ) ).append( HybsSystem.CR );

				throw new HybsSystemException( buf.toString() );		// 4.0.0 (2005/01/31)
			}
		}

		boolean useGamenId = false;
		String gamenId = request.getParameter( "GAMENID" );
		String requestURI = request.getRequestURI();

		int index2 = requestURI.lastIndexOf( '/' );
		String jspID   = requestURI.substring( index2+1 );
		if( gamenId == null ) {
			int index1 = requestURI.lastIndexOf( '/',index2-1 );
			gamenId = requestURI.substring( index1+1,index2 );
		}
		else {
			useGamenId = true;
		}

		ResourceManager resource = ResourceFactory.newInstance( userInfo.getLang() );
		// 4.0.0 (2004/12/31)
		if( ! userInfo.isGUIInfoSet() ) {
			resource.makeGUIInfos( userInfo );
		}

		GUIInfo guiInfo = null;
		String gui_title ;
		// 3.8.0.4 (2005/08/08) custom を呼び出せるように追加
		if( "menu".equals( gamenId ) || "jsp".equals( gamenId ) || "custom".equals( gamenId ) || "common".equals( gamenId ) ) {
			// 3.5.4.5 (2004/01/23) トップメニューの設定
	 		// 4.3.4.1 (2008/12/08) title 属性を使用しなくします。(doStartTag() メソッドで処理)
			gui_title = HybsSystem.sys( "GUI_TOP_TITLE" );
			if( gui_title == null ) { gui_title = "Hayabusa " + gamenId ; }
		}
		else {
			guiInfo = userInfo.getGUIInfo( gamenId );		// 4.0.0 (2004/12/31)

			if( guiInfo != null ) {
				session.setAttribute( HybsSystem.GUIINFO_KEY, guiInfo );
				session.setAttribute( "JSPID", jspID );
			}
			else {
				// エラー表示ページを表示する。4.0.0 (2005/01/31)
				String msg = userInfo.getJname();

				StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
				buf.append( resource.getLabel( "ERR0003.T" ) ).append( HybsSystem.CR );
				buf.append( resource.getLabel( "ERR0002.1",new String[] { msg } ) ).append( HybsSystem.CR );
				buf.append( resource.getLabel( "ERR0002.2" ) ).append( HybsSystem.CR );

				throw new HybsSystemException( buf.toString() );		// 4.0.0 (2005/01/31)
			}

			// 画面のタイトルをリクエスト変数でも指定できるようにします。
			// 相当、やっつけ仕事が入っていますが、必ず見直します。
	 		// 4.3.4.1 (2008/12/08) title 属性を使用しなくします。(doStartTag() メソッドで処理)
			gui_title = request.getParameter( "GUI_TITLE" );
			if( gui_title == null || gui_title.length() == 0 ) {
				String cmd = request.getParameter( "command" );
				if( cmd == null || cmd.equals( "NEW" ) ||
					 cmd.equals( "RENEW" ) || cmd.equals( "INIT" ) ||
					 cmd.equals( "RESET" ) || cmd.length() == 0  ) {
					gui_title = guiInfo.getLongName();
				}
			}
		}

		// 5.2.0.0 (2010/09/01) アクセス禁止アドレスによる不正アクセス防止機能追加
		if( useGamenId ) {
			int idx1 = requestURI.lastIndexOf( "/jsp/" ) + 5;
			int idx2 = requestURI.lastIndexOf( '/' );
			if( idx1 >= 0 && idx2 >= 0 && idx1 < idx2 ) {
				String reqAddr = requestURI.substring( idx1, idx2 );
				if( reqAddr.indexOf( '/' ) < 0 && !userInfo.isValidAddr( reqAddr ) ) {
					String msg = userInfo.getJname();
					StringBuilder buf = new StringBuilder( HybsSystem.BUFFER_MIDDLE );
					buf.append( resource.getLabel( "ERR0003.T" ) ).append( HybsSystem.CR );
					buf.append( resource.getLabel( "ERR0002.1",new String[] { msg } ) ).append( HybsSystem.CR );
					throw new HybsSystemException( buf.toString() );
				}
			}
		}

 		// 4.3.4.1 (2008/12/08) title 属性を使用しなくします。(doStartTag() メソッドで処理)
		if( gui_title != null ) {
			session.setAttribute( "GUI_TITLE", gui_title );
		}
	}

	/**
	 * 【TAG】HTML の タイトルを設定します(初期値:session に"GUI_TITLE" キーで登録されている値)。
	 *
	 * @og.tag
	 *  head タグのBody部に、title タグを設定します。
	 * &lt;title&gt;タイトル &lt;/title&gt; を追加します。
	 * 設定は、このタグでの指定が最優先されます。次に、リクエスト情報の
	 * GUI_TITLE があれば、使用されます。最後に、初期値である、session の
	 * "GUI_TITLE" キーで登録されている値を使用します。
	 * session に登録されている値とは、GUIリソースで定義された、名前(長)です。
	 *
	 * 初期値は、session に"GUI_TITLE" キーで登録されている値です。
	 * トップメニューのタイトルは、システムパラメータ の、GUI_TOP_TITLE キー
	 * で登録されている値です。
	 *
	 * @param	titl	タイトル
	 */
	public void setTitle( final String titl ) {
		title = nval( getRequestParameter( titl ),title );
	}

	/**
	 * 【TAG】ヘッダータイプを設定します(初期値:null)。
	 *
	 * @og.tag
	 * ヘッダータイプ とは、個別の設定をしなくても、基本的なバリエーションを
	 * 簡易的に設定する場合に使用するキーワードです。
	 * ここでは、null,xml,init,query,result,menu,simple の7つが指定できます。
	 * query を指定すると、query.jsp で使用する基本的な設定を再現できます。
	 * timeSet.js , toggleFrame.js , inputFocus.js が標準で設定されます。
	 * result も同様で、result.jsp で使用する基本的な設定を再現できます。
	 * hilightRow.js , ScrollBar.js , printLine.js ,
	 * timeCheck.js , inputFocus.js が標準で設定されます。
	 * inputFocus.js に関しては、query,result 両タイプとも設定されます。
	 * init の場合は、通常はユーザーに見えない画面ですが、登録時エラーで
	 * 処理が停止する場合があり、CSSファイルのみ組み込む必要があります。
	 * menu はmultiMenu.jsp等のメニュー画面用の設定です。
	 * なにも指定しない場合(null)では、index.jsp , forward.jsp などの、
	 * 直接画面に表示されないタイプのヘッダーになります。
	 * simple は何もインクルードしないタイプです。携帯電話用画面等でdefault.css
	 * すらインクルードしたくないが、JSPの作りは同じにしておきたい場合に利用します。
	 * xml は、Webサービスや、Ajax などで使用する XML レスポンス用の
	 * 特別なヘッダーを作成します。
	 *
	 * 初期値は、null です。
	 *
	 * @og.rev 4.2.1.0 (2008/04/16) headTypeのenum化に伴う変更
	 * @og.rev 5.4.4.2 (2012/02/03) simpleタイプ追加
	 *
	 * @param	type	ヘッダータイプ(null,xml,init,query,result)
	 */
	public void setHeadType( final String type ) {
		String hType = nval( getRequestParameter( type ),null );
		if( "null".equals( hType )){ // nullを入れた時
			hType = null;
		}
		if( hType != null ) {
			try {
				headType = HEAD_TYPE_ENUM.valueOf( hType );
			}
			catch (IllegalArgumentException  iaex) {
				StringBuilder errBuf = new StringBuilder(100);
				errBuf.append( "headType は" );
				for( HEAD_TYPE_ENUM obj : HEAD_TYPE_ENUM.values() ) {
					errBuf.append( ',' );
					errBuf.append( obj.name() );
				}
				errBuf.append("から選んでください。");
				throw new HybsSystemException( errBuf.toString(),iaex );	// 4.3.1.1 (2008/08/23)
			}
		}
	}

	/**
	 * 【TAG】timeSet.js , timeCheck.js JavaScript を使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、timeSet.js, timeCheck.js を使用するかどうかを
	 * 指定します。
	 * timeSet.js は、query タイプで、timeCheck.js は、result タイプで有効です。
	 * 初期値は、使用する(true)です。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 *
	 * @param	flag	使用するかどうか(true:使用する/false:使用しない)
	 */
//	public void setUseTimeSet( final String flag ) {
//		useTimeSet = nval( getRequestParameter( flag ),useTimeSet );
//	}

	/**
	 * 【TAG】toggleFrame.js JavaScript を使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、toggleFrame.js を使用するかどうかを指定します。
	 * query タイプで有効です。
	 * 初期値は、使用する(true)です。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 *
	 * @param	flag	使用するかどうか(true:使用する/false:使用しない)
	 */
//	public void setUseToggleFrame( final String flag ) {
//		useToggleFrame = nval( getRequestParameter( flag ),useToggleFrame );
//	}

	/**
	 * 【廃止】inputFocus.js JavaScript を使用するかどうかを指定します。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、inputFocus.js を使用するかどうかを指定します。
	 * query,result 両タイプで有効です。
	 * 初期値は、使用する(true)です。
	 *
	 * inputFocus.jsが廃止されたため、この属性は無効です。
	 * フォーカス色はfocusHilight.js
	 * リアルタイムチェックはrealTimeCheck.js
	 * 大小文字自動変換はautoConvert.js
	 * を利用するように変更されました。
	 * これらについては
	 * useFocusHilight,useRealTimeCheck,useAutoConvert
	 * 属性をご覧下さい。
	 *
	 * @og.rev 4.3.5.0 (2009/02/01) Deprecated
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 *
	 * @param	flag	使用するかどうか(true:使用する/false:使用しない)
//	 * @deprecated  4.3.5.0 (2009/02/01) JavaScript整理による廃止
	 */
//	@Deprecated public void setUseInputFocus( final String flag ) {
//		useInputFocus = nval( getRequestParameter( flag ),useInputFocus );
//	}

	/**
	 * 【TAG】hilightRow.js JavaScript を使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、hilightRow.js を使用するかどうかを指定します。
	 * result タイプで有効です。
	 * 初期値は、使用する(true)です。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 *
	 * @param	flag	使用するかどうか(true:使用する/false:使用しない)
	 */
//	public void setUseHilightRow( final String flag ) {
//		useHilightRow = nval( getRequestParameter( flag ),useHilightRow );
//	}

	/**
	 * 【TAG】ScrollBar.js JavaScript を使用するかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、ScrollBar.js を使用するかどうかを指定します。
	 * (従来のTYPE2です。)
	 * result タイプで有効です。
	 * 初期値は、使用する(true)です。
	 *
	 * @param	flag	使用するかどうか [true:使用する/false:使用しない]
	 */
	public void setUseScrollBar( final String flag ) {
		useScrollBar = nval( getRequestParameter( flag ),useScrollBar );
	}

	/**
	 * 【TAG】printLine.js JavaScript を使用するかどうかを指定します(初期値:true)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、printLine.js を使用するかどうかを指定します。
	 * result タイプで有効です。
	 * 初期値は、使用する(true)です。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 *
	 * @param	flag	使用するかどうか(true:使用する/false:使用しない)
	 */
//	public void setUsePrintLine( final String flag ) {
//		usePrintLine = nval( getRequestParameter( flag ),usePrintLine );
//	}

	/**
	 * 【TAG】tabstrip.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、tabstrip.js を使用するかどうかを指定します。
	 * init,query,result タイプで有効です。
	 * この機能は特別な為、初期値は、使用しない(false)です。
	 *
	 * @param	flag	使用するかどうか [true:使用する/false:使用しない]
	 */
	public void setUseTabstrip( final String flag ) {
		useTabstrip = nval( getRequestParameter( flag ),useTabstrip );
	}

	/**
	 * 【TAG】option/gantt.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、option/gantt.js を使用するかどうかを指定します。
	 * result タイプで有効です。
	 * この機能は特別な為、初期値は、使用しない(false)です。
	 *
	 * @param	flag	使用するかどうか [true:使用する/false:使用しない]
	 */
	public void setUseGantt( final String flag ) {
		useGantt = nval( getRequestParameter( flag ),useGantt );
	}

	/**
	 * 【TAG】option/adjustGantTable.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、option/adjustGantTable.js を使用するかどうかを指定します。
	 * result タイプで有効です。
	 * これを、true に設定すると、option/adjustGantTable.js 、option/adjustGantTable.css が組み込まれます。
	 * また、useGantt = false; 、useScrollBar = false; も同時に内部設定されます。
	 * この機能は特別な為、初期値は、使用しない(false)です。
	 *
	 * なお、ガントを利用するには、起動時にJavaScriptをロード/実行する必要があるので、
	 * addEvent もセットしておく必要があります。
	 *
	 * adjustTable : テーブル分割
	 * adjustGant  : ガント/積み上げガント
	 * adjustTask  : 積み上げガント（内部積み上げ）
	 *
	 * script として、addEvent( this, "load", adjustGant ); という形式を出力します。
	 *
	 * @og.rev 5.6.2.1 (2013/03/08) 新規追加
	 *
	 * @param	flag	使用するかどうか [true:使用する/false:使用しない]
	 * @see		#setAddEvent( String )
	 */
	public void setUseGantt2( final String flag ) {
		useGantt2 = nval( getRequestParameter( flag ),useGantt2 );
		if( useGantt2 ) {
			useGantt     = false;
			useScrollBar = false;
		}
	}

	/**
	 * 【TAG】option/treeBOM.js JavaScript を使用するかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、option/treeBOM.js を使用するかどうかを指定します。
	 * result タイプで有効です。
	 * この機能は特別な為、初期値は、使用しない(false)です。
	 *
	 * @param	flag	使用するかどうか [true:使用する/false:使用しない]
	 */
	public void setUseTreeBOM( final String flag ) {
		useTreeBOM = nval( getRequestParameter( flag ),useTreeBOM );
	}

	/**
	 * 【TAG】ゲストユーザを利用するかどうか[true/false]を指定します(初期値false)。
	 *
	 * @og.tag
	 *
	 * ユーザIDが指定されていなかった場合にユーザのIDをGUESTとしてログインさせるかを指定します。
	 * TOMCATの認証機能を外した領域でユーザを限定せずにログインさせたい場合等に利用します。
	 * なお、システム定数RESOURCE_USER_SQLで帰ってくる結果に「GUEST」IDのユーザが必要です。
	 * 初期値はfalse(利用しない)となります。
	 *
	 * @param	flag	ゲストユーザを使用するかどうか [true:使用する/false:使用しない]
	 *
	 * @og.rev 4.2.2.0 (2008/05/28) 新規作成
	 */
	public void setUseGuest( final String flag ) {
		useGuest = nval( getRequestParameter( flag ),useGuest );
	}

	/**
	 * 【TAG】Ajaxを利用したツリー形式のテーブル表示を使用するかを指定します。
	 *
	 * @og.tag
	 * Ajaxを利用したツリー形式のテーブル表示を使用するかを指定します。
	 * viewタグのviewFormType="HTMLAjaxTreeTable"を使用する場合は、必ずtrueを指定して下さい。
	 *
	 * また、ツリー形式のテーブルに対して登録を行う場合は、noTransition属性も
	 * 同時にtrueにしておく必要があります。
	 *
	 * 初期値は、false(使用しない)です。
	 *
	 * @og.rev 4.3.1.0 (2008/09/02) 新規追加
	 *
	 * @param   flag Ajaxを利用したツリー形式のテーブル表示を使用するか
	 */
	public void setUseAjaxTreeTable( final String flag ) {
		useAjaxTreeTable = nval( getRequestParameter( flag ), useAjaxTreeTable );
	}

	/**
	 * 【TAG】画面遷移を行わない形式の登録方法を使用するかを指定します。
	 *
	 * @og.tag
	 * 画面遷移なしの登録を行うかどうかを指定します。
	 * trueが指定された場合、画面遷移を行わず選択行のみを動的に切り替えることで、
	 * 編集、登録を行います。サーバーへのデータ送信には、jQueryのFormPluginを
	 * 利用しています。
	 * また、データを送信する対象のフォームには、id="noTransition"を指定して下さい。
	 * (該当のIDのフォームが存在しない場合は、JSPの1番目のフォームが送信されます。)
	 * 初期値は、システムリソースのNO_TRANSITION_MODEで設定することが可能です。
	 *
	 * システムパラメーターまたは、このタグの属性でtrueに指定された場合でも、submitされた
	 * commandがINSERT,COPY,MODIFY,DELETE,ENTRY,UNDOの場合のみ、バックグラウンドで
	 * 実行されます。
	 * それ以外のコマンドでは通常通り、画面遷移を伴うsubmitとなります。
	 *
	 * また、システムパラメーターまたは、このタグの属性でtrueに指定された場合、バックグラウンド
	 * で実行されるJSPには、その値が引き継がれるため、これらのJSPの各タグでtrueを指定する
	 * 必要はありません。
	 *
	 * @og.rev 4.3.1.0 (2008/09/02) 新規追加
	 * @og.rev 4.3.8.0 (2009/08/01) noTransitionにnullを許容する
	 * @og.rev 5.1.3.0 (2010/02/01) リクエストに戻す
	 *
	 * @param   noTrnstn 画面遷移を行わない形式の登録方法を使用するか
	 */
	public void setNoTransition( final String noTrnstn ) {
		noTransition = nval( getRequestParameter( noTrnstn ), noTransition );
	}

	/**
	 * 【TAG】og:tabLinkタグで使用される common/tagLink.js をインクルードするかどうか[true/false]を指定します(初期値:false)。
	 *
	 * @og.tag
	 * og:tabLinkタグで使用される common/tagLink.js をインクルードするかを指定します。
	 * trueが指定された場合、該当のJavaScriptをインクルードします。
	 * og:tabLinkを使用する場合は、必ずtrueを指定して下さい。
	 * 初期値は、false(インクルードしない)です。
	 *
	 * @og.rev 4.3.3.0 (2008/10/01) 新規追加
	 *
	 * @param   flag common/tagLink.js をインクルード [true:する/false:しない]
	 */
	public void setUseTabLink( final String flag ) {
		useTabLink = nval( getRequestParameter( flag ), useTabLink );
	}

	/**
	 * 【TAG】カラムチェックをリアルタイムで行うかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * &lt;del&gt;common/realTimeCheck.jsをインクルードするかの指定です。
	 * インクルードするとカラムチェックがリアルタイムで行われます。
	 * (フォーカスが離れた時に動作)
	 * &lt;del&gt;
	 * リアルタイムチェックイベントを無効にするかどうかを決めます。
	 * falseを指定すると、リアルタイムチェックを行いません。
	 * query,result 両タイプで有効です。
	 * このチェックはDBTypeを用いた簡易的なチェックです。
	 * チェックにかかった場合はバルーンで警告が表示されます。
	 * &lt;del&gt;初期値は、システムリソースのUSE_REAL_TIME_CHECKで設定することが可能です。&lt;del&gt;
	 * 初期値はtrue(利用する)です。
	 *
	 * @og.rev 4.3.5.0 (2009/02/01) 新規追加
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 * @og.rev 5.4.5.0 (2012/03/01) 復活 ローカルリソース時等で使いたくない場合がある
	 *
	 * @param   flag common/realTimeCheck.jsをインクルード [true:する/false:しない]
	 */
	public void setUseRealTimeCheck( final String flag ) {
		useRealTimeCheck = nval( getRequestParameter( flag ), useRealTimeCheck );
	}

	/**
	 * 【TAG】大文字・小文字の自動変換を行うかを指定します。
	 *
	 * @og.tag
	 * common/autoConvert.jsをインクルードするかの指定です。
	 * インクルードするとDBTypeがXL,XUの場合に大文字、小文字の
	 * 自動変換が行われます。(フォーカスが離れた時に動作)
	 * query,result 両タイプで有効です。
	 * 初期値は、システムリソースのUSE_INPUT_AUTO_CONVERTで設定することが可能です。
	 *
	 * @og.rev 4.3.5.0 (2009/02/01) 新規追加
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 *
	 * @param   flag common/autoConvert.jsをインクルードするか
	 */
//	public void setUseAutoConvert( final String flag ) {
//		useAutoConvert = nval( getRequestParameter( flag ), useAutoConvert );
//	}

	/**
	 * 【TAG】フォーカスが当っている入力ボックスの色を変更するかどうか[true/false]を指定します(初期値:true)。
	 *
	 * @og.tag
	 * &lt;del&gt;標準組み込みJavaScriptである、focusHilight.jsをインクルードするかの指定です。
	 * フォーカスが当っている入力ボックスの背景色を変更します。
	 * インクルードすると自動的に有効になります。
	 * &lt;del&gt;
	 * 入力ボックス色変更JavaScriptのイベントを無効にするフラグの制御を行います。
	 * falseを指定するとイベントを無効にします。
	 * ポップアップ時にプルダウンが正常動作しない場合はこれをfalseとしてください。
	 * query,result 両タイプで有効です。
	 * 初期値は使用する(true)です。
	 *
	 * @og.rev 4.3.5.0 (2009/02/01) 新規追加
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 * @og.rev 5.4.5.0 (2012/03/01) モーダルダイアログのプルダウン挙動対応で復活
	 *
	 * @param   flag common/focusHilight.jsをインクルード [true:する/false:しない]
	 */
	public void setUseFocusHilight( final String flag ) {
		useFocusHilight = nval( getRequestParameter( flag ), useFocusHilight );
	}

	/**
	 * 【TAG】フォーカスが当っている入力ボックスの色を変更するかどうか(初期値:true)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、focusFirstElm.jsをインクルードするかの指定です。
	 * 最初の入力ボックスにフォーカスが当たり、フレームの高さを調整します。
	 * インクルードすると自動的に有効になります。
	 * queryタイプで有効です。
	 * 初期値は使用する(true)です。
	 *
	 * @og.rev 4.3.5.0 (2009/02/01) 新規追加
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 *
	 * @param   flag common/focusFirstElm.jsをインクルードするか
	 */
//	public void setUseFocusFirstElm( final String flag ) {
//		useFocusFirstElm = nval( getRequestParameter( flag ), useFocusFirstElm );
//	}

	/**
	 * 【TAG】フレームの高さを調節するかどうか(初期値:true)。
	 *
	 * @og.tag
	 * 標準組み込みJavaScriptである、default.jsの関数adjustFrameHeightを実行するかの指定です。
	 * trueとするとイベント登録JavaScriptのadjustFrameHeightEvent.jsがインクルードされます。
	 * インクルードされると自動的に有効になります。
	 * queryタイプで有効です。
	 * 初期値は使用する(true)です。
	 *
	 * @og.rev 4.3.5.0 (2009/02/01) 新規追加
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 *
	 * @param   flag common/adjustFrameHeightEvent.jsをインクルードするか
	 */
//	public void setUseAdjustFrameHeight( final String flag ) {
//		useAdjustFrameHeight = nval( getRequestParameter( flag ), useAdjustFrameHeight );
//	}

	/**
	 * 【TAG】親子カラム設定用JSをインクルードするか(初期値:true)。
	 *
	 * @og.tag
	 * 親子プルダウン等で使用される common/makeColumn.js をインクルードするかを指定します。
	 * trueが指定された場合、該当のJavaScriptをインクルードします。
	 * eventColumnを使用する場合は、必ずtrueを指定して下さい。
	 * 初期値は、true(インクルードする)です。
	 *
	 * @og.rev 4.3.6.0 (2009/04/01)
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 *
	 * @param   flag common/makeColumn.jsをインクルードするか
	 */
//	public void setUseEventColumn( final String flag ) {
//		useEventColumn = nval( getRequestParameter( flag ), useEventColumn );
//	}

	/**
	 * 【TAG】IE8でIE7エミュレートモードで動作させるタグを出力するかどうか[true/false]を指定します
	 *		(初期値:USE_IE7_HEADER[={@og.value org.opengion.hayabusa.common.SystemData#USE_IE7_HEADER}])。
	 *
	 * @og.tag
	 * IE8でIE7エミュレートモード動作をさせるためのMETAタグを出力するか指定します。
	 * (初期値:システム定数のUSE_IE7_HEADER[={@og.value org.opengion.hayabusa.common.SystemData#USE_IE7_HEADER}])。
	 *
	 * @og.rev 4.3.7.0 (2009/06/01)
	 *
	 * @param   flag IE7エミュレートモード用METAタグを吐くか [true:出力する/false:出力しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_IE7_HEADER
	 */
	public void setUseIE7Header( final String flag ) {
		useIE7Header = nval( getRequestParameter( flag ), useIE7Header );
	}
	
	/**
	 * 【TAG】タブレット等の画面サイズ調整METAタグ(viewport）を吐くかどうか[true/false]を指定します
	 *		(初期値:USE_VIEWPORT_HEADER[={@og.value org.opengion.hayabusa.common.SystemData#USE_VIEWPORT_HEADER}])。
	 *
	 * @og.tag
	 * タブレット等の画面サイズ調整METAタグ(viewport）を吐くかどうか[true/false]を指定します。
	 * trueを指定すると、METAタグで
	 * content="width=device-width, initial-scale=1" name="viewport"
	 * の値が出力されます。
	 * (初期値:システム定数のUSE_VIEWPORT_HEADER[={@og.value org.opengion.hayabusa.common.SystemData#USE_VIEWPORT_HEADER}])。
	 *
	 * @og.rev 5.5.8.0 (2012/11/01) 新規作成
	 *
	 * @param   flag タブレット画面サイズ調整METAタグを吐くか [true:出力する/false:出力しない]
	 * @see		org.opengion.hayabusa.common.SystemData#USE_VIEWPORT_HEADER
	 */
	public void setUseViewPort( final String flag ) {
		useViewPort = nval( getRequestParameter( flag ), useViewPort );
	}

	/**
	 * 【TAG】フォームのSubmit処理をJavaScriptを利用して行うか(ポップアップエラー表示)。
	 *
	 * @og.tag
	 * この属性をtrueにセットする事で、ajaxを利用したsubmitを行います。
	 * submit後のページでエラーが発生した場合は画面が切り替わらずに
	 * ポップアップ形式でエラー表示されます(command=ENTRYの場合のみ)。
	 * 動的カラム(eventColumn属性)をRESULT部分で利用する場合は、戻るボタンが
	 * 正常に動作しないため、この属性をtrueにする方が望ましいです。
	 *
	 * データを送信する対象のフォームには、id="ajaxSubmit"を指定して下さい。
	 * (該当のIDのフォームが存在しない場合は、JSPの1番目のフォームが送信されます。)
	 *
	 * システムパラメーターまたは、このタグの属性でtrueに指定された場合でも、submitされた
	 * commandがENTRYの場合のみ、バックグラウンドで実行されます。
	 * それ以外のコマンドでは通常通り、画面遷移を伴うsubmitとなります。
	 *
	 * また、システムパラメーターまたは、このタグの属性でtrueに指定された場合、バックグラウンド
	 * で実行されるJSPには、その値が引き継がれるため、これらのJSPの各タグでtrueを指定する
	 * 必要はありません。
	 *
	 * 初期値はシステムパラメータの「USE_AJAX_SUBMIT」です。
	 *
	 * @og.rev 4.3.8.0 (2009/08/01) 新規追加
	 * @og.rev 5.1.3.0 (2010/02/01) リクエストに戻す
	 *
	 * @param   flag ajaxSubmitを行うかどうか
	 */
	public void setUseAjaxSubmit( final String flag ) {
		useAjaxSubmit = nval( getRequestParameter( flag ), useAjaxSubmit );
	}

	
	/**
	 * 【TAG】javaScript を初期ロードし、起動するメソッド名を指定します(初期値:null)。
	 *
	 * @og.tag
	 * これは、addEvent( this, "load", method ); をJavaScript として組み込みます。
	 * 現時点では、useGantt2 で、adjustTable、adjustGant、adjustTask を設定する場合に使います。
	 * 当然、汎用的に使う事も可能です。
	 *
	 * @og.rev 5.6.3.0 (2013/04/01) 新規作成
	 *
	 * @param   method javaScriptを初期ロードし、起動するメソッド名
	 * @see		#setUseGantt2( String )
	 */
	public void setAddEvent( final String method ) {
		addEvent = nval( getRequestParameter( method ), addEvent );
	}

	/**
	 * 【TAG】viewで出力される件数[1-4/4]の表示場所をヘッダにするか。
	 *
	 * @og.tag
	 * データ表示場所の確保のために、ビューの件数表示場所を
	 * JavaScriptを利用してヘッダ部分に移します。
	 * 複数のviewがある場合は最後のviewの件数になります。
	 * 初期値はシステムパラメータの「USE_ROWCNT_REPLACE」です。
	 *
	 * @og.rev 4.3.8.0 (2009/08/01)
	 * @og.rev 5.2.1.0 (2010/10/01) JavaScript の統合、廃止
	 *
	 * @param   flag viewの件数表示場所を変更するか
	 */
//	public void setUseRowCountReplace( final String flag ) {
//		useRowCountReplace = nval( getRequestParameter( flag ), useRowCountReplace );
//	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 */
	@Override
	public String toString() {
		return org.opengion.fukurou.util.ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "body"		,body		)
				.println( "title"		,title		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
