/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.xml;

import java.util.Map;

/**
 * JSP/XMLファイルを読み取って、構築される最上位の OGDocument オブジェクト を定義します。
 *
 * JspSaxParser での read/write で構築される トップレベルの OGNode になります。
 *
 * @og.rev 5.1.8.0 (2010/07/01) 新規作成
 *
 * @version  5.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK6.0,
 */
public class OGDocument extends OGNode {
	private Map<String,OGElement> idMap = null;		// 5.1.9.0 (2010/08/01)
	private String file   = null;
	private String encode = "UTF-8";				// 基本的には、XML は、UTF-8 で扱うのが良い。
	private String nameSpace = null;				// 5.2.1.0 (2010/10/01) トップエレメントの名前空間のキャッシュ

	/**
	 * デフォルトコンストラクター
	 *
	 * NodeType を Document に設定し、デフォルトの OGNode を構築します。
	 *
	 */
	public OGDocument() {
		super();
		setNodeType( OGNodeType.Document );
	}

	/**
	 * 最上位の エレメントを返します。
	 *
	 * ノードが、まだ登録されていない、または、エレメントノードでない場合は、null が返されます。
	 *
	 * @return	最上位のエレメント
	 */
	public OGElement getTopElement() {
		if( nodeSize() > 0 ) {
			OGNode node = getNode( 0 );
			if( node.getNodeType() == OGNodeType.Element ) {
				return (OGElement)node;
			}
		}
		return null ;
	}

	/**
	 * このドキュメントの読み込み元のファイル名をセットします。
	 *
	 * ドキュメントの処理の可否を判断したり、デバッグ時に使用します。
	 *
	 * @param	name	読み込んだファイル名
	 */
	public void setFilename( final String name ) {
		file = name;
	}

	/**
	 * このドキュメントの読み込み元のファイル名を取得します。
	 *
	 * ドキュメントの処理の可否を判断したり、デバッグ時に使用します。
	 *
	 * @return	読み込んだファイル名
	 */
	public String getFilename() {
		return file;
	}

	/**
	 * このドキュメントのXML宣言のエンコード属性をセットします(初期値:UTF-8)。
	 *
	 * ドキュメントの出力時の XML 宣言に付与します。
	 * null の設定が可能です。 null をセットすると、XML宣言出力時に、
	 * encode 属性になにもセットしません。
	 *
	 * @param	encode	XML宣言のエンコード属性
	 */
	public void setEncode( final String encode ) {
		this.encode = encode;
	}

	/**
	 * このドキュメントのXML宣言のエンコード属性を取得します(初期値:UTF-8)。
	 *
	 * ドキュメントの出力時の XML 宣言に付与します。
	 * encode 属性になにもセットされていない場合は、null が返されます。
	 *
	 * @return	XML宣言のエンコード属性
	 */
	public String getEncode() {
		return encode;
	}

	/**
	 * このドキュメントの "hybs-taglib.tld" の nameSpace(名前空間)を返します。
	 *
	 * 通常は、標準エンジンの場合は "og" になります。
	 * 社内システムの場合は、互換性の関係から、"mis" になっています。
	 * Ver3 の時代は、"mis-taglib.tld" だったので、この処理はできません。
	 *
	 * @og.rev 5.2.1.0 (2010/10/01) 新規作成
	 *
	 * @return	名前空間(存在しないときは、ゼロ文字列)
	 */
	public String getNameSpace() {
		if( nameSpace != null ) { return nameSpace; }

		nameSpace = "" ;		// 存在しないときは、ゼロ文字列になる。
		OGElement ele = getTopElement();
		if( ele != null ) {
			OGAttributes attri = ele.getOGAttributes();
			for( int i=0; i<attri.size(); i++ ) {
				if( "hybs-taglib.tld".equals( attri.getVal(i) ) ) {
					String key = attri.getKey(i);			// xmlns:og , xmlns:mis などのはず
					int cm = key.indexOf( ':' );
					nameSpace = key.substring( cm+1 );			// : は削除している。"og" など
					break;
				}
			}
		}
		return nameSpace;
	}

	/**
	 * このドキュメントに含まれる id に対する、エレメントのマップをセットします。
	 *
	 * id 指定されているエレメントは、特別にキャッシュされます。
	 * また、ドキュメント中には、id は、一意に設定されていることとします。
	 * 複数、おなじ id が存在する場合は、(保障されませんが)性質上、最後に登録された
	 * エレメントが、設定されます。
	 *
	 * @param	map	エレメントのマップ
	 */
	public void setIdMap( final Map<String,OGElement> map ) {
		idMap = map;
	}

	/**
	 * 指定の id に対するエレメントを返します。
	 *
	 * id に対するエレメントが存在しない場合は、null が返されます。
	 *
	 * @param	id	指定のID
	 *
	 * @return	OGエレメント
	 */
	public OGElement getElementById( final String id ) {
		OGElement ele = null;

		if( idMap != null ) { ele = idMap.get( id ); }

		return ele ;
	}

	/**
	 * オブジェクトの文字列表現を返します。
	 *
	 * ここでは、ドキュメントにXML宣言をつけた状態の文字列を返します。
	 * 基本的には、ドキュメントには、一つだけ エレメントを含むことになります。
	 * よって、XML宣言をつけない状態で出力する場合は、#getTopElement() で
	 * エレメントを取得して、それを、toString() してください。
	 *
	 * @return	このオブジェクトの文字列表現
	 * @see OGNode#toString()
	 * @see #getTopElement()
	 */
	@Override
	public String toString() {
		StringBuilder buf = new StringBuilder();

		buf.append( "<?xml version=\"1.0\"" );
		if( encode != null ) {
			buf.append( " encoding=\"" ).append( encode ).append( "\"" );
		}
		buf.append( "?>" ).append( CR ).append( super.toString() );

		return buf.toString() ;
	}
}
