/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.midiMotion.gui;

import java.awt.Color;
import java.awt.datatransfer.Transferable;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.TransferHandler;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sourceforge.mmd.motion.Motion;
import jp.sourceforge.mmd.motion.swing.FileDropEvent;
import jp.sourceforge.mmd.motion.swing.FileDropListener;
import jp.sourceforge.mmd.motion.swing.FileDropTarget;
import jp.sourceforge.mmd.motion.swing.MmdDialog;

/**
 * MMDモーションデータを表示・ドラック &amp; ドロップするための swing JTextField クラス.
 * @author nazo
 * @since 1.2
 */
public class MotionTextBox extends JTextField implements FileDropListener,MouseListener {
    private class FileTransferHandler extends TransferHandler {
        @Override
        public int getSourceActions(JComponent c) {
            return TransferHandler.COPY;
        }
        @Override
        protected Transferable createTransferable(JComponent c){
            return new FileTransferable(orgFile);
        }
    }

    private Motion motion;
    /**
     * motion の元ファイル(得られる場合)
     * @since 1.2
     */
    protected File orgFile;

    public MotionTextBox(){
        super();
        orgFile = null;
        motion = null;
        setEditable(false);
        setBackground(Color.CYAN);
        setText("(まだ)");
        setToolTipText("MMDモーション.VMD,CSVをドロップ可能.");
        setDropTarget(new FileDropTarget(this));
        addMouseListener(this);
        setDragEnabled(false);
        setTransferHandler(new FileTransferHandler());
    }

    public void setMotion(Motion m){
        motion=m;
        if(m==null){
            orgFile=null;
            setDragEnabled(false);
            setText("(まだ)");
            setToolTipText("MMDモーション.VMD,CSVをドロップ可能.");
        } else {
            setText(m.getModelName());
            setToolTipText("<html>"+m.getModelName()+":<br>"+m.getMaxFrame()+" frames<br>file:"+((orgFile==null)?"なし":orgFile.getPath())+"</html>");
        }
    }

    public void setMotion(File file){
        if(file==null){
            setMotion((Motion)null);
            return;
        }

        FileInputStream fis=null;
        Motion m=null;
        String name=file.getName().toLowerCase();
        try {
            fis=new FileInputStream(file);
            if(name.endsWith(".vmd")){
                m=new Motion().fromVMD(fis);
            }else if(name.endsWith(".csv")){
                m=new Motion().fromVMD(fis);
            }
            fis.close();
        } catch(FileNotFoundException ex){
            JOptionPane.showMessageDialog(this, "ファイルが見つかりません: "+ ex.getLocalizedMessage());
        } catch(IOException ex){
            JOptionPane.showMessageDialog(this, "ファイルが読み取れません: "+ ex.getLocalizedMessage());
        } catch(MmdFormatException ex){
            JOptionPane.showMessageDialog(this, "MMDのモーションファイルではありません: "+ ex.getLocalizedMessage());
            try{fis.close();}catch(IOException ex2){
                JOptionPane.showMessageDialog(this, "ファイルが読み取れません: "+ ex2.getLocalizedMessage());
            }
        }
        if(m==null)return;
        orgFile=file;
        setDragEnabled(true);
        setMotion(m);
    }

    public Motion getMotion(){
        return motion;
    }

    /**
     * 得られる場合に限り, モーションファイルを取得する.
     * @return 得られないときは{@code null}.
     */
    public File getFile(){
        return orgFile;
    }

    public void openDialog(){
        Motion m=MmdDialog.loadMotion(getParent());
        if(m==null)return;
        orgFile=MmdDialog.lastSelected;
        setDragEnabled(true);
        setMotion(m);
    }

    @Override
    public void fileDrop(FileDropEvent e) {
        File file=e.getFiles()[0];
        setMotion(file);
    }

    @Override
    public void mouseClicked(MouseEvent e) {
        if(e.getClickCount()>1){
            if(e.getButton()==MouseEvent.BUTTON3){// right double clicked
                setMotion((Motion)null);
            } else if(e.getButton()==MouseEvent.BUTTON1) {
                openDialog();
            }
        }
    }

    @Override
    public void mousePressed(MouseEvent e) {
        selectAll();
    }

    @Override
    public void mouseReleased(MouseEvent e) {}

    @Override
    public void mouseEntered(MouseEvent e) {}

    @Override
    public void mouseExited(MouseEvent e) {}
}
