/*
 * The MIT License
 *
 * Copyright 2015 nazo.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package jp.sourceforge.mmd.motion;

import jp.sourceforge.mmd.motion.geo.Matrix;
import jp.sourceforge.mmd.motion.geo.Vector3D;
import jp.sfjp.mikutoga.bin.parser.MmdFormatException;
import jp.sfjp.mikutoga.bin.parser.ParseStage;
import jp.sfjp.mikutoga.vmd.parser.VmdBasicHandler;
import jp.sfjp.mikutoga.vmd.parser.VmdBoolHandler;
import jp.sfjp.mikutoga.vmd.parser.VmdCameraHandler;
import jp.sfjp.mikutoga.vmd.parser.VmdLightingHandler;

/**
 *
 * @author nazo
 */
public class VMDMotionHander implements VmdBasicHandler,VmdCameraHandler,
        VmdLightingHandler,VmdBoolHandler{
    private final Motion motion;
    private BonePose bonePose;
    private CameraPose cameraPose;
    private LightPose lightPose;
    private ShadowPose shadowPose;
    private BooleanPose booleanPose;
    private int offset;

    public VMDMotionHander(Motion motion,int offset){
        super();
        this.motion=motion;
        this.offset=offset;
    }
    @Override
    public void vmdParseStart() throws MmdFormatException {
        bonePose=null;
        cameraPose=null;
        lightPose=null;
        shadowPose=null;
        booleanPose=null;
    }

    @Override
    public void vmdHeaderInfo(byte[] header) throws MmdFormatException {
        String a=new String(header,0,26);
        if(!a.startsWith("Vocaloid Motion Data 0002")){
            throw new MmdFormatException("Header error",0);
        }
    }

    @Override
    public void vmdModelName(String modelName) throws MmdFormatException {
        motion.setModelName(modelName);
    }

    @Override
    public void loopStart(ParseStage stage, int loops) throws MmdFormatException {
        //notiong
    }

    @Override
    public void vmdBoneMotion(String boneName, int keyFrameNo) throws MmdFormatException {
        bonePose=new BonePose();
        bonePose.frame=keyFrameNo+offset;
        bonePose.nameOfBone=boneName;
    }

    @Override
    public void vmdBonePosition(float x, float y, float z) throws MmdFormatException {
        bonePose.v=new Vector3D(x,y,z);
    }

    @Override
    public void vmdBoneRotationQt(float qx, float qy, float qz, float qw) throws MmdFormatException {
        bonePose.mr=Matrix.rotationQ(qx, qy, qz, qw);
    }

    @Override
    public void vmdBoneIntpltXpos(byte xP1x, byte xP1y, byte xP2x, byte xP2y) throws MmdFormatException {
        bonePose.interpX=new byte []{xP1x,xP1y,xP2x,xP2y};
    }

    @Override
    public void vmdBoneIntpltYpos(byte yP1x, byte yP1y, byte yP2x, byte yP2y) throws MmdFormatException {
        bonePose.interpY=new byte []{yP1x,yP1y,yP2x,yP2y};
    }

    @Override
    public void vmdBoneIntpltZpos(byte zP1x, byte zP1y, byte zP2x, byte zP2y) throws MmdFormatException {
        bonePose.interpZ=new byte []{zP1x,zP1y,zP2x,zP2y};
    }

    @Override
    public void vmdBoneIntpltRot(byte rP1x, byte rP1y, byte rP2x, byte rP2y) throws MmdFormatException {
        bonePose.interpR=new byte []{rP1x,rP1y,rP2x,rP2y};
    }

    @Override
    public void vmdMorphMotion(String morphName, int keyFrameNo, float f) throws MmdFormatException {
        MorphPose p=new MorphPose();
        p.frame=keyFrameNo+offset;
        p.nameOfBone=morphName;
        p.factor=f;
        motion.put(p);
    }

    @Override
    public void loopNext(ParseStage stage) throws MmdFormatException {
        if(bonePose!=null){
            motion.put(bonePose);
            bonePose=null;
        }
        if(cameraPose!=null){
            motion.put(cameraPose);
            cameraPose=null;
        }
        if(lightPose!=null){
            motion.put(lightPose);
            lightPose=null;
        }
        if(shadowPose!=null){
            motion.put(shadowPose);
            shadowPose=null;
        }
        if(booleanPose!=null){
            motion.put(booleanPose);
            booleanPose=null;
        }
    }

    @Override
    public void loopEnd(ParseStage stage) throws MmdFormatException {
    }
    
    @Override
    public void vmdParseEnd(boolean hasMoreData) throws MmdFormatException {
        //nothing
    }

    @Override
    public void vmdCameraMotion(int keyFrameNo) throws MmdFormatException {
        cameraPose=new CameraPose();
        cameraPose.frame=keyFrameNo+offset;
    }

    @Override
    public void vmdCameraRange(float range) throws MmdFormatException {
        cameraPose.range=range;
    }

    @Override
    public void vmdCameraPosition(float xPos, float yPos, float zPos) throws MmdFormatException {
        cameraPose.v=new Vector3D(xPos,yPos,zPos);
    }

    @Override
    public void vmdCameraRotation(float latitude, float longitude, float roll) throws MmdFormatException {
        cameraPose.lat=latitude;
        cameraPose.lon=longitude;
        cameraPose.roll=roll;
    }

    @Override
    public void vmdCameraProjection(int angle, boolean hasPerspective) throws MmdFormatException {
        cameraPose.angle=angle;
        cameraPose.perspective=hasPerspective;
    }

    @Override
    public void vmdCameraIntpltXpos(byte p1x, byte p1y, byte p2x, byte p2y) throws MmdFormatException {
        cameraPose.interpX=new byte []{p1x,p1y,p2x,p2y};
    }

    @Override
    public void vmdCameraIntpltYpos(byte p1x, byte p1y, byte p2x, byte p2y) throws MmdFormatException {
        cameraPose.interpY=new byte []{p1x,p1y,p2x,p2y};
    }

    @Override
    public void vmdCameraIntpltZpos(byte p1x, byte p1y, byte p2x, byte p2y) throws MmdFormatException {
        cameraPose.interpZ=new byte []{p1x,p1y,p2x,p2y};
    }

    @Override
    public void vmdCameraIntpltRotation(byte p1x, byte p1y, byte p2x, byte p2y) throws MmdFormatException {
        cameraPose.interpR=new byte []{p1x,p1y,p2x,p2y};
    }

    @Override
    public void vmdCameraIntpltRange(byte p1x, byte p1y, byte p2x, byte p2y) throws MmdFormatException {
        cameraPose.interpRange=new byte []{p1x,p1y,p2x,p2y};
    }

    @Override
    public void vmdCameraIntpltProjection(byte p1x, byte p1y, byte p2x, byte p2y) throws MmdFormatException {
        cameraPose.interpProjection=new byte []{p1x,p1y,p2x,p2y};
    }

    @Override
    public void vmdLuminousMotion(int keyFrameNo) throws MmdFormatException {
        lightPose=new LightPose();
        lightPose.frame=keyFrameNo+offset;
    }

    @Override
    public void vmdLuminousColor(float rVal, float gVal, float bVal) throws MmdFormatException {
        lightPose.rgb=new float[]{rVal,gVal,bVal};
    }

    @Override
    public void vmdLuminousDirection(float xVec, float yVec, float zVec) throws MmdFormatException {
        lightPose.v=new Vector3D(xVec,yVec,zVec);
    }

    @Override
    public void vmdShadowMotion(int keyFrameNo) throws MmdFormatException {
        shadowPose=new ShadowPose();
        shadowPose.frame=keyFrameNo+offset;
    }

    @Override
    public void vmdShadowMode(byte shadowMode) throws MmdFormatException {
        shadowPose.mode=shadowMode;
    }

    @Override
    public void vmdShadowScopeRaw(float shadowScope) throws MmdFormatException {
        shadowPose.distance=shadowScope;
    }

    @Override
    public void vmdModelSight(boolean show, int keyFrameNo) throws MmdFormatException {
        booleanPose=new BooleanPose();
        booleanPose.show=show;
        booleanPose.frame=keyFrameNo+offset;
    }

    @Override
    public void vmdIkSwitch(String boneName, boolean validIk, int keyFrameNo) throws MmdFormatException {
        if(booleanPose==null){
            booleanPose=new BooleanPose();
            booleanPose.frame=keyFrameNo+offset;
        }
        booleanPose.iks.put(boneName, validIk);
    }
}
