﻿/*
 * 
 *  Copyright (c) 2012 Jarrett Webb & James Ashley
 * 
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated 
 *  documentation files (the "Software"), to deal in the Software without restriction, including without limitation 
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, 
 *  and to permit persons to whom the Software is furnished to do so, subject to the following conditions:
 *  
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED 
 *  TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF 
 *  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 *  IN THE SOFTWARE.
 * 
 * 
 */


using System.Linq;
using System.Windows;
using System.Windows.Media;
using System.Windows.Shapes;

using Microsoft.Kinect;



namespace BeginningKinect.Chapter4.SkeletonViewer
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window
    {
        #region メンバー変数
        private KinectSensor _KinectDevice;
        private readonly Brush[] _SkeletonBrushes;
        private Skeleton[] _FrameSkeletons;        
        #endregion メンバー変数


        #region コンストラクタ
        public MainWindow()
        {
            InitializeComponent();

            _SkeletonBrushes = new Brush[] { Brushes.Black, 
                                             Brushes.Crimson, 
                                             Brushes.Indigo, 
                                             Brushes.DodgerBlue, 
                                             Brushes.Purple,
                                             Brushes.Pink };

            KinectSensor.KinectSensors.StatusChanged += KinectSensors_StatusChanged;
            this.KinectDevice = KinectSensor.KinectSensors.FirstOrDefault(x => x.Status == KinectStatus.Connected);
        }
        #endregion コンストラクタ


        #region メソッド
        private void KinectSensors_StatusChanged(object sender, StatusChangedEventArgs e)
        {
            switch (e.Status)
            {
                case KinectStatus.Initializing:
                case KinectStatus.Connected:
                case KinectStatus.NotPowered:
                case KinectStatus.NotReady:
                case KinectStatus.DeviceNotGenuine:
                    this.KinectDevice = e.Sensor;                                        
                    break;
                case KinectStatus.Disconnected:
                    //TODO：Kinectデバイスを接続するようユーザーに求める    
                    this.KinectDevice = null;
                    break;
                default:
                    //TODO：エラー状態を表示する
                    break;
            }
        }


        // Listing 4-2
        private void KinectDevice_SkeletonFrameReady(object sender, SkeletonFrameReadyEventArgs e)
        {
            using(SkeletonFrame frame = e.OpenSkeletonFrame())
            {
                if(frame != null)
                {
                    Polyline figure;            
                    Brush userBrush;
                    Skeleton skeleton;
                    
                    LayoutRoot.Children.Clear();
                    frame.CopySkeletonDataTo(this._FrameSkeletons);  
                    
                    Skeleton[] dataSet2 = new Skeleton[this._FrameSkeletons.Length];
                    frame.CopySkeletonDataTo(dataSet2);
                                  

                    for(int i = 0; i < this._FrameSkeletons.Length; i++)
                    {
                        skeleton = this._FrameSkeletons[i];
                        
                        if(skeleton.TrackingState == SkeletonTrackingState.Tracked)                
                        {
                            userBrush = this._SkeletonBrushes[i % this._SkeletonBrushes.Length];

                            //骨格の頭と胴体を描画する
                            figure = CreateFigure(skeleton, userBrush, new [] { JointType.Head, 
                                                                                JointType.ShoulderCenter,
                                                                                JointType.ShoulderLeft, 
                                                                                JointType.Spine,
                                                                                JointType.ShoulderRight,
                                                                                JointType.ShoulderCenter,
                                                                                JointType.HipCenter
                                                                                });              
                            LayoutRoot.Children.Add(figure);

                            figure = CreateFigure(skeleton, userBrush, new [] { JointType.HipLeft,
                                                                                JointType.HipRight });
                            LayoutRoot.Children.Add(figure);

                            //骨格の左足を描画する
                            figure = CreateFigure(skeleton, userBrush, new [] { JointType.HipCenter,
                                                                                JointType.HipLeft,
                                                                                JointType.KneeLeft,
                                                                                JointType.AnkleLeft,
                                                                                JointType.FootLeft });
                            LayoutRoot.Children.Add(figure);

                            //骨格の右足を描画する
                            figure = CreateFigure(skeleton, userBrush, new [] { JointType.HipCenter,
                                                                                JointType.HipRight,
                                                                                JointType.KneeRight,
                                                                                JointType.AnkleRight,
                                                                                JointType.FootRight });                
                            LayoutRoot.Children.Add(figure);

                            //骨格の左腕を描画する
                            figure = CreateFigure(skeleton, userBrush, new [] { JointType.ShoulderLeft,
                                                                                JointType.ElbowLeft,
                                                                                JointType.WristLeft,
                                                                                JointType.HandLeft });
                            LayoutRoot.Children.Add(figure);

                            //骨格の右腕を描画する
                            figure = CreateFigure(skeleton, userBrush, new[] { JointType.ShoulderRight,
                                                                               JointType.ElbowRight,
                                                                               JointType.WristRight,
                                                                               JointType.HandRight });
                            LayoutRoot.Children.Add(figure);                                            
                        }
                    }
                }                
            }                       
        }


        // Listing 4-3
        private Polyline CreateFigure(Skeleton skeleton, Brush brush, JointType[] joints)
        {
            Polyline figure = new Polyline();

            figure.StrokeThickness  = 8;
            figure.Stroke           = brush;

            for(int i = 0; i < joints.Length; i++)
            {                
                figure.Points.Add(GetJointPoint(skeleton.Joints[joints[i]]));
            }

            return figure;
        }

                                                                                  
        private Point GetJointPoint(Joint joint)
        {
            DepthImagePoint point = this.KinectDevice.MapSkeletonPointToDepth(joint.Position,
                                                                              this.KinectDevice.DepthStream.Format);
            point.X *= (int) this.LayoutRoot.ActualWidth / KinectDevice.DepthStream.FrameWidth;
            point.Y *= (int) this.LayoutRoot.ActualHeight / KinectDevice.DepthStream.FrameHeight;
                        
            return new Point(point.X, point.Y);
        }      
        #endregion メソッド


        #region プロパティ
        public KinectSensor KinectDevice 
        {
            get { return this._KinectDevice; }
            set
            {
                if(this._KinectDevice != value)
                {
                    //リソースを解放する
                    if(this._KinectDevice != null)
                    {
                        this._KinectDevice.Stop();
                        this._KinectDevice.SkeletonFrameReady -= KinectDevice_SkeletonFrameReady;
                        this._KinectDevice.SkeletonStream.Disable();
                        this._FrameSkeletons = null;
                    }
                   
                    this._KinectDevice = value;

                    //初期化する
                    if(this._KinectDevice != null)
                    {
                        if(this._KinectDevice.Status == KinectStatus.Connected)
                        {
                            this._KinectDevice.SkeletonStream.Enable();                        
                            this._FrameSkeletons = new Skeleton[this._KinectDevice.SkeletonStream.FrameSkeletonArrayLength];
                            this.KinectDevice.SkeletonFrameReady += KinectDevice_SkeletonFrameReady;
                            this._KinectDevice.Start();                             
                        }
                    }                
                }
            }
        }        
        #endregion プロパティ
    }
}
