/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Fine Kernel Project, All rights reserved.
 *
 *	Redistribution and use in source and binary forms,
 *	with or without modification, are permitted provided that the
 *	following conditions are met:
 *
 *		- Redistributions of source code must retain the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer.
 *
 *		- Redistributions in binary form must reproduce the above
 *			copyright notice, this list of conditions and the
 *			following disclaimer in the documentation and/or
 *			other materials provided with the distribution.
 *
 *		- Neither the name of the copyright holders nor the names
 *			of its contributors may be used to endorse or promote
 *			products derived from this software without specific
 *			prior written permission.
 *
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *	"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *	LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *	FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *	COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *	INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *	HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 *	STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 *	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *	POSSIBILITY OF SUCH DAMAGE. 
 *
 ****************************************************************************/
/****************************************************************************
 *
 *	Copyright (c) 1999-2009, Fine Kernel Project, All rights reserved.
 *
 *	本ソフトウェアおよびソースコードのライセンスは、基本的に
 *	「修正 BSD ライセンス」に従います。以下にその詳細を記します。
 *
 *	ソースコード形式かバイナリ形式か、変更するかしないかを問わず、
 *	以下の条件を満たす場合に限り、再頒布および使用が許可されます。
 *
 *	- ソースコードを再頒布する場合、上記の著作権表示、本条件一覧、
 *		および下記免責条項を含めること。
 *
 *	- バイナリ形式で再頒布する場合、頒布物に付属のドキュメント等の
 *		資料に、上記の著作権表示、本条件一覧、および下記免責条項を
 *		含めること。
 *
 *	- 書面による特別の許可なしに、本ソフトウェアから派生した製品の
 *		宣伝または販売促進に、本ソフトウェアの著作権者の名前または
 *		コントリビューターの名前を使用してはならない。
 *
 *	本ソフトウェアは、著作権者およびコントリビューターによって「現
 *	状のまま」提供されており、明示黙示を問わず、商業的な使用可能性、
 *	および特定の目的に対する適合性に関す暗黙の保証も含め、またそれ
 *	に限定されない、いかなる保証もないものとします。著作権者もコン
 *	トリビューターも、事由のいかんを問わず、損害発生の原因いかんを
 *	問わず、かつ責任の根拠が契約であるか厳格責任であるか(過失その
 *	他の)不法行為であるかを問わず、仮にそのような損害が発生する可
 *	能性を知らされていたとしても、本ソフトウェアの使用によって発生
 *	した(代替品または代用サービスの調達、使用の喪失、データの喪失、
 *	利益の喪失、業務の中断も含め、またそれに限定されない)直接損害、
 *	間接損害、偶発的な損害、特別損害、懲罰的損害、または結果損害に
 *	ついて、一切責任を負わないものとします。
 *
 ****************************************************************************/
#include <FK/VRMLOut.H>
#include <FK/Vertex.h>
#include <FK/Loop.h>
#include <FK/Solid.h>
#include <FK/IndexFace.h>
#include <FK/Material.h>
#include <FK/Error.H>

using namespace std;


fk_VRMLOut::fk_VRMLOut(fk_Solid *argSolid)
{
	solid = argSolid;
	ifs = (fk_IndexFaceSet *)NULL;

	return;
}

fk_VRMLOut::fk_VRMLOut(fk_IndexFaceSet *argIFS)
{
	ifs = argIFS;
	solid = (fk_Solid *)NULL;

	return;
}

fk_VRMLOut::~fk_VRMLOut()
{
	return;
}

bool fk_VRMLOut::WriteVRMLFile(string argFileName, fk_Material *argMaterial,
							   bool triFlag)
{
	FILE		*writeFP;

	if(solid == (fk_Solid *)NULL && ifs == (fk_IndexFaceSet *)NULL) {
		return false;
	}

	if((writeFP = fopen(argFileName.c_str(), "w")) == (FILE *)NULL) {
		return false;
	}

	WriteVRMLHeader(writeFP);
	WriteVRMLShape(writeFP, argMaterial, triFlag,
				   (vector<double> *)NULL, (vector<fk_Vector> *)NULL);
	WriteVRMLFooter(writeFP, FK_VRML_NONE);

	fclose(writeFP);

	return true;
}

bool fk_VRMLOut::WriteVRMLFile(string argFileName,
							   vector<double> *argTime,
							   vector<fk_Vector> *argPos,
							   fk_Material *argMaterial,
							   bool triFlag)
{
	FILE		*writeFP;

	if((writeFP = fopen(argFileName.c_str(), "w")) == (FILE *)NULL) {
		return false;
	}

	WriteVRMLHeader(writeFP);
	WriteVRMLShape(writeFP, argMaterial, triFlag, argTime, argPos);
	WriteVRMLFooter(writeFP, FK_VRML_INTR);

	fclose(writeFP);

	return true;
}

void fk_VRMLOut::WriteVRMLHeader(FILE *argFP)
{
	fprintf(argFP, "#VRML V2.0 utf8\n\n");
	fprintf(argFP, "Transform {\n");
	fprintf(argFP, "\tchildren [\n");

	return;
}

void fk_VRMLOut::WriteVRMLShape(FILE *argFP,
								fk_Material *argMaterial, bool triFlag,
								vector<double> *argTime,
								vector<fk_Vector> *argPos)
{
	WriteVRMLLightInfo(argFP);
	WriteVRMLNaviInfo(argFP);

	if(argTime != (vector<double> *)NULL &&
	   argPos != (vector<fk_Vector> *)NULL) {
		WriteVRMLCoordInterp(argFP, argTime, argPos);
	}

	fprintf(argFP, "\t\tShape {\n");

	WriteVRMLMaterial(argFP, argMaterial);
	if(solid != (fk_Solid *)NULL) {
		WriteVRMLPointData_Solid(argFP);
		WriteVRMLIFData_Solid(argFP, triFlag);
	} else {
		WriteVRMLPointData_IFS(argFP);
		WriteVRMLIFData_IFS(argFP, triFlag);
	}

	fprintf(argFP, "\t\t\t\tcolorPerVertex TRUE\n");
	fprintf(argFP, "\t\t\t\tconvex TRUE\n");
	fprintf(argFP, "\t\t\t}\n");
	fprintf(argFP, "\t\t}\n");

	return;
}

void fk_VRMLOut::WriteVRMLLightInfo(FILE *argFP)
{
	string	tab, outStr;

	tab = "\t\t";
	outStr = tab + "DirectionalLight {\n";
	
	tab = "\t\t\t";
	outStr += tab + "ambientIntensity 1.0\n";
	outStr += tab + "direction -1.0 -1.0 1.0\n";

	tab = "\t\t";
	outStr += tab + "}\n\n";
	fprintf(argFP, "%s", outStr.c_str());

	return;
}

void fk_VRMLOut::WriteVRMLNaviInfo(FILE *argFP)
{
	string	outStr, tab;

	tab = "\t\t";
	outStr = tab + "NavigationInfo {\n";

	tab = "\t\t\t";
	outStr += tab + "speed 5.0\n";
	outStr += tab + "type \"EXAMINE\"\n";

	tab = "\t\t";
	outStr += tab + "}\n\n";

	fprintf(argFP, "%s", outStr.c_str());
}

void fk_VRMLOut::WriteVRMLMaterial(FILE *argFP, fk_Material *argMaterial)
{
	fk_Color	*curColor;

	if(argMaterial == (fk_Material *)NULL) return;

	fprintf(argFP, "\t\t\tappearance Appearance {\n");
	fprintf(argFP, "\t\t\t\tmaterial Material {\n");

	// Ambient
	curColor = argMaterial->getAmbient();
	fprintf(argFP, "\t\t\t\t\tambientIntensity\t%f\n",
			(curColor->getR() + curColor->getG() + curColor->getB())/3.0);

	// Diffuse
	curColor = argMaterial->getDiffuse();
	fprintf(argFP, "\t\t\t\t\tdiffuseColor\t%f %f %f\n",
			curColor->getR(), curColor->getG(), curColor->getB());

	// Specular
	curColor = argMaterial->getSpecular();
	fprintf(argFP, "\t\t\t\t\tspecularColor\t%f %f %f\n",
			curColor->getR(), curColor->getG(), curColor->getB());

	// Emission
	curColor = argMaterial->getEmission();
	fprintf(argFP, "\t\t\t\t\temissiveColor\t%f %f %f\n",
			curColor->getR(), curColor->getG(), curColor->getB());

	// Shininess
	fprintf(argFP, "\t\t\t\t\tshininess\t%f\n",
			argMaterial->getShininess()/128.0);

	fprintf(argFP, "\t\t\t\t}\n\t\t\t}\n");

	return;
}


void fk_VRMLOut::WriteVRMLPointData_Solid(FILE *argFP)
{
	fk_Vertex		*curV;
	fk_Vector		pos;

	curV = solid->getNextV((fk_Vertex *)NULL);

	fprintf(argFP, "\t\t\tgeometry IndexedFaceSet {\n");
	fprintf(argFP, "\t\t\t\tcoord DEF C Coordinate {\n");
	fprintf(argFP, "\t\t\t\t\tpoint [\n");

	while(curV != (fk_Vertex *)NULL) {

		pos = curV->getPosition();
		fprintf(argFP, "\t\t\t\t\t\t");
		fprintf(argFP, "%f %f %f", pos.x, pos.y, pos.z);

		curV = solid->getNextV(curV);
		if(curV != (fk_Vertex *)NULL) {
			fprintf(argFP, ",");
		}
		fprintf(argFP, "\n");
	}

	fprintf(argFP, "\t\t\t\t\t]\n");
	fprintf(argFP, "\t\t\t\t}\n");

	return;
}

void fk_VRMLOut::WriteVRMLPointData_IFS(FILE *argFP)
{
	fk_Vector		pos;
	int				i, pSize;

	fprintf(argFP, "\t\t\tgeometry IndexedFaceSet {\n");
	fprintf(argFP, "\t\t\t\tcoord DEF C Coordinate {\n");
	fprintf(argFP, "\t\t\t\t\tpoint [\n");

	pSize = ifs->getPosSize();
	for(i = 0; i < pSize; i++) {
		
		pos = ifs->getPosVec(i);
		fprintf(argFP, "\t\t\t\t\t\t");
		fprintf(argFP, "%f %f %f", pos.x, pos.y, pos.z);

		if(i != pSize - 1) fprintf(argFP, ",");

		fprintf(argFP, "\n");
	}

	fprintf(argFP, "\t\t\t\t\t]\n");
	fprintf(argFP, "\t\t\t\t}\n");

	return;
}

void fk_VRMLOut::WriteVRMLIFData_Solid(FILE *argFP, bool triFlag)
{
	fk_Loop					*curL;
	vector<fk_Vertex *>		vArray;
	fk_Vertex				*tmpV;
	unsigned int			i;
	unsigned int			maxID, mapID;
	vector<int>				vMap;

	vMap.clear();
	tmpV = solid->getLastV();
	if(tmpV == (fk_Vertex *)NULL) return;
	maxID = tmpV->getID();
	vMap.resize(maxID+1);
	mapID = 0;
	for(i = 1; i <= maxID; i++) {
		if(solid->existVertex(i) == true) {
			vMap[i] = mapID;
			mapID++;
		}
	}

	curL = solid->getNextL((fk_Loop *)NULL);

	fprintf(argFP, "\t\t\t\tcoordIndex [\n");

	while(curL != (fk_Loop *)NULL) {
		vArray = solid->getAllVOnL(curL);

		if(triFlag == true && vArray.size() == 4) {
			// 四角形を二つの三角形に分割

			fprintf(argFP, "\t\t\t\t\t\t");
			fprintf(argFP, "%d, ", vMap[vArray[0]->getID()]);
			fprintf(argFP, "%d, ", vMap[vArray[1]->getID()]);
			fprintf(argFP, "%d, ", vMap[vArray[3]->getID()]);
			fprintf(argFP, "-1,\n");
			fprintf(argFP, "\t\t\t\t\t\t");
			fprintf(argFP, "%d, ", vMap[vArray[3]->getID()]);
			fprintf(argFP, "%d, ", vMap[vArray[1]->getID()]);
			fprintf(argFP, "%d, ", vMap[vArray[2]->getID()]);
			fprintf(argFP, "-1");

		} else {
			// Index データをそのまま出力

			fprintf(argFP, "\t\t\t\t\t\t");
			for(i = 0; i < vArray.size(); i++) {
				fprintf(argFP, "%d, ", vMap[vArray[i]->getID()]);
			}
			fprintf(argFP, "-1");
		}

		curL = solid->getNextL(curL);
		if(curL != (fk_Loop *)NULL) {
			fprintf(argFP, ",");
		}
		fprintf(argFP, "\n");
	}
 
	fprintf(argFP, "\t\t\t\t]\n");

	return;
}

void fk_VRMLOut::WriteVRMLIFData_IFS(FILE *argFP, bool)
{
	vector<int>		fData;
	int				fSize, i, j;


	fSize = ifs->getFaceSize();

	fprintf(argFP, "\t\t\t\tcoordIndex [\n");
	
	for(i = 0; i < fSize; i++) {
		fprintf(argFP, "\t\t\t\t\t\t");
		fData = ifs->getFaceData(i);
		for(j = 0; j < int(fData.size()); j++) {
			fprintf(argFP, "%d, ", fData[j]);
		}
		fprintf(argFP, "-1");
		if(i != fSize - 1) {
			fprintf(argFP, ",");
		}
		fprintf(argFP, "\n");
	}

	fprintf(argFP, "\t\t\t\t]\n");
	return;
}

void fk_VRMLOut::WriteVRMLFooter(FILE *argFP, fk_VRMLOutMode argMode)
{
	string 	outStr, tab;

	outStr.erase();

	if(argMode == FK_VRML_INTR) {
		tab = "\t\t";
		outStr += tab + "DEF TS TimeSensor {\n";
		tab = "\t\t\t";
		outStr += tab + "cycleInterval 5.0\n";
		outStr += tab + "loop TRUE\n";
		tab = "\t\t";
		outStr += tab + "}\n";
	}

	outStr += "\t]\n";
	outStr += "}\n";

	if(argMode == FK_VRML_INTR) {
		outStr += "ROUTE CI.value_changed TO C.point\n";
		outStr += "ROUTE TS.fraction_changed TO CI.set_fraction\n";
	}

	fwrite(outStr.c_str(), sizeof(char), outStr.size(), argFP);
	return;
}

void fk_VRMLOut::WriteVRMLCoordInterp(FILE *argFP,
									  vector<double> *argTime,
									  vector<fk_Vector> *argPos)
{
	int			i;
	int			vNum;
	string		baseTab = "\t\t";
	string		tab;
	string		buf;
	string		tmpStr;
	fk_Vector	tmpPos;

	vNum = int(argPos->size() / (3*argTime->size()));

	tab = baseTab;
	buf = tab + "DEF CI CoordinateInterpolator {\n";

	tab = baseTab + "\t";
	buf += tab + "key [\n";
	fwrite(buf.c_str(), sizeof(char), buf.size(), argFP);

	tab = baseTab + "\t\t";
	for(i = 0; i < int(argTime->size()); i++) {
		tmpStr = fk_StrPrintf("%f", (*argTime)[i]);
		buf = tab + tmpStr;
		if(i != int(argTime->size()) - 1) {
			buf += ",";
		}
		buf += "\n";
		fwrite(buf.c_str(), sizeof(char), buf.size(), argFP);
	}

	tab = baseTab + "\t";
	buf = tab + "]\n";
	buf += tab + "keyValue [\n";
	fwrite(buf.c_str(), sizeof(char), buf.size(), argFP);

	tab = baseTab + "\t\t";
	for(i = 0; i < int(argPos->size()); i++) {
		tmpPos = (*argPos)[i];
		tmpStr = fk_StrPrintf("%f %f %f", tmpPos.x, tmpPos.y, tmpPos.z);
		buf = tab + tmpStr;
		if(i != int(argPos->size()) - 1) {
			buf += ",";
		}
		buf += "\n";
		fwrite(buf.c_str(), sizeof(char), buf.size(), argFP);
	}

	tab = baseTab + "\t";
	buf = tab + "]\n";
	tab = baseTab;
	buf += tab + "}\n";
	fwrite(buf.c_str(), sizeof(char), buf.size(), argFP);

	return;
}
