/**
 * Copyright (c) 2004, yher2 project
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, 
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * * Neither the name of the nor the names of its contributors may be used to endorse or 
 *   promote products derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES 
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; 
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY 
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 */
package net.yher2.junit.db.data;

import junit.framework.TestCase;

/**
 * @auther  hisaboh
 */
public class XmlParserTest extends TestCase {

	public void testNomalize() {
		assertEquals(
				"<date value=\"2004%2F12%2F21+13%3A52%3A11\" format=\"yyyy%2FMM%2Fdd+HH%3Amm%3Ass\"/>",
				new XmlParser("<date value=\"2004/12/21 13:52:11\" format=\"yyyy/MM/dd HH:mm:ss\"/>").getXml());
		assertEquals(
				"<date/>",
				new XmlParser("<date/>").getXml());
		assertEquals(
				"<char value=\"1%3C%3E%26quot%3B\"/>",
				new XmlParser("<char value \t\r\n= \t\r\n\"1<>&quot;\"/>").getXml());
		assertEquals(
				"<random length=\"3\" type=\"alphabet\"/>",
				new XmlParser("<random length=\"3\" \n type=\"alphabet\"/>").getXml());
		assertEquals(
				"<random length=\"2\" type=\"alphabet\"/>",
				new XmlParser("<random length=\"2\" \r\n type=\"alphabet\"/>").getXml());
		assertEquals(
				"<random length=\"2\" type=\"number\"/>",
				new XmlParser("<random length=\"2\" \t type=\"number\"/>").getXml());
		assertEquals(
				"<date/>",
				new XmlParser("\n\r \t<date/>\n\r \t").getXml());
	}
	
	public void testIsXml() {
		assertTrue(new XmlParser("<date value=\"2004/12/21 13:52:11\" format=\"yyyy/MM/dd HH:mm:ss\"/>").isXml());
		assertTrue(new XmlParser("<date/>").isXml());
		assertTrue(new XmlParser("<string value=\"1\"/>").isXml());
		assertTrue(new XmlParser("<random length=\"3\" \n type=\"alphabet\"/>").isXml());
		assertTrue(new XmlParser("<random length=\"2\" \r\n type=\"alphabet\"/>").isXml());
		assertTrue(new XmlParser("<random length=\"2\" \t type=\"number\"/>").isXml());
		assertTrue(new XmlParser("\n\r \t<date/>\n\r \t").isXml());
		assertFalse(new XmlParser("d<ate/>").isXml());
		assertFalse(new XmlParser("<date/").isXml());
		assertFalse(new XmlParser("<date>").isXml());
	}

	public void testParse() {
		XmlParser parser =new XmlParser("<date value=\"2004/12/21 13:52:11\" format=\"yyyy/MM/dd HH:mm:ss\"/>");
		parser.parse();
		assertEquals("date", parser.getElementName());
		assertEquals("2004/12/21 13:52:11", parser.getAttribute("value"));
		assertEquals("yyyy/MM/dd HH:mm:ss", parser.getAttribute("format"));
		
		parser = new XmlParser("<dAte/>");
		parser.parse();
		assertEquals("date", parser.getElementName());
		assertFalse(parser.hasAttribute("value"));
		assertFalse(parser.hasAttribute("format"));

		parser = new XmlParser("<char value \t\r\n= \t\r\n\"1<>&quot;&amp; \n\r\t\"/>");
		parser.parse();
		assertEquals("char", parser.getElementName());
		assertEquals("1<>\"& \n\r\t", parser.getAttribute("value"));
	}
	public void testParseNoElementName() {
		try {
			new XmlParser("<hoge=\"\">").parse();
		} catch (Exception e) {
			return;
		}
		fail();
	}
	public void testParseNoAttributeName() {
		try {
			new XmlParser("<hoge =\"\">").parse();
		} catch (Exception e) {
			return;
		}
		fail();
	}
	public void testParseNoAttributeValue() {
		try {
			new XmlParser("<hoge hoge=>").parse();
		} catch (Exception e) {
			return;
		}
		fail();
	}
	
	public void testParseInvalidXml() {
		try {
			new XmlParser("<hoge>").parse();
		} catch (Exception e) {
			return;
		}
		fail();
	}
	public void testParseInvalidXml2() {
		try {
			new XmlParser("d<hoge/>").parse();
		} catch (Exception e) {
			return;
		}
		fail();
	}
	public void testParseInvalidXml3() {
		try {
			new XmlParser("<hoge").parse();
		} catch (Exception e) {
			return;
		}
		fail();
	}
}
