﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Web;

namespace Alternative.Controls.Browsers
{
	/// <summary><see cref="TabBrowserControl.TbMovieDownloadRequired"/>のイベントデータを提供します。
	/// <para>このクラスは、イベントの発行元であるリクエストURLに格納されているパラメータへのアクセスを補助します。</para></summary>
	public class MovieDownloadRequiredEventArgs : EventArgs
	{
		Dictionary<String, String> _parameters;

		#region Public Property / Method

		/// <summary>動画ダウンロードに関連するURLを取得します。この値は「url」パラメータに指定された値です。</summary>
		/// <remarks>パラメータ「url」は必須パラメータですが、値はパラメータを作成した<see cref="Alternative.Functions.IHtmlDocumentEditor"/>に依存します。
		/// <para>例えば<see cref="NicovideoDownloadHtmlEditor"/>は、ニコニコ動画の視聴ページのURLを格納します。</para></remarks>
		public string Url
		{
			get { return this.GetParameter(RequestParams.Url.Name); }
		}

		/// <summary>保存名としてタイトルが明示的に指示された場合に、ここから取得できます。
		/// 指定が無かった場合はnullが返ります。</summary>
		public string NewName
		{
			get { return this.GetParameter(RequestParams.NewName.Name); }
		}

		/// <summary>「title」パラメータに指定された文字列を取得します。特に指定されていない場合はnullが返ります。</summary>
		/// <remarks>値はパラメータを作成した<see cref="Alternative.Functions.IHtmlDocumentEditor"/>に依存します。
		/// <para>例えば<see cref="NicovideoDownloadHtmlEditor"/>は、ニコニコ動画の視聴ページのURLを格納します。</para></remarks>
		public string DisplayTitle
		{
			get { return this.GetParameter(RequestParams.DisplayTitle.Name); }
		}

		/// <summary>ダウンロードを保留登録するように指示されたかどうかを取得します</summary>
		public bool IsReserveMode
		{
			get { return RequestParams.Reserved.GetBoolean(this.GetParameter(RequestParams.Reserved.Name)); }
		}

		/// <summary>詳細なリクエストの作成を要求されたかどうかを取得します。</summary>
		public bool IsDetailMode
		{
			get { return RequestParams.Custom.GetBoolean(this.GetParameter(RequestParams.Custom.Name)); }
		}

		/// <summary>指定したキーのリクエストパラメータを取得します。キーが無効であった場合はnullが返ります。</summary>
		/// <param name="key">取得するパラメータ名</param>
		/// <returns></returns>
		public String GetParameter(String key)
		{
			String value;
			if(this._parameters.TryGetValue(key, out value)) { return HttpUtility.UrlDecode(value); }
			return null;
		}

		/// <summary>リクエストに格納されているパラメータ名を列挙します。</summary>
		/// <returns></returns>
		public IEnumerable<String> GetParameterKeys()
		{
			return (IEnumerable<String>)this._parameters.Keys;
		}

		/// <summary>リクエストに格納されているパラメータを列挙します。</summary>
		/// <returns></returns>
		public IEnumerable<KeyValuePair<String,String>> GetParameters()
		{
			foreach(KeyValuePair<String, String> parameter in this._parameters)
			{
				yield return parameter;
			}
		}

		#endregion

		/// <summary>このクラスのコンストラクタです。</summary>
		/// <param name="requestUrl">ダウンロードリクエストURL</param>
		private MovieDownloadRequiredEventArgs(String requestUrl)
		{
			if(requestUrl == null) { throw new ArgumentNullException(); }
			this._parameters = WebServiceClient.HttpHelper.ParseUrlValues(requestUrl);
			if(!this._parameters.ContainsKey(RequestParams.Url.Name))
			{
				throw new ArgumentException("リクエストURLに必須パラメータのurlが含まれていません。");
			}
		}


		#region Static / Factory Methods

		/// <summary>動画視聴ページのURLからこのクラスのインスタンスを作成します。</summary>
		/// <param name="pageUrl">動画視聴ページのURL文字列</param>
		/// <returns></returns>
		public static MovieDownloadRequiredEventArgs FromVideoPageUrl(String pageUrl)
		{
			return FromVideoPageUrl(pageUrl, null);
		}
		/// <summary>動画視聴ページのURLからこのクラスのインスタンスを作成します。</summary>
		/// <param name="pageUrl">動画視聴ページのURL文字列</param>
		/// <returns></returns>
		public static MovieDownloadRequiredEventArgs FromVideoPageUrl(String pageUrl, Options addParamOptions)
		{
			return FromVideoPageUrl(pageUrl, GetAddParams(addParamOptions));
		}
		/// <summary>動画視聴ページのURLからこのクラスのインスタンスを作成します。</summary>
		/// <param name="pageUrl">動画視聴ページのURL文字列</param>
		/// <returns></returns>
		public static MovieDownloadRequiredEventArgs FromVideoPageUrl(String pageUrl, IEnumerable<KeyValuePair<string, string>> additionalParams)
		{
			String downLink = CommonRequestProvider.CreateRequest(pageUrl, additionalParams);
			MovieDownloadRequiredEventArgs args = new MovieDownloadRequiredEventArgs(downLink);
			return args;
		}

		/// <summary>ダウンロードリクエストURLを使用してこのクラスのインスタンスを作成します。</summary>
		/// <param name="requestUrl">ダウンロードリクエストURL文字列</param>
		/// <returns></returns>
		public static MovieDownloadRequiredEventArgs FromRequestUrl(String requestUrl)
		{
			return FromRequestUrl(requestUrl, null);
		}
		/// <summary>ダウンロードリクエストURLを使用してこのクラスのインスタンスを作成します。</summary>
		/// <param name="requestUrl">ダウンロードリクエストURL文字列</param>
		/// <param name="addParamOption">既定の追加パラメータを表すオプション情報</param>
		/// <returns></returns>
		public static MovieDownloadRequiredEventArgs FromRequestUrl(String requestUrl, Options addParamOption)
		{
			return FromRequestUrl(requestUrl, GetAddParams(addParamOption));
		}
		/// <summary>ダウンロードリクエストURLを使用してこのクラスのインスタンスを作成します。</summary>
		/// <param name="requestUrl">ダウンロードリクエストURL文字列</param>
		/// <param name="additionalParams">追加するパラメータを列挙する列挙子</param>
		/// <returns></returns>
		public static MovieDownloadRequiredEventArgs FromRequestUrl(String requestUrl, IEnumerable<KeyValuePair<String,String>> additionalParams)
		{
			MovieDownloadRequiredEventArgs args =
				new MovieDownloadRequiredEventArgs(requestUrl);

			if(additionalParams != null)
			{
				foreach(KeyValuePair<string, string> pair in additionalParams)
				{
					if(args._parameters.ContainsKey(pair.Key))
					{
						args._parameters[pair.Key] = pair.Value;
					}
					else
					{
						args._parameters.Add(pair.Key, pair.Value);
					}
				}
			}
			return args;
		}

		#endregion

		#region Static / Parameter Creator

		/// <summary>システムで使用するパラメータを取得します。このパラメータはリクエストを保留登録することを表します。</summary>
		public static KeyValuePair<String, String> ParamPair_ReserveMode
		{
			get { return new KeyValuePair<string, string>(RequestParams.Reserved.Name, RequestParams.Reserved.V_Reserved); }
		}
		/// <summary>システムで使用するパラメータを取得します。このパラメータはリクエストの詳細設定を行うことを表します。</summary>
		public static KeyValuePair<String, String> ParamPair_CustomizeRequired
		{
			get { return new KeyValuePair<string, string>(RequestParams.Custom.Name, RequestParams.Custom.V_Custom); }
		}
		/// <summary>システムで使用するパラメータを取得します。このパラメータは保存用にタイトルを指定することを表します。</summary>
		public static KeyValuePair<String, String> ParamPair_NewName(String name)
		{
			return new KeyValuePair<string, string>(RequestParams.NewName.Name, name);
		}

		/// <summary><see cref="FromRequestUrl"/>等で使用する追加パラメータリストを作成します。</summary>
		/// <param name="option">事前に定義されたパラメータを表すビットフィールド</param>
		/// <returns></returns>
		public static IEnumerable<KeyValuePair<String, String>> GetAddParams(Options option)
		{
			if(option == Options.None) { yield break; }
			if((option & Options.Reserved) == Options.Reserved) { yield return ParamPair_ReserveMode; }
			if((option & Options.CustomMode) == Options.CustomMode) { yield return ParamPair_CustomizeRequired; }
		}
		/// <summary><see cref="FromRequestUrl"/>等で使用する追加パラメータリストを作成します。</summary>
		/// <param name="option">事前に定義されたパラメータを表すビットフィールド</param>
		/// <param name="newName">ファイル名を変更して保存する際の新しい名前の指定。</param>
		/// <returns></returns>
		public static IEnumerable<KeyValuePair<String, String>> GetAddParams(Options option, string newName)
		{
			if(option == Options.None) { yield break; }
			if((option & Options.Reserved) == Options.Reserved) { yield return ParamPair_ReserveMode; }
			if((option & Options.CustomMode) == Options.CustomMode) { yield return ParamPair_CustomizeRequired; }
			if(!String.IsNullOrEmpty(newName)) { yield return ParamPair_NewName(newName); }
		}

		#endregion

		/// <summary>システムが使用する既定の追加パラメータを表します。</summary>
		public enum Options
		{
			/// <summary>リクエストは通常通りです。</summary>
			None,
			/// <summary>リクエストは保留登録するよう指示されました。</summary>
			Reserved,
			/// <summary>リクエストに対して詳細な設定を行うように指示されました。</summary>
			CustomMode,
		}
	}


	/// <summary><see cref="TabBrowserControl.TbCallCommand"/>イベントで使用するデータを格納します。</summary>
	public class TbCallCommandEventArgs : EventArgs
	{
		string _command;
		object _param;

		/// <summary>コマンドを表す文字列を取得します。</summary>
		public string Command
		{
			get { return this._command; }
		}
		/// <summary>パラメータを表す<see cref="Object"/>を取得します。</summary>
		public object Param
		{
			get { return this._param; }
		}

		/// <summary>このクラスのコンストラクタです。</summary>
		/// <param name="command">コマンド文字列</param>
		/// <param name="param">パラメータを表すObject</param>
		public TbCallCommandEventArgs(string command, object param)
		{
			this._command = command;
			this._param = param;
		}
	}


	/// <summary><see cref="TabBrowserControl"/>がホスティングしているオブジェクトからメッセージを受信する際のイベントの情報を格納します。</summary>
	public class TbMessagePostEventArgs : EventArgs
	{
		/// <summary>ブラウザからのロック要求の際に使用する<see cref="TbMessagePostEventArgs"/>を取得します。</summary>
		/// <returns>作成された<see cref="TbMessagePostEventArgs"/></returns>
		public static TbMessagePostEventArgs GetForRequireLock() { return new TbMessagePostEventArgs(MessageRequireLock); }
		/// <summary>ブラウザからのアンロック要求の際に使用する<see cref="TbMessagePostEventArgs"/>を取得します。</summary>
		/// <returns>作成された<see cref="TbMessagePostEventArgs"/></returns>
		public static TbMessagePostEventArgs GetForRequireUnlock() { return new TbMessagePostEventArgs(MessageRequireUnlock); }

		/// <summary>ロック要求を表すメッセージ文字列</summary>
		internal const string MessageRequireLock = "lock.require";
		/// <summary>アンロック要求を表すメッセージ文字列</summary>
		internal const string MessageRequireUnlock = "unlock.require";

		string _message;
		object _tag;

		/// <summary>伝達されたメッセージを取得します。</summary>
		public string Message { get { return this._message; } }
		/// <summary>オプション情報を取得します。</summary>
		public object Tag { get { return this._tag; } }


		private TbMessagePostEventArgs(string message) : this(message, null) { }
		private TbMessagePostEventArgs(string message, object tag)
		{
			this._message = message;
			this._tag = tag;
		}
	}

	/// <summary><see cref="TabBrowserControl"/>がホスティングしているオブジェクトからメッセージを受信する際のイベントを表します。</summary>
	public delegate void TbMessagePostEventHandler(object sender, TbMessagePostEventArgs e);
}
