﻿using System;
using System.Text.RegularExpressions;
using System.Runtime.Serialization;

namespace Alternative.Controls.Browsers
{

	/// <summary>マッチの為の正規表現の情報を格納します。</summary>
	[Serializable]
	public class RegexInfo : ICloneable, ISerializable
	{
		#region Static Members

		/// <summary>どんな文字列にもマッチしない正規表現パターンです。</summary>
		public const string NonAffectMatch = @"[^\s\S]";

		/// <summary>Regexオブジェクト作成を試行することで、受け取ったパターンが適切かどうかを確認します。</summary>
		/// <param name="patten">チェックするパターン</param>
		/// <param name="error">エラーが発生した場合はエラーメッセージが格納されます。</param>
		/// <returns>作成試行に成功した場合はtrueが返ります。</returns>
		public static bool CheckRegexPattern(string patten, out string error)
		{
			error = null;
			try { Regex rx = new Regex(patten); }
			catch(Exception e) { error = e.Message; }
			return (error == null);
		}

		/// <summary>Regexオブジェクト作成を試行します。</summary>
		/// <param name="patten">正規表現パターン</param>
		/// <param name="product">作成に成功した場合は、Regexオブジェクトが格納されます。</param>
		/// <returns>作成試行に成功した場合はtrueが返ります。</returns>
		public static bool TryCreateRegex(string pattern, out Regex product)
		{
			return TryCreateRegex(pattern, RegexOptions.None, out product);
		}
		/// <summary>Regexオブジェクト作成を試行します。</summary>
		/// <param name="patten">正規表現パターン</param>
		/// <param name="options">正規表現オプション。</param>
		/// <param name="product">作成に成功した場合は、Regexオブジェクトが格納されます。</param>
		/// <returns>作成試行に成功した場合はtrueが返ります。</returns>
		public static bool TryCreateRegex(string pattern, RegexOptions options, out Regex product)
		{
			try { product = new Regex(pattern, options); }
			catch { product = null; }
			return (product != null);
		}
		/// <summary>RegexInfoオブジェクトの情報に問題が含まれているかどうかをチェックします。</summary>
		/// <param name="info">確認するRegexInfoオブジェクト。</param>
		/// <param name="errorMsg">エラーが発生した場合は、エラーメッセージが格納されます。</param>
		/// <returns>問題が無ければtrueが返ります。</returns>
		public static bool CheckRegexInfo(RegexInfo info, out string errorMsg)
		{
			if(info is RegexReplaceInfo)
			{
				return RegexReplaceInfo.CheckRegexInfo((RegexReplaceInfo)info, out errorMsg);
			}
			else
			{
				errorMsg = null;
				try
				{
					Regex rx = null;
					RegexOptions opt = info.Options;
					if(!String.IsNullOrEmpty(info.Pattern))
					{
						rx = new Regex(info.Pattern);
					}
					else { }
				}
				catch(Exception e) { errorMsg = e.Message; }
				return (errorMsg == null);
			}
		}

		#endregion

		// vvvvvvvvvvvvvv  Instance Members  vvvvvvvvvvvvvvv //

		#region Fields

		/// <summary><see cref="Name"/>プロパティのフィールドです。</summary>
		protected string _name;
		/// <summary><see cref="Pattern"/>プロパティのフィールドです。</summary>
		protected string _pattern;
		/// <summary><see cref="Options"/>プロパティのフィールドです。</summary>
		protected RegexOptions _options;
		/// <summary><see cref="Enabled"/>プロパティのフィールドです。</summary>
		protected bool _enabled;

		#endregion


		#region Constructors

		/// <summary>インスタンスを作成します。</summary>
		/// <param name="name">この情報につけられた名前。</param>
		/// <param name="pattern">マッチさせる正規表現パターン。</param>
		/// <param name="options">正規表現で使用するオプション情報</param>
		/// <param name="enabled">この情報を使用するかどうかを表すフラグ。</param>
		public RegexInfo(string name, string pattern, RegexOptions options, bool enabled)
		{
			this._name = String.IsNullOrEmpty(name) ? "<Default Name>" : name;
			this._options = options;
			this._pattern = pattern ?? NonAffectMatch;
			this._enabled = enabled;
		}

		/// <summary>インスタンスを作成します。</summary>
		/// <param name="name">この情報につけられた名前。</param>
		/// <param name="pattern">マッチさせる正規表現パターン。</param>
		/// <param name="enabled">この情報を使用するかどうかを表すフラグ。</param>
		public RegexInfo(string name, string pattern, bool enabled) : this(name, pattern, RegexOptions.None, enabled) { }

		/// <summary>インスタンスを作成します。コピーコンストラクタです。</summary>
		/// <param name="info">コピー元のインスタンス。</param>
		public RegexInfo(RegexInfo info) : this(info._name, info._pattern, info._options, info._enabled) { }

		/// <summary>XmlSerializerのためのデフォルトコンストラクタ。</summary>
		public RegexInfo() : this("<Default Name>", NonAffectMatch, RegexOptions.None, false) { }

		/// <summary>インスタンスを作成します。</summary>
		/// <param name="si">インスタンスの情報を格納したSerializationInfo</param>
		/// <param name="context">ストリームの情報を表すStreamingContext</param>
		public RegexInfo(SerializationInfo si, StreamingContext context)
		{
			this._name = si.GetString(SN_Name);
			this._pattern = si.GetString(SN_Pattern);
			this._options = (RegexOptions)si.GetValue(SN_Options, typeof(RegexOptions));
			this._enabled = si.GetBoolean(SN_Enabled);
		}

		#endregion


		#region Property

		/// <summary>このオブジェクトに付けられた名前を取得・設定します。</summary>
		public string Name
		{
			get { return this._name; }
			set { this._name = String.IsNullOrEmpty(value) ? "<No Title>" : value; }
		}
		/// <summary>正規表現のマッチパターンを取得・設定します。</summary>
		public string Pattern
		{
			get { return this._pattern; }
			set { this._pattern = value ?? String.Empty; }
		}
		/// <summary>正規表現のマッチオプションを取得・設定します。</summary>
		public RegexOptions Options { get { return this._options; } set { this._options = value; } }
		/// <summary>この情報を使用するかどうかを表すフラグを取得・設定します。</summary>
		public bool Enabled { get { return this._enabled; } set { this._enabled = value; } }

		#endregion


		#region Methods

		/// <summary>（オーバーライドされます）このオブジェクトの名前を取得します。<see cref="Name"/>に等価です。</summary>
		/// <returns>Nameプロパティと同じ文字列。</returns>
		public override string ToString()
		{
			return this._name;
		}

		/// <summary>格納した情報から、<see cref="System.Text.RegularExpressions.Regex"/>オブジェクトを作成します。</summary>
		/// <param name="product">作成されたRegexオブジェクト。又はnull</param>
		/// <returns>作成に成功した場合はtrue</returns>
		public bool TryCreateRegex(out Regex product)
		{
			return RegexInfo.TryCreateRegex(this.Pattern, this.Options, out product);
		}

		/// <summary>この情報からRegexオブジェクトを取得します。</summary>
		public Regex Regex
		{
			get { return new Regex(this._pattern, this._options); }
		}

		#region ICloneable メンバ

		/// <summary>オブジェクトの簡易コピーを作成します。</summary>
		/// <returns>新しく作成されたインスタンス。</returns>
		public virtual object Clone()
		{
			RegexInfo rx = new RegexInfo(this);
			return rx;
		}

		#endregion

		#endregion


		#region ISerializable メンバ

		/// <summary><see cref="_name"/>フィールドに対するシリアライズキー</summary>
		protected const string SN_Name = "name";
		/// <summary><see cref="_pattern"/>フィールドに対するシリアライズキー</summary>
		protected const string SN_Pattern = "pattern";
		/// <summary><see cref="_options"/>フィールドに対するシリアライズキー</summary>
		protected const string SN_Options = "options";
		/// <summary><see cref="_enabled"/>フィールドに対するシリアライズキー</summary>
		protected const string SN_Enabled = "enabled";

		/// <summary>インスタンスをシリアライズします。</summary>
		/// <param name="si">インスタンスの情報を格納するSerializationInfo</param>
		/// <param name="context">ストリームの情報を表すStreamingContext</param>
		public virtual void GetObjectData(SerializationInfo info, StreamingContext context)
		{
			info.AddValue(SN_Name, this._name);
			info.AddValue(SN_Pattern, this._pattern);
			info.AddValue(SN_Options, this._options, typeof(RegexOptions));
			info.AddValue(SN_Enabled, this._enabled);
		}

		#endregion
	}

	/// <summary>置換の為の正規表現の情報を格納します。</summary>
	/// <seealso cref="RegexInfo"/>
	[Serializable]
	public class RegexReplaceInfo : RegexInfo
	{
		#region Static Member

		/// <summary>入力をそのまま出力する置換パターンです。</summary>
		public const string NonAffectReplace = @"$_";

		/// <summary>RegexReplaceInfoオブジェクトの情報に問題が含まれているかどうかをチェックします。</summary>
		/// <param name="info">確認するRegexReplaceInfoオブジェクト。</param>
		/// <param name="errorMsg">エラーが発生した場合は、エラーメッセージが格納されます。</param>
		/// <returns>問題が無ければtrueが返ります。</returns>
		public static bool CheckRegexInfo(RegexReplaceInfo info, out string errorMsg)
		{
			errorMsg = null;
			try
			{
				Regex rx = null;
				RegexOptions opt = info.Options;
				if(!String.IsNullOrEmpty(info.Pattern))
				{
					rx = new Regex(info.Pattern);
					if(!String.IsNullOrEmpty(info.Replace))
					{
						rx.Replace("dummy", info.Replace, 1);
					}
					else { errorMsg = "置換出力の内容を空には出来ません。"; }
				}
				else { }
			}
			catch(Exception e) { errorMsg = e.Message; }

			return (errorMsg == null);
		}

		#endregion

		// vvvvvvvvvvv  Instance Members  vvvvvvvvvvvv //

		/// <summary><see cref="Replace"/>プロパティのフィールドです。</summary>
		protected string _replace;


		#region Constructor

		/// <summary>インスタンスを作成します。</summary>
		/// <param name="name">この情報につけられた名前。</param>
		/// <param name="pattern">マッチさせる正規表現パターン。</param>
		/// <param name="replace">正規表現置換出力パターン。</param>
		/// <param name="options">正規表現で使用するオプション情報。</param>
		/// <param name="enabled">この情報を使用するかどうかを表すフラグ。</param>
		public RegexReplaceInfo(string name, string pattern, string replace, RegexOptions options, bool enabled)
			: base(name, pattern, options, enabled)
		{
			this._replace = replace ?? NonAffectReplace;
		}

		/// <summary>インスタンスを作成します。コピーコンストラクタです。</summary>
		/// <param name="info">コピー元のインスタンス。</param>
		public RegexReplaceInfo(RegexReplaceInfo info)
			: base(info)
		{
			this._replace = info._replace;
		}

		/// <summary>XmlSerializerのためのデフォルトコンストラクタ。</summary>
		public RegexReplaceInfo()
			: base("<Default Name>", NonAffectMatch, RegexOptions.None, false)
		{
			this._replace = NonAffectReplace;
		}

		/// <summary>インスタンスを作成します。</summary>
		/// <param name="si">インスタンスの情報を格納したSerializationInfo</param>
		/// <param name="context">ストリームの情報を表すStreamingContext</param>
		public RegexReplaceInfo(SerializationInfo si, StreamingContext context) : base(si, context)
		{
			this._replace = si.GetString(SN_Replace);
		}


		#endregion


		/// <summary>正規表現の置換出力フォーマット文字列を取得・設定します。</summary>
		public string Replace
		{
			get { return this._replace; }
			set { this._replace = value ?? NonAffectReplace; }
		}


		#region IClonable メンバ

		/// <summary>オブジェクトの簡易コピーを作成します。</summary>
		/// <returns>新しく作成されたインスタンス。</returns>
		public override object Clone()
		{
			RegexReplaceInfo rx = new RegexReplaceInfo(this);
			return rx;
		}

		#endregion

		#region ISerializable メンバ

		/// <summary><see cref="_name"/>フィールドに対するシリアライズキー</summary>
		protected const string SN_Replace = "replace";

		/// <summary>インスタンスをシリアライズします。</summary>
		/// <param name="si">インスタンスの情報を格納するSerializationInfo</param>
		/// <param name="context">ストリームの情報を表すStreamingContext</param>
		public override void GetObjectData(SerializationInfo info, StreamingContext context)
		{
			base.GetObjectData(info, context);
			info.AddValue(this._replace, SN_Replace);
		}

		#endregion
	}
}
