﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

namespace Alternative.Controls.Browsers
{
	/// <summary>
	/// <see cref="TabBrowserControl"/>で使用される<see cref="System.Windows.Forms.Control"/>オブジェクトが公開する必要のあるインターフェースです。
	/// </summary>
	public interface ITbPageControl
	{
		/// <summary>表示中のURLを取得します。</summary>
		/// <remarks>このメソッドは、失敗時にも常にString.Emptyの戻り値を保障するようにして下さい。</remarks>
		string DisplayUrl { get; }
		/// <summary>
		/// 表示中のタイトルを取得します。
		/// <remarks>このメソッドは、失敗時にも常にString.Emptyの戻り値を保障するようにして下さい。</remarks>
		/// </summary>
		string DisplayTitle { get; }
		/// <summary>
		/// 表示中のStatusTextを取得します。
		/// <remarks>このメソッドは、失敗時にも常にString.Emptyの戻り値を保障するようにして下さい。</remarks>
		/// </summary>
		string DisplayStatusText { get; }
		/// <summary><para>このページ上で指定したURL移動を行います。</para>
		/// この操作をサポートしていないオブジェクトであった場合にはfalseが返り、何も起こりません。</summary>
		/// <param name="url">移動先のURL</param>
		/// <returns>この操作をサポートしていないオブジェクトであった場合にはfalseが返ります。</returns>
		bool Navigate(string url);
		/// <summary>このオブジェクトが<see cref="WebBrowserAx"/>であるかどうかを返します。</summary>
		bool IsWebBrowser { get; }
		/// <summary>このオブジェクトを<see cref="WebBrowserAx"/>として参照を取得します。
		/// <see cref="WebBrowserAx"/>で無い場合はnullが返ります。</summary>
		WebBrowserAx AsBrowser { get; }
		/// <summary><para>このオブジェクトを、<see cref="System.Windows.Forms.Control"/>として参照を取得します。</para>
		/// このプロパティは、必ず<see cref="System.Windows.Forms.Control"/>を返すように実装しなければなりません。</summary>
		Control AsControl { get; }
		/// <summary>このオブジェクトが、<see cref="TabBrowserControl"/>による参照から離れた時に呼び出されます。
		/// <para>このメソッドを外部から使用しないでください。</para></summary>
		void ReleaseControl();
		/// <summary>表示中の情報が変化した時に発生します。</summary>
		event ITbDisplayEventHandler DisplayTextChanged;
	}
	/// <summary>
	/// <see cref="ITbPageControl.DisplayTextChanged"/>イベントの情報を格納します。
	/// </summary>
	public class ITbDisplayEventArgs : EventArgs
	{
		private ITbDisplayEventType _type;
		private string _data;

		/// <summary>このイベントを発生させた要素を取得します。ビットフィールドです。</summary>
		public ITbDisplayEventType EventType
		{
			get { return this._type; }
		}
		/// <summary>このイベントで新しく設定された値を取得します。
		/// 例えば、<see cref="EventType"/>が<see cref="ITbDisplayEventType.Url"/>であれば、この値は新しいITbPageControl.DisplayUrlの値になります。</summary>
		public string Data
		{
			get { return this._data; }
		}

		/// <summary>必要な情報を受け取ってEventArgsを作成します。</summary>
		/// <param name="eventType">イベントを発生させた表示テキスト要素を表す<see cref="ITbDisplayEventType"/></param>
		/// <param name="data">新しい値</param>
		public ITbDisplayEventArgs(ITbDisplayEventType eventType, string data)
		{
			this._type = eventType;
			this._data = data;
		}
	}

	/// <summary>イベントを発生させた表示テキスト要素を表す列挙型。ビットフィールドです。</summary>
	[Flags]
	public enum ITbDisplayEventType
	{
		/// <summary>未定義値</summary>
		NotDefined = 0x0,
		/// <summary><see cref="ITbPageControl.DisplayUrl"/>を表す。</summary>
		Url = 0x1,
		/// <summary><see cref="ITbPageControl.DisplayTitle"/>を表す。</summary>
		Title = 0x2,
		/// <summary><see cref="ITbPageControl.DisplayStatusText"/>を表す。</summary>
		StatusText = 0x4,
	}

	/// <summary>ITbPageControlをsenderにもつハンドラ</summary>
	/// <param name="sender">イベントsender</param>
	/// <param name="e">イベントデータ</param>
	public delegate void ITbPageEventHandler(ITbPageControl sender, EventArgs e);
	/// <summary>ITbPageControlをsenderにもつジェネリックハンドラ</summary>
	/// <param name="sender">イベントsender</param>
	/// <param name="e">ジェネリックなイベントデータ</param>
	/// <typeparam name="TEventArgs"><see cref="System.EventArgs"/>を継承するクラス</typeparam>
	public delegate void ITbPageEventHandler<TEventArgs>(ITbPageControl sender, TEventArgs e) where TEventArgs : EventArgs;
	/// <summary><see cref="ITbPageControl.DisplayTextChanged"/>イベント用のハンドラ</summary>
	/// <param name="sender"></param>
	/// <param name="e"></param>
	public delegate void ITbDisplayEventHandler(ITbPageControl sender, ITbDisplayEventArgs e);

}
