/*
 * blanco Framework
 * Copyright (C) 2004-2006 IGA Tosiki
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.bcel;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;

import org.apache.bcel.classfile.ClassParser;
import org.apache.bcel.classfile.Code;
import org.apache.bcel.classfile.Field;
import org.apache.bcel.classfile.JavaClass;
import org.apache.bcel.classfile.Method;
import org.apache.bcel.classfile.Utility;
import org.apache.bcel.generic.Type;

import blanco.bcel.util.BlancoBcelXmlMarshaller;
import blanco.bcel.valueobject.BlancoBcelClassStructure;
import blanco.bcel.valueobject.BlancoBcelFieldStructure;
import blanco.bcel.valueobject.BlancoBcelLineStructure;
import blanco.bcel.valueobject.BlancoBcelMethodStructure;
import blanco.commons.util.BlancoNameUtil;

/**
 * Apache Jakarta BCELpāA JavaoCgR[h(classt@C)͂ BCEL͌ʂXMLt@Cɏo͂܂B
 * 
 * @author IGA Tosiki
 */
public class BlancoBcelJavaClass2Xml {
    /**
     * fobO[hœ삳邩ǂB<br>
     * fobO[h̏ꍇAIWisf[^XMLo͂܂B
     */
    private static final boolean IS_DEBUG = false;

    /**
     * JavaoCgR[h(classt@C)͂ BCEL͌ʂXMLt@Cɏo͂܂B
     * 
     * @param sourceFile
     *            \[Xt@CB
     * @param outFileBcel
     *            BCEL͌ʂo͂t@CB
     * @throws IOException
     *             o͗OꍇB
     */
    public final void process(final File sourceFile, final File outFileBcel)
            throws IOException {
        if (sourceFile == null) {
            throw new IllegalArgumentException(
                    "Classt@C͂XML鏈ɂāA̓NXt@Cnull^܂B");
        }
        if (sourceFile.exists() == false) {
            throw new IllegalArgumentException(
                    "Classt@C͂XML鏈ɂāA̓NXt@C["
                            + sourceFile.getAbsolutePath() + "]܂łB");
        }

        JavaClass javaClass = null;
        final InputStream inStream = new BufferedInputStream(
                new FileInputStream(sourceFile));
        try {
            javaClass = new ClassParser(inStream, sourceFile.getName()).parse();
        } finally {
            inStream.close();
        }

        final BlancoBcelClassStructure classStructure = new BlancoBcelClassStructure();

        processClass(javaClass, classStructure);

        processField(javaClass, classStructure);

        processMethod(javaClass, classStructure);

        new BlancoBcelXmlMarshaller().marshal(classStructure, outFileBcel);
    }

    /**
     * NXѐeNX^ꂽC^[ɏo͂܂B
     * 
     * @param javaClass
     *            BCELJavaClassIuWFNg
     * @param document
     *            o͐XMLhLg
     * @throws IOException
     *             o͗OꍇB
     */
    private void processClass(final JavaClass javaClass,
            final BlancoBcelClassStructure classStructure) throws IOException {
        classStructure.setName(javaClass.getClassName());
        classStructure.setSuperclassName(javaClass.getSuperclassName());

        final String[] interfaceNames = javaClass.getInterfaceNames();
        for (int index = 0; index < interfaceNames.length; index++) {
            classStructure.getInterfaceNameList().add(interfaceNames[index]);
        }
    }

    /**
     * NX̒ static final ȃtB[h񋓂܂B
     * 
     * @param javaClass
     *            BCELJavaClassIuWFNg
     * @param document
     *            o͐XMLhLg
     * @throws IOException
     *             o͗OꍇB
     */
    private void processField(final JavaClass javaClass,
            final BlancoBcelClassStructure classStructure) throws IOException {

        final Field[] fields = javaClass.getFields();
        for (int index = 0; index < fields.length; index++) {
            final BlancoBcelFieldStructure fieldStructure = new BlancoBcelFieldStructure();
            classStructure.getFieldList().add(fieldStructure);

            fieldStructure.setName(fields[index].getName());
            if (fields[index].isPublic()) {
                fieldStructure.setAccess("public");
            } else if (fields[index].isProtected()) {
                fieldStructure.setAccess("protected");
            } else if (fields[index].isPrivate()) {
                fieldStructure.setAccess("private");
            }
            fieldStructure.setStatic(fields[index].isStatic());
            fieldStructure.setFinal(fields[index].isFinal());

            if (fields[index].isFinal() && fields[index].isStatic()) {
                if (fields[index].getConstantValue() != null) {
                    fieldStructure.setConstantValue(fields[index]
                            .getConstantValue().toString());
                } else {
                    fieldStructure.setConstantValue("null");
                }
            }
        }
    }

    /**
     * \bhу\bh̓e^ꂽC^[ɏo͂܂B
     * 
     * @param javaClass
     *            BCELJavaClassIuWFNg
     * @param method
     *            BCEL̃\bhIuWFNg
     * @param document
     *            o͐XMLhLg
     * @throws IOException
     *             o͗OꍇB
     */
    private void processMethod(final JavaClass javaClass,
            final BlancoBcelClassStructure classStructure) throws IOException {

        final Method[] methods = javaClass.getMethods();
        for (int indexMethod = 0; indexMethod < methods.length; indexMethod++) {
            final Method method = methods[indexMethod];

            final BlancoBcelMethodStructure methodStructure = new BlancoBcelMethodStructure();
            classStructure.getMethodList().add(methodStructure);

            if (method.isPublic()) {
                methodStructure.setAccess("public");
            } else if (method.isProtected()) {
                methodStructure.setAccess("protected");
            } else if (method.isPrivate()) {
                methodStructure.setAccess("private");
            }

            methodStructure.setStatic(method.isStatic());
            methodStructure.setFinal(method.isFinal());

            methodStructure.setName(method.getName());

            final Type[] argumentTypes = method.getArgumentTypes();
            for (int indexArgument = 0; indexArgument < argumentTypes.length; indexArgument++) {
                methodStructure.getArgumentList().add(
                        argumentTypes[indexArgument].getSignature());
            }

            methodStructure.setReturn(method.getReturnType().getSignature());

            final Code code = method.getCode();
            if (code == null) {
                /**
                 * R[hꍇɂ͏f܂B
                 */
                return;
            }

            /**
             * \bĥȂ̃oCgR[hC^[ɏo͂܂B <br>
             * ɍۂVerbose[hOFFɂĂ܂B
             */
            final String result = Utility.codeToString(code.getCode(),
                    javaClass.getConstantPool(), 0, -1, false);
            final BufferedReader reader = new BufferedReader(new StringReader(
                    result));
            for (;;) {
                final String line = reader.readLine();
                if (line == null) {
                    break;
                }

                final BlancoBcelLineStructure lineStructure = new BlancoBcelLineStructure();
                methodStructure.getLinesList().add(lineStructure);

                final String[] splitedWithKolon = BlancoNameUtil.splitString(
                        line, ':');
                if (splitedWithKolon.length < 2) {
                    throw new IllegalArgumentException("sf[^̎擾ɕsȒl(" + line
                            + ") ̐(" + splitedWithKolon.length
                            + ")߂܂Bf܂B");
                }

                lineStructure.setNumber(Integer.parseInt(splitedWithKolon[0]));

                final String[] splitedWithTab = BlancoNameUtil.splitString(
                        splitedWithKolon[1], '\t');
                final String operation = splitedWithTab[0].trim();

                lineStructure.setOperation(operation);

                String operand = line;
                for (boolean isTabFound = false; operand.length() > 0; operand = operand
                        .substring(1)) {
                    if (isTabFound == false) {
                        // tabłB
                        if (operand.charAt(0) == '\t') {
                            // tab܂
                            isTabFound = true;
                        }
                    } else {
                        // 擪 ܂B
                        if (operand.charAt(0) == '\t') {
                            // s
                        } else {
                            // Ŋ]̉ӏɓBĂ܂B
                            break;
                        }
                    }
                }
                lineStructure.setOperand(operand);

                // if (IS_DEBUG) {
                // BlancoXmlUtil.addChildElement(document, eleLine,
                // "DebugOriginalLine", line);
                // }
            }
        }
    }
}
