/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.spawn;

import plus.io.Device;
import plus.io.Io;
import plus.io.IoHelper;
import plus.spawn.system.UtilInterface;

import java.io.FilterWriter;
import java.io.IOException;
import java.io.Writer;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * [%command%] implementation of the `date` command.
 * <p>
 * Usage: date [OPTION]... [+FORMAT] <br>
 * 現在時刻を有効なFORMATで表示する<br>
 * <br>
 * フォーマット指定は、 java.text.SimpleDateFormat 参照 <br>
 * </p>
 *
 * @author kunio himei.
 */
public final class SimpleDate extends FilterWriter implements UtilInterface {

    /**
     * USAGE.
     */
    private static final String USAGE = """
            Usage: date [OPTION]... [+FORMAT]
            Display the current time in the given FORMAT.

                  --help      display this help and exit (should be alone)""";
    /**
     * format.
     */
    private final String format;

    /**
     * date.
     */
    public SimpleDate(String[] args, Writer output) {
        super(output);
        StringBuilder flags = new StringBuilder(); // options
        String fmt = null;
        for (String arg : args) {
            char c = arg.charAt(0);
            if (('-' == c)) {
                flags.append(arg.substring(1));
            } else if (('+' == c)) {
                fmt = arg.substring(1);
            }
        }
        this.format = fmt;

        if ((0 <= flags.indexOf("v"))) {
            StringBuilder sb = new StringBuilder("`date");
            if (0 < flags.length()) {
                sb.append(" -").append(flags);
            }
            if (null != fmt) {
                sb.append(" +").append(fmt);
            }
            System.err.println(sb.append('`'));
        }
    }

    /**
     *
     */
    public static void main(String[] args) throws IOException {
        if ((0 != args.length) && args[0].startsWith("--help")) {
            System.out.println(USAGE);
        } else {
            new SimpleDate(args, Device.openOutput("", Io.STDOUT)).close();
        }
    }

    /**
     * このストリームを閉じる.
     */
    @Override
    public void close() throws IOException {
        Date date = new Date();
        String x = (null == this.format) ? //
                date.toString() : new SimpleDateFormat(this.format)
                .format(date);
        IoHelper.writeln(super.out, x, true);
//        Io.close(super.out); // PowerShell と相性が悪いためリファクタリング
        super.out.flush();
    }

    /**
     * サブプロセスの終了値を返す.
     */
    @Override
    public int exitValue() {
        return 0;
    }

    /**
     *
     */
    @Override
    public boolean hasInput() {
        return false;
    }
}