/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.io;

import plus.BiIO;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileTime;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static java.nio.file.StandardCopyOption.*;

/**
 * nanoShell - To the top of the world.
 *
 * @author kunio himei.
 */
public class NanoTools {
    private static final int LIST_CAPACITY = 2048;
    private static final int MOD_CAPACITY = 256 * 4 / 3; // 341
    private final Thread myThread = Thread.currentThread(); // 親スレッド
    private final ExecutorService pool = Executors.newWorkStealingPool(); // スレドプール
    //    private final ExecutorService pool =
//            Executors.newFixedThreadPool(8); // .close() が必要
    private final List<Path> arr = new ArrayList<>(LIST_CAPACITY); // 出力リスト
    private final List<Path> drop = new ArrayList<>(LIST_CAPACITY); // move input
    private final Map<Path, FileTime> mod = new HashMap<>(MOD_CAPACITY); // 日時設定
    private long start; // コマンド開始時刻(ゴミフォルダ判定にも使用する)
    private Analysis ana; // 入力フォルダの解析
    private AnalysisOut jal; // 出力フォルダの解析
    private AsyncCtrl async = new AsyncCtrl(); // async コントロール
    private boolean isRemove; // Removeの可否判定

    private int DISPLAY_WIDTH = 35; // 画面への表示幅
    private int MAX_PATH = 260; // 長いパスの警告を表示する敷居値
    private static final String DISPLAY_WIDTH_KEY = "DISPLAY_WIDTH"; // all
    private static final String MAX_PATH_KEY = "MAX_PATH"; // tree, ls

    /**
     * @param x command line arguments
     */
    private void initialize(String input, Object... x) {
        start = System.currentTimeMillis();
        ana = new Analysis(input, DISPLAY_WIDTH);
        jal = null; // 出力フォルダの解析
        async.clear(); // async コントロール(ダミー)
        isRemove = false; // never delete
        //noinspection ResultOfMethodCallIgnored
        Thread.interrupted(); // スレッドの割込みのクリア
        arr.clear(); // 出力リスト
        mod.clear(); // 日時設定
        drop.clear(); // moveの in側ファイルを遅延削除依頼
        optAll.clear(); // そのモジュールで利用可能なオプション
        optIn.clear(); // 明に適用されたオプション
        optARGV.clear(); // ユーザ定義された(生)オプション
        for (Object o : x) {
            String[] opts = o.toString().trim().toLowerCase()
                    .split("\\s+");
            optARGV.addAll(List.of(opts));
        }
        optARGV.remove(""); // ゴミを削除
        __STDOUT = defaultOUT; // リダイレクト(初期値)
        __STDERR = defaultERR;
        redirect(x);
    }

    private void close() {
        closeImpl(__STDOUT);
        closeImpl(__STDERR);
    }

    private void closeImpl(Redirect re) {
        try {
            BiIO.fflush(re.file);
            if (re.redirect)
                BiIO.close(re.file);
        } catch (IOException e) {
            throw new RuntimeException(exception("close", re.name));
        }
    }

    /**
     * 出力パスが ./foo/ - '/'で終わるとき、フォルダを作成する
     */
    private AnalysisOut analysisOut(Path input, String output) {
        String src = output.trim();
        Path out = Path.of(src);
        Path in = getParent(input);
        Path parent = out;
        boolean createHolder = false;
        if (src.endsWith("/")) {
            if (!Files.exists(out)) { // root X:/
                if (createFolder(in, out))
                    createHolder = true;
            }
        } else if (!Files.isDirectory(out)) {
            parent = getParent(out);
            if (!Files.exists(parent)) {
                if (createFolder(in, parent))
                    createHolder = true;
            }
        }
        // root Folder に日時を設定
        setModifiedTime(parent, getModifiedTime(in), THROW_ERROR);

        async = new AsyncCtrl(in, parent); // async コントロール
        return new AnalysisOut(out, createHolder);
    }

    /**
     * 非同期 I/O の完了 - Completion of asynchronous I/O.
     * フォルダ更新日付を設定
     */
    private void finishAsyncModTime(Progress pro, Path output) {
        if (isSystem(output)) return; // skip System folder
        if (Files.isDirectory(output)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(output)) {
                for (Path path : ds) {
                    if (Files.isDirectory(path)) {
                        if (isSystem(path)) continue; // skip System folder
                        finishAsyncModTime(pro, path);
                    }
                }
                if (removeStartFolder(output)) { // >= Start time
                    pro.decrementFolder();
                } else {
                    FileTime time = mod.get(output);
                    if (time != null) {
                        // 作成したフォルダのアクセス日時は、ここで設定する
                        setModifiedTime(output, time, THROW_ERROR);
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("cleanup", e));
            }
        }
    }

    // move 入力側ファイルをドロップ
    private void finishMoveDropFile(Path input) {
        for (Path path : drop) {
            removeSafetyFile(path); // moveの in側ファイルを削除依頼
        }
        finishMoveCleanFolder(getParent(input));
    }

    // move 入力側フォルダをクリーンアップ
    private void finishMoveCleanFolder(Path input) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (Files.isDirectory(path)) {
                        if (isSystem(path)) continue; // skip System folder
                        finishMoveCleanFolder(path);
                    }
                }
                removeSafetyFolder(input); // Time0フォルダも削除する
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("moveClean", e));
            }
        }
    }

    private static final int STATUS_ERROR = 3;
    private static final int STATUS_WARNING = 2;
    private static final int STATUS_FILE = 1; // request file
    private static final int STATUS_INFORMATION = STATUS_FILE;
    private static final int STATUS_NORMAL = 0; // folder or file

    private static final int CHK_ROOT = 1;
    private static final int CHK_DIRECTORY = 2;
    private static final int CHK_FILE = 4;

    /**
     * @param input Check paths for deletion
     * @param flags Allow root specification ('/') (e.g. tree)
     * @return STATUS
     */
    private static int checkPath(Path input, int flags) {
        Path path = input.toAbsolutePath();
        if (Files.isDirectory(path)) {
            boolean isRoot = path.equals(path.getRoot());
            if (isRoot && (flags & CHK_ROOT) == 0) {
                rootCannotBeSpecified(input);
                return STATUS_ERROR; // ルートフォルダは指定できない
            }
            return STATUS_NORMAL;
        } else if ((flags & CHK_FILE) != 0) {
            if (Files.exists(path) && !Files.isDirectory(path))
                return STATUS_FILE; // ファイルが検出された
            PathDoesNotExist(input);
            return STATUS_WARNING; // ファイルが存在しない
        }
        PathDoesNotExist(input); // 間違ったパス指定
        return STATUS_WARNING;
    }

    /**
     * Parsing redirect definitions
     *
     * @param x command line arguments (生データ)
     */
    private void redirect(Object... x) {
        for (Object o : x) {
            String arg = o.toString().trim();
            int ix1 = arg.indexOf('>');
            if (ix1 < 0) continue;
            int ix2 = arg.indexOf('>', ix1 + 2); // skip >>
            if (ix2 >= 0) {
                ix2 = arg.lastIndexOf(' ', ix2);
                if (ix2 >= 0) {
                    redirectImpl(arg.substring(0, ix2));
                    redirectImpl(arg.substring(ix2 + 1));
                } else {
                    throw new IllegalArgumentException(
                            exception("redirect", arg));
                }
            } else {
                redirectImpl(arg);
            }
        }
    }

    private static final String RE_QUOTE = "'";
    private static final String RE_ESCAPE_01 = "\\" + RE_QUOTE;
    private static final String RE_ESCAPE_02 = RE_QUOTE + RE_QUOTE;
    private static final String BOM_QUOTE_BE = "\uFEFF"; // UTF16-BE <- ''
    private static final Pattern RE_DIRECT = Pattern
            .compile("([12]?)(>{1,2})\\s*(" + RE_QUOTE + "?)(.+)(\\3)");
    //                          g1     g2          (g3)               g4  g5:前方参照
    private static final Pattern RE_FILE = Pattern
            .compile("(.+[.]\\S+)"); // 空白無しの拡張子付き

    /**
     * 'foo' → <'>foo<'> // ※※※
     * foo → <>foo<>
     * 'foo → <>'foo<>
     * foo' → <>foo'<>
     */
    private void redirectImpl(String redirect) {
        String src = redirect.trim()
                .replace(RE_ESCAPE_01, BOM_QUOTE_BE) // \'
                .replace(RE_ESCAPE_02, BOM_QUOTE_BE); // ''
        Matcher m = RE_DIRECT.matcher(src);
        if (m.find()) {
            String rno = getValue(m.group(1));
            String rid = getValue(m.group(2));
            String bracket = getValue(m.group(3));
            String file = getValue(m.group(4)).trim();
//            String bracket2 = getValue(m.group(5));

//            messageGREEN("redirect " + m.group(0),
//                    "<" + m.group(3) + ">" + file + "<" + m.group(5) + ">");
            // 前方参照のため g3、g5 は必ず同一値、３<'> ５<>は無い -->(<>'foo<>)
            if (bracket.isEmpty()) {
                if (file.contains(RE_QUOTE)) { // 'foo
                    messageMAGENTA("Redirect, Paired <'> mistake", redirect);
                    file = file.replaceFirst(RE_QUOTE, "");
                }
                Matcher mx = RE_FILE.matcher(file); // 空白無しの拡張子付き
                if (mx.find()) {
                    file = getValue(mx.group(1)).trim();
//                    messageGREEN("redirect (1)", file);
                } else {
                    int ix = file.indexOf(' '); // foo< >bar
                    if (ix >= 0) {
                        file = file.substring(0, ix); // foo
//                        messageGREEN("redirect (2)", file);
                    }
                }
            }
            file = file.replace(BOM_QUOTE_BE, RE_QUOTE);
            String name = rno + rid + file;
            Redirect re = new Redirect(true, name,
                    rid, file, rno.equals("2"));
            if (re.isSTDERR) this.__STDERR = re;
            else this.__STDOUT = re;
            optIn.add(name); // Used in option display
        }
    }

    private Redirect __STDOUT; // リダイレクト
    private Redirect __STDERR; //
    private static final Redirect defaultOUT =
            new Redirect(false, "", "", Io.STDOUT, false);
    private static final Redirect defaultERR =
            new Redirect(false, "", "", Io.STDERR, true);

    // Optional parameter variable
    private final Map<String, Object> optAll = new TreeMap<>(); // そのモジュールで利用可能なオプション
    private final Set<String> optIn = new TreeSet<>(); // 明に適用されたオプション
    private final Set<String> optARGV = new HashSet<>(32); // ユーザ定義された(生)オプション

    private static final String CLEAN = "clean"; // clean
    private static final String COMMA = ","; // ls size
    private static final String FILE = "file"; // tree
    private static final String MX_260 = "260"; // ls
    private static final String MX_no260 = "no260"; // ls
    private static final String PATH = ""; // ls  length
    private static final String PRO = "progress"; // copy, move, remove
    private static final String ROOT = "root"; // remove
    private static final String SIMPLE = "0"; // ls size
    private static final String SYNC = "sync"; // copy
    private static final String UNIT = "k"; // ls size
    private static final String noRECURSIVE = "noRecursive";
    private static final String noTIME = "noTime";

    /*
     * 大小文字を区別しないで前方一致で比較
     */
    private boolean applyOption(String name, boolean... value) {
        boolean val = value.length == 0; // 省略時は真
        String key = name.toLowerCase();
        String key2 = key.startsWith("no") ?
                '-' + key.substring(2) : key;
        for (String x : optARGV) {
            if (x.isEmpty()) continue;
            if (key.startsWith(x) || key2.startsWith(x)) {
                optIn.add(name);
                optAll.put(name, val);
                return val;
            }
        }
        optAll.put(name, !val);
        return !val;
    }

    private static final Pattern IS_SET = Pattern
            .compile("(\\w+)=([-+]?\\d+)");

    private int applyShellVariable(String name, int value) {
        String key = name.toLowerCase();
        for (String x : optARGV) {
            if (x.isEmpty()) continue;
            Matcher m = IS_SET.matcher(x);
            while (m.find()) {
                String g1 = getValue(m.group(1));
                if (key.startsWith(g1)) {
                    String g2 = getValue(m.group(2));
                    int val = Integer.parseInt(g2);
                    String var = name + '=' + g2;
                    optIn.add(BLUE + var + RESET);
                    return val;
                }
            }
        }
        optIn.add(name + '=' + value);
        return value;
    }

    private boolean getBooOption(String key) {
        if (optAll.containsKey(key)) {
            Object o = optAll.get(key);
            if (o instanceof Boolean e)
                return e;
        }
        throw new RuntimeException(exception("No option", key));
    }

    @SuppressWarnings({"all"}) // ※※※
    private void resetOptions(String... x) {
        for (String key : x) {
            if (optAll.containsKey(key)) {
                optAll.put(key, false);
                optIn.remove(key);
            } else {
                throw new RuntimeException(exception("No option", key));
            }
        }
    }

    private String listOptions() {
        StringBuilder sb = new StringBuilder(64);
        sb.append(BLUE);
        for (String x : optIn) {
            sb.append(' ').append(x);
        }
        return sb.append(RESET).toString();
    }

    /**
     * ファイルまたはディレクトリの名前を返す
     *
     * @param path D:/ -> null
     * @return ""(null)
     */
    private static String getFileName(Path path) {
        Path name = path.getFileName();
        return name == null ? "" : name.toString();
    }

    /**
     * フォルダの場合はそのフォルダ、以外は親フォルダを返す
     */
    private Path getParent(Path path) {
        if (Files.isDirectory(path)) return path;
        Path pa = path.getParent();
        if (pa == null)
            pa = path.toAbsolutePath().normalize().getParent();
        if (pa == null)
            throw new RuntimeException(exception("getParent", path));
        return pa;
    }

    // System フォルダをスキップする
    private static final String SYSTEM =
            "System Volume Information".toLowerCase();
    private static final String RECYCLE =
            "$RECYCLE.BIN".toLowerCase(); // C:$Recycle.Bin

    private static boolean isSystem(Path path) {
        String str = path.toString().toLowerCase();
        return str.contains(SYSTEM) ||
                str.contains(RECYCLE);
    }

    // ワイルドカードにマッチング
    private boolean isMatch(Path path) {
        if (Files.exists(path) && !Files.isDirectory(path)) {
            if (ana.alwaysTrue) return true;
            String name = getFileName(path);
            return ana.regex.matcher(name).matches();
        }
        return false;
    }

    /**
     * コマンド起動(start)以降に作成さた空フォルダを削除
     */
    private boolean removeStartFolder(Path path) {
        if (Files.isDirectory(path)) {
            boolean started = getModifiedTime(path).toMillis() >= start;
            return started && removeSafetyFolder(path);
        }
        return false;
    }

    /**
     * Time0 の空フォルダ以外を削除 - Other than Time0.
     */
    private boolean removeNoTime0Folder(Path path) {
        if (Files.isDirectory(path)) {
            boolean time0 = getModifiedTime(path).toMillis() != 0;
            return time0 && removeSafetyFolder(path);
        }
        return false;
    }

    /**
     * throw を発生させない (フォルダが空でない場合の対策)
     */
    private boolean removeSafetyFolder(Path path) {
        return isRemove && !isSystem(path) &&
                Files.isDirectory(path) && path.toFile().delete();
    }

    /**
     * ファイルを削除
     */
    private boolean removeSafetyFile(Path path) {
        try {
            return isRemove && !isSystem(path) &&
                    Files.exists(path) && !Files.isDirectory(path) &&
                    Files.deleteIfExists(path);
        } catch (IOException e) {
            throw new RuntimeException(exception("removeSafetyFile", e));
        }
    }

    /**
     * ファイルサイズを取得
     */
    private long getFileSize(Path path) {
        try {
            return Files.exists(path) && !Files.isDirectory(path) ?
                    Files.size(path) : 0;
        } catch (IOException e) {
            throw new RuntimeException(exception("getFileSize", e));
        }
    }

    /**
     * 　更新日時を取得
     */
    private FileTime getModifiedTime(Path path) {
        try {
            if (Files.exists(path))
                return Files.getLastModifiedTime(path);
            return FileTime.fromMillis(0);

        } catch (IOException e) {
            throw new RuntimeException(exception("getModifiedTime", e));
        }
    }

    private static final boolean THROW_ERROR = true;
    private static final boolean EAT_ERROR = false;

    // 更新日時を設定 (error 処理の可否)
    private void setModifiedTime(Path path, FileTime time, boolean error) {
        try {
            if (Files.exists(path)) {
                FileTime curTime = getModifiedTime(path);
                if (curTime.equals(time)) return;
                Files.setLastModifiedTime(path, time);
            }
        } catch (Exception e) { // AccessDeniedException
            if (error) {
                messageRED("setModifiedTime(ro.)", path);
                throw new RuntimeException(exception("setModifiedTime", e));
                // ※※ \\prince6\share\nano\VSCode\.vscode
            }
        }
    }

    /**
     * 出力側のホルダを作成し、タグとして Time0 日時を設定
     *
     * @param input null を許容する
     */
    private boolean createFolder(Path input, Path output) {
        try {
            FileTime time = getModifiedTime(input);
            mod.put(output, time);
            if (!Files.exists(output)) {
                Files.createDirectories(output); // ホルダを作成
                if (time.toMillis() == 0) {
                    // Time0 日時を設定
                    setModifiedTime(output, time, THROW_ERROR);
                }
                return true;
            }
        } catch (IOException e) { // AccessDeniedException
            messageRED("Make directory", output);
            throw new RuntimeException(e);
        }
        return false;
    }

    private static final boolean _STDOUT = true;
    private static final boolean _STDERR = false;

    /**
     * @param stdout [Default value: hasSTDOUT] else, hasSTDERR
     */
    private void printX(boolean stdout, String x) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        BiIO.print(re.rid, re.file, x);
    }

    /**
     * @param stdout True: hasSTDOUT, False: hasSTDERR
     * @param type   [Default value:  value] else, USE_PATH_NAME
     */
    private void printX(boolean stdout, Path path, boolean... type) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        if (re.redirect) {
            printX(stdout, applySlashSeparator(path, type));
        } else {
            printX(stdout, pack(path, type));
        }
    }

    /**
     * @param stdout True: hasSTDOUT, False: hasSTDERR
     * @param type   [Default value:  value] else, USE_PATH_NAME
     */
    @SuppressWarnings("SameParameterValue")
    private String sprintX(boolean stdout, Path path, boolean... type) {
        Redirect re = stdout ? __STDOUT : __STDERR;
        if (re.redirect) {
            return applySlashSeparator(path, type);
        } else {
            return pack(path, type);
        }
    }

    /**
     * set - シェル変数の設定.
     */
    public int set(Object... x) {
        initialize("./", x);
        boolean hasARGV = optARGV.size() > 0;
        DISPLAY_WIDTH = applyShellVariable(DISPLAY_WIDTH_KEY, DISPLAY_WIDTH);
        MAX_PATH = applyShellVariable(MAX_PATH_KEY, MAX_PATH);
        String args = hasARGV ? listOptions() : "";
        messageTitle("set", args);
        if (!hasARGV) {
            for (String k : optIn) {
                System.out.println(k);
            }
        }
        messageFinish("Number of processed", optIn.size(), "");
        return STATUS_NORMAL;
    }

    /**
     * clean - クリーンアップ.
     */
    public int clean(String input, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        isRemove = true; // always delete
        boolean isClean = applyOption(CLEAN);
        boolean isTime = applyOption(noTIME, false);
        String args = ana.virtualPath + ' ' + listOptions();
        messageTitle("clean", args);
        Clean cln = new Clean();
        int ri = checkPath(in, CHK_ROOT | CHK_DIRECTORY);
        if (ri <= STATUS_INFORMATION) {
            cleanupFolderImpl(in, cln, true, isTime, isClean);
        }
        messageFinish("Number of processed",
                cln.file, cln.folder);
        return STATUS_NORMAL;
    }

    private long cleanupFolderImpl(Path input, Clean cln, boolean warning,
                                   boolean isTime, boolean isClean) {
        if (isSystem(input)) return 0; // skip System folder
        long modTime = 0;
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    long time;
                    if (Files.isDirectory(path)) {
                        time = cleanupFolderImpl(path, cln, warning,
                                isTime, isClean);
                    } else {
                        time = getModifiedTime(path).toMillis();
                        cln.file += 1;
                    }
                    if (modTime < time) modTime = time;
                }
                if (Files.isDirectory(input)) { // 更新日時を設定
                    long time2 = getModifiedTime(input).toMillis();
                    if (isTime && time2 != modTime) {
                        FileTime fileTime = FileTime.fromMillis(modTime);
                        // クリーンナップ 日時を設定
                        setModifiedTime(input, fileTime, EAT_ERROR);
                    }

                    if (warning && !isClean && modTime == 0) {
                        messageMAGENTA("Empty folder", pack(input));

                    } else if (isClean && removeSafetyFolder(input)) {
                        if (warning)
                            messageMAGENTA("remove", pack(input));
                        cln.folder -= 1;
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("clean", e));
            }
            cln.folder += 1;
        }
        return modTime;
    }

    // Clean Results
    private static class Clean {
        private int folder; // フォルダ数
        private int file; // ファイル数
    }

    /**
     * copy the file.
     * - removeはルートの親フォルダを削除するため、copyのフォルダ数 +1 となる
     * (※ PATCH, copy側で補正する)
     *
     * @param input  file or folder
     * @param output file or folder
     */
    public int copy(String input, String output, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        jal = analysisOut(in, output);
        Path out = jal.out;
        isRemove = true; // always delete
        boolean isPro = applyOption(PRO);
        boolean isSync = applyOption(SYNC);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + ' ' +
                pack(output) + listOptions();
        messageTitle("copy", args);
        if (in.equals(out)) { // same
            inputAndOutputAreSamePath(in);
            return STATUS_ERROR; // in, out are the same
        }
        int ri = checkPath(in, CHK_ROOT | CHK_DIRECTORY | CHK_FILE);
        int ro = checkPath(out, CHK_ROOT | CHK_DIRECTORY | CHK_FILE);
        ri = Math.max(ri, ro);
        if (ri <= STATUS_INFORMATION) {
            Progress pro = new Progress(_STDERR, isPro);
            if (isSync) { // synchronous copy
                copySync(pro, getParent(in), getParent(out));
                pro.flush();
                if (pro.printX) {
                    for (Path sync : arr) {
                        printX(_STDERR, sync);
                    }
                }
                messageCYAN("Synchronized", pro.fileNumber);
            }
            pro = new Progress(_STDOUT, isPro);
            if (jal.hasCreatedHolder) pro.incrementFolder(); // ※ PATCH
            copyImpl(pro, in, out, isRecursive);
            pro.flush();
            if (pro.printX) {
                for (Path copy : arr) {
                    printX(_STDOUT, copy);
                }
            }
            finishAsyncModTime(pro, getParent(out));
            messageFinish("Number of processed", pro.fileNumber, pro.fileSize);
        }
        close();
        return ri;
    }

    private void copyImpl(Progress pro,
                          Path input, Path output, boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    Path newOut = output.resolve(path.getFileName());
                    if (Files.isDirectory(path)) {
                        if (recursive) {
                            if (createFolder(path, newOut)) // ※
                                pro.incrementFolder();
                            //noinspection ConstantValue
                            copyImpl(pro, path, newOut, recursive);
                        }
                    } else if (isMatch(path)) {
                        atomicCopy(pro, path, newOut);
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("copy", e));
            }
        } else if (isMatch(input)) { // ファイル名指定で呼ばれたとき
            atomicCopy(pro, input, output);
        }
    }

    /**
     * Synchronous copy
     */
    private void copySync(Progress pro, Path input, Path output) {
        Map<String, Path> map = new HashMap<>(1024);
        if (isSystem(input) || isSystem(output)) return; // skip System folder
        if (Files.isDirectory(output)) { //
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(output)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    String name = getFileName(path);
                    map.put(name, path);
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("sync", e));
            }
        }
        if (Files.exists(output)) {
            String name = getFileName(output);
            map.put(name, output);
        }
        if (Files.isDirectory(input)) { // input
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    String name = getFileName(path);
                    map.remove(name);
                    if (Files.isDirectory(path)) {
                        Path newOut = output.resolve(path.getFileName());
                        copySync(pro, path, newOut);
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("sync", e));
            }
        }
        if (Files.exists(input)) {
            String name = getFileName(input);
            map.remove(name);
        }
        map.remove("");
        for (Path path : map.values()) {
            copySyncRemove(pro, path); // Synchronous
        }
    }

    /**
     * Synchronous copy - 指定されたパスを無条件に削除
     *
     * @param input 　file or folder
     */
    private void copySyncRemove(Progress pro, Path input) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    if (Files.isDirectory(path)) {
                        copySyncRemove(pro, path);
                    } else { // ワイルドカード無効で良いか？
                        long size = getFileSize(path);
                        if (removeSafetyFile(path)) {
                            pro.addFile(path, size);
                        }
                    }
                }
                if (removeSafetyFolder(input)) // ※親フォルダを削除
                    pro.incrementFolder();

            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("sync", e));
            }
        } else {
            long size = getFileSize(input);
            if (removeSafetyFile(input)) // ファイル名指定で呼ばれたとき
                pro.addFile(input, size);
        }
    }

    /**
     * move the file.
     *
     * @param input  file or folder
     * @param output file or folder
     */
    public int move(String input, String output, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        jal = analysisOut(in, output);
        Path out = jal.out;
        isRemove = true; // always delete
        boolean isPro = applyOption(PRO);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + ' ' +
                pack(output) + listOptions();
        messageTitle("move", args);
        if (in.equals(out)) { // same
            inputAndOutputAreSamePath(in);
            return STATUS_ERROR; // in, out are the same
        }
        int ri = checkPath(in, CHK_ROOT | CHK_DIRECTORY | CHK_FILE);
        int ro = checkPath(out, CHK_ROOT | CHK_DIRECTORY | CHK_FILE);
        ri = Math.max(ri, ro);
        if (ri <= STATUS_INFORMATION) {
            Progress pro = new Progress(_STDOUT, isPro);
            if (jal.hasCreatedHolder) pro.incrementFolder(); // ※ PATCH
            moveImpl(pro, in, out, isRecursive);
            pro.flush();
            if (pro.printX) {
                for (Path move : arr) {
                    printX(_STDOUT, move);
                }
            }
            finishAsyncModTime(pro, getParent(out));
            finishMoveDropFile(in);
            messageFinish("Number of processed", pro.fileNumber, pro.fileSize);
        }
        close();
        return ri;
    }

    private void moveImpl(Progress pro,
                          Path input, Path output, boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    Path newOut = output.resolve(path.getFileName());
                    if (Files.isDirectory(path)) {
                        if (recursive) {
                            if (createFolder(path, newOut)) // ※
                                pro.incrementFolder();
                            //noinspection ConstantValue
                            moveImpl(pro, path, newOut, recursive);
                        }
                    } else if (isMatch(path)) {
                        atomicMove(pro, path, newOut);
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("move", e));
            }
        } else if (isMatch(input)) { // ファイル名指定で呼ばれたとき
            atomicMove(pro, input, output);
        }
    }

    /**
     * 単一のファイルを差分コピー (出力側に存在しない、日付時刻・サイズが異なる場合)
     */
    private void atomicCopy(Progress pro, Path input, Path output) {
        AsyncAction act = isAtomicSameFile(input, output);
        output = act.path;
        if (output == null ||
                act.action == ACT_SAME_FILE) return; // 同一ファイル

        async.atom.incrementAndGet(); // インクリメント
        pool.submit(new AsyncIO(pro, _COPY, act.action, input, output));
    }

    /**
     * ファイルをターゲット・ファイルに移動するか、そのファイル名を変更
     */
    private void atomicMove(Progress pro, Path input, Path output) {
        AsyncAction act = isAtomicSameFile(input, output);
        output = act.path;
        if (output == null) return;

        if (act.action == ACT_SAME_FILE) // move を Skip
            drop.add(input); // moveの in側ファイルを削除依頼
        async.atom.incrementAndGet(); // インクリメント
        pool.submit(new AsyncIO(pro, _MOVE, act.action, input, output));
    }

    //////////////////////////////////////////////////////////////////////
    // AsyncAction.                     action
    private static final int ACT_SKIP = -2;      // null (スキップ)
    private static final int ACT_SAME_PATH = -1; // null (スキップ、同一パス)
    private static final int ACT_SAME_FILE = 0;  //  (同一ファイル)
    private static final int ACT_ACCEPT = 1;     // 　(処理を実行)
    private static final boolean _COPY = true;   // use AsyncIO
    private static final boolean _MOVE = false;

    private static final AsyncAction ACTION_HAS_SKIP =
            new AsyncAction(ACT_SKIP, null);
    private static final AsyncAction ACTION_HAS_SAME_PATH =
            new AsyncAction(ACT_SAME_PATH, null);

    // FAT の時間誤差を除外
    private static final long FAT_TIME_ERROR = 2000; // 2sec.

    /**
     * ファイル属性が同一かどうかの判定
     *
     * @param input  パス(ファイル名)
     * @param output パス(ファイル名) or フォルダ
     * @return AsyncAction 状態コード、出力パス(null: 要件に該当しないとき、skip)
     */
    private AsyncAction isAtomicSameFile(
            Path input, Path output) {
        if (!Files.exists(input) || Files.isDirectory(input))
            throw new RuntimeException(
                    exception("Folders are not allowed", input));
        if (!isMatch(input) || isSystem(input)) // wildcard mismatch
            return ACTION_HAS_SKIP; // wildcard unmatched, skip System folder
        if (Files.isDirectory(output)) {  // out がフォルダならファイル名を設定
            output = output.resolve(input.getFileName());
        }
        if (input.equals(output)) {
            inputAndOutputAreSamePath(input); // Message
            return ACTION_HAS_SAME_PATH; // 同一ファイルを指す場合
        }

        Path outParent = getParent(output); // out の親を取得
        if (!Files.exists(outParent)) {
            Path inParent = getParent(input); // in の親を取得
            createFolder(inParent, outParent); // ※
        }
        if (Files.exists(output) && !Files.isDirectory(output) &&
                input.getFileName().equals(output.getFileName())) {
            long iSize = getFileSize(input); // Check file attributes
            long oSize = getFileSize(output);
            long iMod = getModifiedTime(input).toMillis() / FAT_TIME_ERROR;
            long oMod = getModifiedTime(output).toMillis() / FAT_TIME_ERROR;
            if (iSize == oSize && iMod == oMod) { // FAT(誤差2sec) <> NTFS
                return new AsyncAction(ACT_SAME_FILE, output);
            } else if (iMod < oMod) {
                return ACTION_HAS_SKIP; // The output file is newer
            }
        }
        return new AsyncAction(ACT_ACCEPT, output);
    }

    /**
     * remove - 指定されたパス以下にあるワイルドカードに一致するファイルを削除する
     * - Delete files matching wildcards under the specified
     * - Path, Files version
     * - removeはルートの親フォルダを削除するため、copyのフォルダ数 +1 となる
     * (※ PATCH, copy側で補正する)
     *
     * @param input Requires at least one sub-
     */
    public int remove(String input, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        isRemove = true; // always delete
        boolean isPro = applyOption(PRO);
        boolean isRoot = applyOption(ROOT);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + listOptions();
        messageTitle("remove", args);
        int check = CHK_DIRECTORY | CHK_FILE;
        if (isRoot) check |= CHK_ROOT;
        int ri = checkPath(in, check);
        if (ri <= STATUS_INFORMATION) {
            Progress pro = new Progress(_STDOUT, isPro);
            removeImpl(pro, in, isRecursive);
            pro.flush();
            if (pro.printX) {
                for (Path remove : arr) {
                    if (!Files.isDirectory(remove))
                        printX(_STDOUT, remove);
                }
            }
            messageFinish("Number of processed", pro.fileNumber, pro.fileSize);
        }
        close();
        return ri;
    }

    private void removeImpl(Progress pro, Path input,
                            boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    if (Files.isDirectory(path)) {
                        if (recursive) {
                            //noinspection ConstantValue
                            removeImpl(pro, path, recursive);
                        }
                    } else if (isMatch(path)) {
                        long size = getFileSize(path);
                        if (removeSafetyFile(path)) { // ファイルを削除
                            pro.addFile(path, size);
                        }
                    }
                }
                if (removeNoTime0Folder(input)) { // Other than Time0
                    pro.incrementFolder();
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("remove", e));
            }
        } else if (isMatch(input)) { // ファイル名指定で呼ばれたとき
            long size = getFileSize(input);
            if (removeSafetyFile(input))
                pro.addFile(input, size);
        }
    }

    /**
     * tree.
     *
     * @param input Path to start listing
     */
    public int tree(String input, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        isRemove = false; // never delete
        boolean isFile = applyOption(FILE);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        String args = ana.virtualPath + listOptions();
        messageTitle("tree", args);
        int ri = checkPath(in, CHK_ROOT | CHK_DIRECTORY);
        if (ri <= STATUS_INFORMATION) {
            int max = treeImpl(in, "", isFile, isRecursive);
            messageFinish("MAX_PATH", max, " chars");
        }
        close();
        return ri;
    }

    @SuppressWarnings("ConstantValue")
    private int treeImpl(Path input, String indent, boolean isFile, boolean recursive) {
        if (isSystem(input)) return 0; // skip System folder
        int max = 0;
        if (Files.isDirectory(input)) {
            // 大量なら ArrayList#sort を検討
            Set<Path> tree = new TreeSet<>();
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    tree.add(path);
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("ls", e));
            }
            for (Path path : tree) { // ファイル処理
                if (isMatch(path)) {
                    if (isFile) {
                        String x = indent + "| " + packTree(path);
                        printX(_STDOUT, x);
                    }
                    max = Math.max(max, path.toString().length());
                }
            }
            for (Path path : tree) { // フォルダ処理
                if (Files.isDirectory(path)) {
                    String x = indent + '/' + packTree(path);
                    printX(_STDOUT, x);
                    max = Math.max(max, path.toString().length());
                    if (recursive) {
                        max = Math.max(max, treeImpl(path,
                                indent + " ", isFile, recursive));
                    }
                }
            }
            // ファイル名指定で呼ばれたとき(認められない)
        }
        return max;
    }

    /**
     * ls - list segments.
     *
     * @param input Path to start listing
     */
    public int ls(String input, Object... x) {
        initialize(input, x);
        Path in = ana.path;
        isRemove = false; // never delete
        boolean isPath = applyOption(PATH);
        boolean isSimple = applyOption(SIMPLE);
        boolean isComma = applyOption(COMMA);
        boolean isUnit = applyOption(UNIT);
        boolean is260 = applyOption(MX_260);
        boolean isNo260 = applyOption(MX_no260);
        boolean isRecursive = applyOption(noRECURSIVE, false);
        boolean isAttr = isPath || isSimple || isComma || isUnit;
        if (isPath) {
            resetOptions(SIMPLE, COMMA, UNIT); // オプションの無効化
        }
        String args = ana.virtualPath + listOptions();
        messageTitle("ls", args);
        int ri = checkPath(in, CHK_ROOT | CHK_DIRECTORY);
        if (ri <= STATUS_INFORMATION) {
            Progress pro = new Progress(_STDOUT, false);
            lsImpl(pro, in, is260, isNo260, isRecursive);
            pro.flush();
            if (pro.printX) {
                for (Path path : arr) {
                    String ls = lsAttr(path, isAttr);
                    printX(_STDOUT, ls);
                }
            }
            messageFinish("Number of processed", pro.fileNumber, "");
        }
        close();
        return ri;
    }

    private void lsImpl(Progress pro, Path input,
                        boolean is260, boolean isNo260, boolean recursive) {
        if (isSystem(input)) return; // skip System folder
        if (Files.isDirectory(input)) {
            try (DirectoryStream<Path> ds = Files.newDirectoryStream(input)) {
                for (Path path : ds) {
                    if (isSystem(path)) continue; // skip System folder
                    if (Files.isDirectory(path)) {
                        if (recursive) {
                            //noinspection ConstantValue
                            lsImpl(pro, path, is260, isNo260, recursive);
                        }
                    } else {
                        lsSelect260(pro, path, is260, isNo260);
                    }
                }
            } catch (IOException e) { // NotDirectoryException
                throw new RuntimeException(exception("ls", e));
            }
            // ファイル名指定で呼ばれたとき(認められない)
        }
    }

    private String lsAttr(Path input, boolean isAttr) {
        StringBuilder sb = new StringBuilder(256);
        if (isAttr) {
            SimpleDateFormat sdf = new SimpleDateFormat(
                    "yy/MM/dd HH:mm");
            long time = getModifiedTime(input).toMillis();
            String daytime = sdf.format(time);
            String len = lsLength(
                    getBooOption(PATH) ?
                            input.toString().length() : //  length
                            getFileSize(input)); // file size
            sb.append(daytime).append('\t');
            sb.append(len).append('\t');
        }
        String path = sprintX(_STDOUT, input);
        return sb.append(path).toString();
    }

    /**
     * Output long
     *
     * @param is260   Output paths longer than 260
     * @param isNo260 Output paths under 260
     */
    private void lsSelect260(Progress pro, Path input,
                             boolean is260, boolean isNo260) {
        if (isMatch(input)) {
            if (is260) {
                if (isLonger260(input)) pro.addFile(input);
            } else if (isNo260) {
                if (!isLonger260(input)) pro.addFile(input);
            } else {
                pro.addFile(input);
            }
        }
    }

    private boolean isLonger260(Path input) {
        return input.toString().length() > MAX_PATH;
    }

    private String lsLength(long len) {
        if (getBooOption(PATH)) {
            return String.format("%, 7d", len); // max 32k, 6 char
        }
        if (getBooOption(COMMA)) {
            return String.format("%,d", len);
        }
        if (getBooOption(UNIT)) {
            return formatSize(len);
        }
        return Long.toString(len);
    }

    @SuppressWarnings("SpellCheckingInspection")
    private static final String NUMBER_UNIT = "BKMGTPEZY";

    /**
     * @return 前後空白に挟まれているため、注意！
     */
    private static String formatSize(long length) {
        double len = length;
        for (int i = 0; i < NUMBER_UNIT.length(); i++) {
            if (len < 1024.) {
                String unit = Character.toString(NUMBER_UNIT.charAt(i));
                String str = "  " + String.format("%3.1f%s", len, unit);
                return str.substring(str.length() - 6);
            }
            len /= 1024.;
        }
        return Long.toString((long) len);
    }

    private static final boolean USE_PATH_NAME = true; // Tree でファイル名を表示

    /**
     * 切り詰めたファイル名・IDを返す (returns a truncated /filename)
     */
    private String packTree(Path input) {
        StringBuilder sb = new StringBuilder(DISPLAY_WIDTH + 32);
        sb.append(sprintX(_STDOUT, input, USE_PATH_NAME)).append(' ');
        String path = input.toString();
        String name = getFileName(input);
        String info = name.length() + "/" + path.length();
        String maxPathID = isLonger260(input) ? " *" : "";
        sb.append(maxPathID.isEmpty() ? info :
                color(MAGENTA, info + maxPathID));
        return sb.toString();
    }

    /**
     * パスを切り詰める
     *
     * @param type [Default value:  value] else, USE_PATH_NAME
     */
    private String pack(Path path, boolean... type) {
        String str = applySlashSeparator(path, type);
        int n = str.replaceAll("[^/]+", "").length();
        int len = str.length();
        while (n-- > 2 && len > DISPLAY_WIDTH) {
            str = str.replaceFirst("^[^/]*/[^/]*", "");
            len = str.length();
        }
        int del = len - DISPLAY_WIDTH;
        if (del > 0) {
            int last = str.lastIndexOf('/');
            if (last >= 0) {
                String file = str.substring(last); // /path/file
                if (file.length() >= DISPLAY_WIDTH) {
                    str = file; // fileを折りたたむ
                } else { // pathを折りたたむ
                    int top = Math.max(0, last - del - 1);
                    str = str.substring(0, top) + "…" + file;
                }
            }
        }
        return pack(str, DISPLAY_WIDTH);
    }

    private String pack(String source) {
        return pack(source, DISPLAY_WIDTH);
    }

    /**
     * Analysis コンストラクタから呼ばれるため　static 必須
     * ※ ターミナルでは漢字は、全角表示される(下記の式では対応できない)
     *
     * @param DISPLAY_WIDTH 画面への表示幅
     */
    private static String pack(String source, int DISPLAY_WIDTH) {
        int len = source.length();
        if (len > DISPLAY_WIDTH) {
            int half = DISPLAY_WIDTH / 2;
            return source.substring(0, half) + "…" +
                    source.substring(len - half + 1);
        }
        return source;
    }

    /**
     * パス区切り文字を変換
     *
     * @param type [Default value:  value] else, USE_PATH_NAME
     */
    private static String applySlashSeparator(Path input, boolean... type) {
        if (type.length != 0)
            return getFileName(input); // これは '\'を含まない
        String path = input.toString();
        return File.separatorChar == '/' ? path : // UNIX
                path.replace('\\', '/'); // Windows
    }

    //////////////////////////////////////////////////////////////////////
    // Message
    private static String color(String color, String message) {
        return color + message + RESET;
    }

    private static void messageTitle(String name, Object arg) {
        System.out.println(color(YELLOW, name + ' ') + arg);
    }

    /**
     * @param arg2 AtomicLong, Integer, else --> String
     */
    private void messageFinish(String name, Number arg1, Object arg2) {
        String cyan = color(CYAN, name + ": ");
        String ela = elapsedTime(System.currentTimeMillis() - start);
        String str1 = String.format(", %,d", arg1.intValue());
        String str2;
        if (arg2 instanceof AtomicLong e)
            str2 = " (" + formatSize(e.get()).trim() // need trim
                    .replace(".0B", "B") + ')';
        else if (arg2 instanceof Number e)
            str2 = String.format(" (%,d)", e.intValue());
        else str2 = arg2.toString();
        String foo = cyan + ela + str1 + str2;
        messageSync(foo);
    }

    private static String elapsedTime(long millis) {
        Calendar cl = Calendar.getInstance();
        cl.setTimeInMillis(millis); // FIXME GMTより9時間進んでるはず！
        cl.set(Calendar.ZONE_OFFSET, -6 * 60 * 60 * 1000);
        SimpleDateFormat sdf = new SimpleDateFormat("H:m:s.SSS");
        return sdf.format(cl.getTime());
    }

    private static synchronized void messageSync(Object arg) {
        System.err.println(arg);
    }

    @SuppressWarnings("SameParameterValue")
    private static void messageCYAN(String name, Object arg) {
        String str;
        if (arg instanceof Number e)
            str = String.format("%,d", e.intValue());
        else str = arg.toString();
        messageSync(color(CYAN, name + ": ") + str);
    }

    @SuppressWarnings("unused, SameParameterValue")
    private static void messageGREEN(String name, Object arg) { // debug
        messageSync(color(GREEN, name + ": ") + arg);
    }

    @SuppressWarnings("unused")
    private static void messageBLUE(String name, Object arg) {
        messageSync(color(BLUE, name + ": ") + arg);
    }

    private static void messageMAGENTA(String name, Object arg) {
        messageSync(color(MAGENTA, name + ": ") + arg);
    }

    private static void messageRED(String name, Object arg) {
        messageSync(color(RED, name + ": ") + arg);
    }

    private static String exception(String name, String str) {
        return color(RED, name + ": ") + str;
    }

    @SuppressWarnings("SameParameterValue")
    private String exception(String name, Path path) {
        return color(RED, name + ": ") + pack(path);
    }

    private static String exception(String name, Throwable e) {
        return color(RED, name + ": ") + e;
    }

    // ルートは指定できません
    private static void rootCannotBeSpecified(Path path) {
        messageRED("Root folder cannot be specified",
                applySlashSeparator(path));
    }

    // 入力と出力は同じファイル
    private static void inputAndOutputAreSamePath(Path path) {
        messageRED("Input and  are the same ",
                applySlashSeparator(path));
    }

    // パスが存在しない - 'No such file or directory'
    private static void PathDoesNotExist(Path path) {
        messageMAGENTA("Path does not exist",
                applySlashSeparator(path));
    }

    private static final String RESET = "\033[m";       // \033[0m
    private static final String RED = "\033[91m";       // error
    private static final String GREEN = "\033[92m";     // debug
    private static final String YELLOW = "\033[93m";    // title
    private static final String BLUE = "\033[94m";      // options
    private static final String MAGENTA = "\033[95m";   // warning
    private static final String CYAN = "\033[96m";      // information

    //////////////////////////////////////////////////////////////////////
    // Analysis input
    private static class Analysis {
        private static final Pattern IS_PATH_HAS_FILE_UNIX = Pattern
                .compile("(.*/)*([^/]*)$");
        private static final Pattern IS_PATH_HAS_FILE_WIN = Pattern
                .compile("(.*[/\\\\])*([^/\\\\]*)$");
        private static final Pattern SPLIT_PATH = // / で終了するかどうか
                File.separatorChar == '/' ?
                        IS_PATH_HAS_FILE_UNIX :
                        IS_PATH_HAS_FILE_WIN;
        private static final Pattern WILD_CARD_ALL = Pattern
                .compile(".*");
        private final String virtualPath; //  / WildCard
        private final Path path; // ワイルドカードを除いた実際のパス
        private final Pattern regex; // ワイルドカード
        private final boolean alwaysTrue; // ワイルドカードが常に真かどうか

        private Analysis(String input, int DISPLAY_WIDTH) {
            String path = input.trim();
            String wild = "";
            Pattern regex = WILD_CARD_ALL;
            Matcher m = SPLIT_PATH.matcher(path);
            if (m.matches()) {
                String g2 = getValue(m.group(2).trim());
                if (hasWildcard(g2)) {
                    path = getValue(m.group(1)).trim();
                    wild = g2;
                    regex = mkWildcard(g2);
                }
            }
            if (path.isEmpty()) path = "./";
            this.path = Path.of(path);
            this.virtualPath = pack(path, DISPLAY_WIDTH) +
                    (wild.isEmpty() ? "" : NanoTools.color(NanoTools.BLUE, wild));
            this.regex = regex;
            alwaysTrue = WILD_CARD_ALL.equals(regex);
        }

        /**
         * hasWildcard - ワイルドカード指定かどうかを返す
         * - wildcard: * ? | endsWith .
         */
        private static boolean hasWildcard(String wild) {
            for (int i = 0; i < wild.length(); i++) {
                char c = wild.charAt(i);
                if (0 <= "*?|".indexOf(c)) return true;
            }
            return wild.endsWith(".");
        }

        /**
         * mkWildcard - 大小文字を区別せずにマッチする (Case-sensitive)
         * ワイルドカード (* ? .) 以外の記号は、正規表現として使用できる
         * e.g. /[fb]*.txt|*.log/
         *
         * @param wild wildcard
         */
        private static Pattern mkWildcard(String wild) {
            StringBuilder sb = new StringBuilder(128);
            if (wild.endsWith("."))
                wild = wild.substring(0, wild.length() - 1); // Delete the last .
            if (wild.isEmpty()) wild = "*"; // Default value
            for (int i = 0; i < wild.length(); i++) {
                char c = wild.charAt(i);
                switch (c) {
                    case '*' -> sb.append(".*");
                    case '?' -> sb.append('.');
                    case '.' -> sb.append("\\.");
                    default -> sb.append(c);
                }
            }
            return Pattern.compile(sb.toString(), Pattern.CASE_INSENSITIVE);
        }
    }

    /**
     * @param out              出力パス
     * @param hasCreatedHolder 出力フォルダを作成したかどうか
     */
    private record AnalysisOut(
            Path out, boolean hasCreatedHolder) {
    }

    /**
     * Async I/O - a-synchronize.
     */
    private class AsyncIO implements Runnable {
        private static final long ERROR_SLEEP_TIME = 1000; // 1 sec.
        private static final int RETRY = 7; // number of retries
        private final Progress pro;
        private final boolean copy; // copy/move
        private final int action; // AsyncAction details
        private final Path input; // 入力ファイル
        private final Path output; // 出力ファイル
        private final String name; // copy/move

        private AsyncIO(Progress pro, boolean copy, int action,
                        Path input, Path output) {
            this.pro = pro;
            this.copy = copy;
            this.action = action;
            this.input = input;
            this.output = output;
            this.name = copy ? "copy" : "move";
        }

        public void run() {
//            Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
            task();
            if (async.atom.decrementAndGet() <= 0) {
                // ここで submit されると atom>0 になる(正常)
                if (async.submitCompleted && async.atom.get() <= 0) {
                    synchronized (this) { // double check
//                        messageGREEN("Interrupt", async.atom.get());
                        myThread.interrupt(); // メインスレッドに割り込む
                    }
                }
            }
        }

        private void task() {
            Throwable ex = null;
            pro.addFile(output, getFileSize(input)); // File パスを登録

            for (int i = 0; i < RETRY; i++) { // retry
                try {
                    if (action != ACT_SAME_FILE) { // 出力先は同一ファイル？
                        Files.deleteIfExists(output);
                        if (copy) {
                            Files.copy(input, output, COPY_ATTRIBUTES, REPLACE_EXISTING);
                        } else {
                            if (async.ATOMIC_MOVE) { // アトミックモード
                                Files.move(input, output, ATOMIC_MOVE);
                            } else {
                                Files.move(input, output);
                            }
                        }
                    }
                    return;
                } catch (IOException e) { // AccessDeniedException
                    ex = e;
                    int err = async.error.incrementAndGet();
                    messageMAGENTA(name + '(' + err + ')', input);
                    messageMAGENTA(name, ex);
                }
                sleep();
            }
            messageRED(name, ex);
            ex.printStackTrace();
            throw new RuntimeException(ex);
        }

        // I/Oエラー発生時のリトライ
        private static synchronized void sleep() {
            try {
                Thread.sleep(AsyncIO.ERROR_SLEEP_TIME);
            } catch (InterruptedException e) {
                // do nothing
            }
        }
    }

    /**
     * Analysis control.
     */
    private static class AsyncCtrl {
        private final String fs1;
        private final String fs2;
        private final boolean ATOMIC_MOVE; // ATOMIC_MOVE モードが利用可能
        private final AtomicInteger atom = new AtomicInteger(); // number of threads
        private final AtomicInteger error = new AtomicInteger();
        private volatile boolean submitCompleted; // submitが全て完了した

        /**
         * @param input  入力パス
         * @param output 出力パス
         */
        AsyncCtrl(Path input, Path output) {
            fs1 = getFileStoreInfo(input);
            fs2 = getFileStoreInfo(output);
            ATOMIC_MOVE = fs1.equalsIgnoreCase(fs2);
        }

        AsyncCtrl() {
            fs1 = fs2 = null;
            ATOMIC_MOVE = false;
        }

        private void clear() {
            atom.set(0);
            error.set(0);
            submitCompleted = false;
        }

        @SuppressWarnings("unused")
        private void debug() {
            System.err.println(this);
        }

        @Override
        public String toString() {
            return color(MAGENTA, "FileStore.1: ") + fs1 + '\n' +
                    color(MAGENTA, "FileStore.2: ") + fs2;
        }

        /*
         *  Windows - ドライブレター | UNCパス
         */
        private static final Pattern EXTRACT_UNC_PATH = Pattern
                .compile("^(.:|//[^/]+/[^/]+/)");
        // //コンピュータ名/共有名/パス
        // 共有名の最後の文字を$にすると、共有名一覧に名前が表示されなくなる(隠し共有)

        /**
         * name ストレージ・プールまたはボリュームラベル
         * type 使用される形式を示したり、ファイル・ストアがローカルかリモートかを示す
         *
         * @return パス情報, NAME(ボリュームラベル), TYPE(NTFS)
         * Linux: rPool/USERDATA/[USER]_XXX0,zfs
         */
        private String getFileStoreInfo(Path path) {
            try {
                StringBuilder sb = new StringBuilder(64);
                Path abs = path.toAbsolutePath().normalize();
                String str = abs.toString()
                        .replace('\\', '/');
                Matcher m = EXTRACT_UNC_PATH.matcher(str + '/');
                if (m.find()) // Windows (ドライブレター、UNC)
                    sb.append(getValue(m.group(1))).append(",");
                FileStore fs = Files.getFileStore(abs);
                sb.append(fs.name()).append(','); // (ストレージ・プール等)
                sb.append(fs.type()); // TYPE(NTFS, zfs)
//            messageGREEN("FileStore " + sb, abs);
                return sb.toString();
            } catch (IOException e) {
                throw new RuntimeException(exception("getFileStoreInfo", e));
            }
        }
    }

    /**
     * AsyncAction - #isAtomicSameFile で生成
     *
     * @param action 状態コード (@see #isAtomicSameFile)
     * @param path   出力パス (null: 要件に該当しないとき、skip)
     */
    private record AsyncAction(int action, Path path) {
    }

    /**
     * Progress (Verbose)
     */
    private class Progress {
        private final boolean enable; // Progress, 有効/無効
        private final PrintStream printStream; // Progressの出力先
        private final boolean printX; // printXを使用するかどうか
        private final AtomicInteger folderNumber = new AtomicInteger();
        private final AtomicInteger fileNumber = new AtomicInteger();
        private final AtomicLong fileSize = new AtomicLong();
        private static final Path EMPTY_PATH = Path.of("/");
        private volatile Path path = EMPTY_PATH; // 初期値、番兵
        private volatile boolean hasPrinted; // printしたかどうか

        private Progress(boolean stdout, boolean enable) {
            this.printStream = stdout ? System.out : System.err;
            this.enable = enable;
            Redirect re = stdout ? __STDOUT : __STDERR; // printXの活性化
            if (re.redirect) this.printX = true; // リダイレクトのとき
            else this.printX = !enable; // ディセーブルのとき
            arr.clear(); // ※ 出力リスト
        }

        private void flush() {
            async.submitCompleted = true; // submitが全て完了した
            waitForInterrupt(); // Task終了割り込みを待ち合わせる
            if (enable) {
                if (path != EMPTY_PATH) { // 1件以上出力済み
                    elapsedTime = 0;
                    print(this.path); // 最後のファイル(時刻)を再表示
                }
                if (hasPrinted) {
                    printStream.println(); // 表示終了、new line
                }
                hasPrinted = false; // unprinted
            }
            if (arr.size() > 1) {
                arr.sort(null); // sort ArrayList (TreeSet より堅牢)
            }
        }

        /**
         * Task終了割り込みを待ち合わせる
         */
        private synchronized void waitForInterrupt() {
            while (async.atom.get() > 0) {
                try {
                    wait();
                } catch (InterruptedException e) {
                    //noinspection ResultOfMethodCallIgnored
                    Thread.interrupted();
//                    messageGREEN("Interrupted", async.atom.get());
                }
            }
        }

        private final AtomicInteger noBlockingPrint = new AtomicInteger();
        private static final long PRINT_ELAPSED = 200; // ms.
        private volatile long elapsedTime;

        // Non-blocking printing.
        private void print(Path path) {
            if (enable) {
                if (noBlockingPrint.getAndIncrement() <= 0) {
                    long now = System.currentTimeMillis();
                    if ((now - elapsedTime) > PRINT_ELAPSED) {
                        printStream.print(mkPrintData(path));
                        hasPrinted = true; // printed
                        elapsedTime = System.currentTimeMillis();
                    }
                }
                noBlockingPrint.decrementAndGet();
            }
        }

        // \033[nA  上にn移動
        private void debugPrint(String name, Object arg) {
            if (noBlockingPrint.getAndIncrement() <= 0) {
                String msg = "\033[1A" +
                        color(MAGENTA, name) + " " + arg + " \n";
                printStream.print(msg);
            }
            noBlockingPrint.decrementAndGet();
        }

        private void incrementFolder() {
            folderNumber.incrementAndGet();
        }

        private void decrementFolder() {
            folderNumber.decrementAndGet();
        }

        private void addFile(Path path) {
            addFile(path, getFileSize(path));
        }

        // File パスを登録 & プログレス表示
        private void addFile(Path path, long size) {
            fileNumber.incrementAndGet();
            fileSize.addAndGet(size);
            synchronized (this) {
                this.path = path;
                if (printX) arr.add(path); // ※ 出力リスト
            }
            print(path);
        }

        // \033[nK 	カーソルより後ろを消去
        // \r       先頭に移動
        // \033[nA  上にn移動
        private String mkPrintData(Path path) {
            String elapsed = elapsedTime(System.currentTimeMillis() - start);
            String progress = String.format("%s %,d(%,d): ",
                    elapsed, fileNumber.get(), folderNumber.get());
            return color(BLUE, progress) +
                    pack(path) + "\033[K\r";
        }
    }

    /**
     * Redirect - #redirect で生成
     *
     * @param redirect リダイレクト定義をした (ディフオルト定義を上書き)
     * @param name     表示名 (e.g. 2>foo)
     * @param rid      >, >>
     * @param file     ファイル名
     * @param isSTDERR 標準エラー？ (rno == 2)
     */
    private record Redirect(boolean redirect, String name,
                            String rid, String file, boolean isSTDERR) {
    }

    private static String getValue(String x) {
        return x == null ? "" : x;
    }
}
/*
@SuppressWarnings("SpellCheckingInspection")
    all : すべての警告の抑止
    boxing : ボックス/アンボックス・オペレーションに関連する警告の抑止
    cast : キャスト・オペレーションに関連する警告の抑止
    dep-ann : 使用すべきではない注釈に関連する警告の抑止
    deprecation : 非推奨に関連する警告の抑止
    fallthrough : 切り替えステートメントでの欠落している中断に関連する警告の抑止
    finally : 戻らない finally ブロックに関連する警告の抑止
    hiding : 変数を隠すローカルに関連する警告の抑止
    incomplete-switch : 切り替えステートメントでの欠落したエントリーに関連する警告の抑止 (列挙型の場合)
    javadoc : javadoc 警告に関連する警告の抑止
    nls : 非 NLS 文字列リテラルに関連する警告の抑止
    null : null を使用した分析に関連する警告の抑止
    rawtypes : raw 型の使用に関連する警告の抑止
    resource : Closeable 型のリソースの使用に関連する警告の抑止
    restriction : 推奨されないまたは禁止された参照の使用に関連する警告の抑止
    serial : シリアライズ可能クラスの欠落した serialVersionUID フィールドに関連する警告の抑止
    static-access : 不正な静的アクセスに関連する警告の抑止
    static-method : static として宣言される可能性のあるメソッドに関連する警告の抑止
    super : スーパーを呼び出さないメソッドのオーバーライドに関連する警告の抑止
    synthetic-access : 内部クラスからの最適化されていないアクセスに関連する警告の抑止
    sync-override : synchronized 宣言されたメソッドをオーバーライドする場合に同期がないことが原因で発生する警告を抑制
    unchecked : 未検査のオペレーションに関連する警告の抑止
    unqualified-field-access : 非修飾フィールド・アクセスに関連する警告の抑止
    unused : 未使用コードおよび不要コードに関連する警告の抑止
 */