/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.gen

import groovy.transform.CompileStatic
import plus.lex.Flags
import plus.lex.Node
import plus.lex.Symbols
import plus.lex.Term
import plus.util.NumHelper

/**
 * Code Generator.
 *
 * @author kunio himei.
 */
@CompileStatic
abstract class GenHelper extends GenBase {

    //* [変数名、配列の添え字].
    String[] varName(Node.YyVariable e) {
        String name = getAlias(e.name) // POSTIT エイリアスの設定☆
        if (functionMap.containsKey(name) && // ユーザ関数定義されている.
                !callMap.containsKey(name)) { // 後方参照のため関数として認識されていない.
            callMap.put(name, new Integer[]{1}) // ユーザ関数を呼び出した.
        }
        if (isNil(e.index)) {
            return new String[]{name, null}
        } else if (1 == e.index.length) {
            switch (e.index[0]) {
                case Term.REGEXP:
                    return new String[]{name, e.toString()} // /.../
                case Term.YyValue:
                    return new String[]{name, mkIndex(e.index)}
            }
        }
        return new String[]{name, mkIndex(e.index)}
    }

    //* 変数名 [配列の添え字].
    String mkVerName(Node.YyVariable e) {
        String[] vNam = varName(e) // name, index
        return mkVerNameX(vNam)
    }

    //* 変数名 [配列の添え字].
    static String mkVerNameX(String... vNam) { // FIXME 3.0.5
        String name = getAlias(vNam[0]) // POSTIT エイリアスの設定☆
        String index = vNam[1]
        return (isNil(index)) ? name : name + '[' + index + ']'
    }

    //* 配列の添え字を返す.
    String mkIndex(Object[] args) {
        Object[] arr = evalArgs(args)
        if (isNil(arr)) return ''
        if (1 == arr.length) return arr[0]
        return '_index' + mkString(arr, '(', ', ', ')')
    }

    //* 配列の添え字(数値タイプは、整数を返す)
    // 'Eratosthenes.awk'で、誤動作するため、'mkIndex'との統合は、不可！
    static String mkIndex2(Object a) {
        Object[] arr = { // CLOSURE で実装.
            if (a instanceof Object[]) a as Object[]
            else if (a instanceof Collection) (a as Collection<?>).toArray()
            else if (a instanceof Map) (a as Map<?, ?>).keySet().toArray()
            else new Object[]{a}
        }.call()
        if (isNil(arr)) return ''
        if (1 == arr.length) return arr[0]
        return '_index' + mkString(arr, '(', ', ', ')')
    }

    /** 関数呼出しパラメータの評価値配列を返す.
     * Interpreterは、副作用対応のため後方から評価.ここは、合わせる.
     */
    Object[] evalArgs(Object[] a) {
        Object[] arr = new Object[a.length]
        for (int i = a.length; --i >= 0;)
            arr[i] = eval(a[i])
        return arr
    }

    /** ユーザ定義関数,呼出しパラメータの設定.
     */
    Object[] callArgs(Node.Func fn, Object[] args, Object[] vals) {
        List<String> buf = []
        int argslen = args.length
        if ((1 <= argslen) && (0 == fn.argsLength)) {
            throw new IllegalArgumentException(fn.toString())
        }
        int idx = 0, k = 0
        while (argslen > k) {
            int typ = fn.parm[idx].nType
            if (Flags.isVarArgs(typ)) { // 可変長引数
                List<String> arr = []
                while (argslen > k) { // 実引数を全て変換
                    String w = ''
                    if (Flags.isReference(typ)) {
                        w = newRefCallArg(args[k]) // 参照変数
                    } else if (!Flags.isArray(typ)) {
                        w = vals[k] // 呼出元の単純変数
                    } else switch (args[k]) {
                        case Node.NAME:
                            Node.NAME x = args[k] as Node.NAME
                            w = x.name // 呼出元の配列
                    }
                    arr.add(w)
                    k += 1
                }
                if (fn.argsLength - 1 != idx) {
                    buf.add(mkString(arr.toArray(), '', ', ', ''))

                } else {
//                    int arrlen = arr.size()
//                    int j = 0
//                    while (arrlen > j) {
//                        buf.add(arr.get(j))
//                        j += 1
//                    }
                    for (x in arr)
                        buf.add(x)
                }
            } else if (!Flags.isArray(typ)) { // 単純変数
                buf.add(parseNumber(vals[k]))

            } else switch (args[k]) {
                case Node.NAME:
                    Node.NAME xx = args[k] as Node.NAME
                    buf.add(xx.name) // 呼出元の配列を参照
            }
            idx += 1
            k += 1
        }
        return buf.toArray()
    }

    // ------------------------------------------------------------------------
    // Helpers.
    // ------------------------------------------------------------------------
    //* 数値を解析して、文字列を返す
    static String parseNumber(Object e) {
        String x = e
        Number num = NumHelper.parseNumberFull(x)
        if (exists(num))
            return NumHelper.normalise(num).toString()
        return x
    }

    //* 未定義ならグローバル変数定義
    static void checkAndGlobalAlloc(String name) {
        if (!Symbols.findType(name)) { // 変数または関数呼び出しでない
            genGlobal(name, mkGlobalNil(name))
            Symbols.markGlobal(name, Flags.T26NIL) // Atomicとして記憶しない
        }
    }

    //* CLOSURE 関数か?
    static boolean isClosure(Node.Func e) {
        return Flags.isClosure(e.nType)
    }

    //* 変数属性を設定
    static Object merkType(String k, int nType, boolean isGlobalDef) {
        if (isGlobalDef) return Symbols.markGlobal(k, nType)
        return Symbols.markLocal(k, nType)
    }

    // ------------------------------------------------------------------------
    // 初期化.
    // ------------------------------------------------------------------------
    //* グローバル配列定義
    static void mkGlobalArray(String name) {
        chainMap.clear() //　連続代入連鎖マップをクリア.
        Symbols.markGlobal(name, Flags.T16ARRAY)
        genGlobal(name, "final def ${name} = new AtomicMap()") // ☆
    }

    //* 変数定義: 初期値 0
    static String mkGlobalNil(String name) {
        chainMap.clear() //　連続代入連鎖マップをクリア.
        "def ${name} = 0"
    }

    //* 初期化クラスを返す
    static String newClass(int type) {
        chainMap.clear() //　連続代入連鎖マップをクリア.
        if (Flags.isArray(type)) {
            finalValDef = 'final '
            return 'new AtomicMap()' // ☆
        }
        if (Flags.isNumber(type))
            return exists(finalValDef) ? 0 : 'new AtomicNumber()'
        if (Flags.isString(type))
            return "''"
        return NIL // REMIND グローバル変数の初期値 0.
    }

    //*
    String newRefCallArg(Object x) {
        switch (x) {
            case Node.YyVariable:
                Node.YyVariable xx = x as Node.YyVariable
                String[] vNam = varName(xx) // vnam, ix
                String name = mkVerNameX(vNam)
                return "{ def ${name} = new DelegateVar(${name}); ${name} }"
        }
        assert false, "newRefCallArg: ${x}"
        return null
    }
}