/*
 * Copyright (C) 2009 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.io;

import plus.spawn.system.Util;
import plus.spawn.system.UtilInterface;

import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * i/o pool.
 * <p>
 * The class to which this annotation is applied is immutable.
 *
 * @author kunio himei.
 */
public final class Command {

    /**
     * 正規表現にマッチしたグループ(3).
     */
    private static final int REGX_GROUP3 = 3;

    /**
     * リダイレクトを表す正規表現 (識別子まで).
     */
    private static final String REGX_REDIRECT_12 = "(\\d?)([<>]+)[ \t]*";

    /**
     * クオート以外の文字列を表す正規表現 (Simple String).
     */
    private static final String REGX_SIMPLE_STRING = "[^ \t'\"|\n]+";

    /**
     * パイプを表す正規表現 (pipe).
     */
    private static final Pattern RX_PIPE = Pattern.compile("^([\n|]&?)[ \t]*");

    /**
     * クオート文字列を表す正規表現 (Quoted String).
     */
    private static final Pattern RX_QUOTE = Pattern.compile("^(?:"
            + "\"([^\"\\\\]*(?:\\\\.[^\"\\\\]*)*)\"" // "([^\"\\]*(?:\\.[^\"\\]*)*)"
            + "|'([^']*)'" // '([^']*)'
            + ')');

    /**
     * リダイレクトを表す正規表現.
     */
    private static final Pattern RX_REDIRECT = Pattern.compile('^'
            + REGX_REDIRECT_12 + '(' + REGX_SIMPLE_STRING + ')');

    /**
     * リダイレクトを表す正規表現 (識別子まで).
     */
    private static final Pattern RX_REDIRECT_12 = Pattern
            .compile('^' + REGX_REDIRECT_12);

    /**
     * 単純な文字列を表す正規表現 (Simple String).
     */
    private static final Pattern RX_SIMPLE_STRING = Pattern
            .compile('^' + REGX_SIMPLE_STRING);

    /**
     * 空白かパイプが出現するまでの文字列の正規表現 (暫定).
     */
    private static final Pattern RX_SPACE_OF = Pattern.compile("^[^ \t<>|]+");

    /**
     * 空白文字列を表す正規表現 (Spaces).
     */
    private static final Pattern RX_SPACES = Pattern.compile("^[ \t]+");

    /**
     * Don't let anyone instantiate this class.
     */
    private Command() {
        super();
    }

    /**
     * split pipeline.
     *
     * @param command command line parameters
     * @return パイプライン配列
     */
    private static List<List<String>> parse(final String command) {
        final String arg = command.trim() + '|';
        final int len = arg.length();
        final List<List<String>> cmds = new ArrayList<>();
        final StringBuilder sb = new StringBuilder();
        for (int i = 0; len > i; ) {
            final List<String> args = new ArrayList<>();
            while (len > i) {
                final String x = arg.substring(i);
                final char c = arg.charAt(i);
                if (('|' == c) || ('\n' == c)) { // パイプ
                    final Matcher m = RX_PIPE.matcher(x);
                    if (!m.find()) {
                        throw new IllegalArgumentException("bug! " + m);
                    }
                    i += m.end();
                    if (0 < sb.length()) {
                        args.add(sb.toString()); // arguments
                        sb.setLength(0);
                    }
                    if (!args.isEmpty()) {
                        cmds.add(args);
                    }
                    break;

                } else if ((' ' == c) || ('\t' == c)) { // 空白
                    final Matcher m = RX_SPACES.matcher(x); // [ ]+
                    if (!m.find()) {
                        throw new IllegalArgumentException("bug! " + m);
                    }
                    i += m.end();
                    if (0 < sb.length()) {
                        args.add(sb.toString()); // arguments
                        sb.setLength(0);
                    }

                } else if (('\'' == c) || ('"' == c)) { // クオート文字列
                    Matcher m = RX_QUOTE.matcher(x); // "STRING\"" | 'STRING'
                    if (m.find()) {
                        sb.append((null != m.group(1)) ? m.group(1) : m
                                .group(2));
                    } else { // クオートが不完全な場合は、空白までの文字を取得する
                        m = RX_SPACE_OF.matcher(x); // ...SPACE
                        if (!m.find()) {
                            throw new IllegalArgumentException("bug! " + m);
                        }
                        sb.append(m.group());
                    }
                    i += m.end();

                } else {
                    Matcher m = RX_REDIRECT_12.matcher(x);
                    if (m.find()) { // リダイレクト
                        String descriptor = m.group(1);
                        final String rid = m.group(2); // リダイレクト識別子
                        if (descriptor.isEmpty()) { // ファイル記述子
                            descriptor = (('<' == rid.charAt(0)) ? "0" : "1");
                        }
                        sb.append(descriptor).append(rid);
                    } else { // "STRING [ \t'\"|\n]
                        m = RX_SIMPLE_STRING.matcher(x);
                        if (!m.find()) {
                            throw new IllegalArgumentException("bug! " + m);
                        }
                        sb.append(m.group());
                    }
                    i += m.end();
                }
            }
        }
        return cmds;
    }

    /**
     * spawn process Reader.
     *
     * @param command command line parameters
     * @return サブプロセスの出力ストリーム
     */
    static Reader processReader(final String command) throws IOException {
        String file = Io.STDIN;
        final List<List<String>> list = parse(command);
        for (final List<String> cmd : list) {
            for (int k = cmd.size(); 0 < --k; ) { // REMIND: 削除のため逆順に抽出
                final String x = cmd.get(k);
                final Matcher md = RX_REDIRECT.matcher(x);
                if (md.find()) {
                    final String rid = md.group(2);
                    assert (null != rid);
                    if ('<' == rid.charAt(0)) { // 標準入力をリダイレクト
                        final String name = md.group(REGX_GROUP3);
                        assert (null != name);
                        if ("&0".equals(name)) {
                            file = Io.STDIN; // 0<&0
                        } else {
                            file = name; // 0<name
                        }
                        System.err.println("`redirect " + x + " to `" + file
                                + '`');
                    }
                    cmd.remove(k); // 逆順のため、安全にリダイレクトを削除
                }
            }
        }
        final PipedWriter writer = new PipedWriter();
        final UtilInterface out = (UtilInterface) Util.spawn(list, writer);
        if (out.hasInput() || !Io.STDIN.equals(file)) {
            // コマンドの入力ファイルが指定されていないか、標準入力がリダイレクト指定されている場合
            IoHelper.copyLiLne(file, (Writer) out);
        }
        out.close();
        return writer.getReader();
    }

    /**
     * spawn process Writer.
     *
     * @param command command line parameters
     * @return サブプロセスの入力ストリーム
     */
    public static Writer processWriter(final String command) throws IOException {
        String file = Io.STDOUT;
        String rid = ">";
        final List<List<String>> list = parse(command);
        for (final List<String> cmd : list) {
            for (int k = cmd.size(); 0 < --k; ) { // REMIND: 削除のため逆順に抽出
                final String x = cmd.get(k);
                final Matcher md = RX_REDIRECT.matcher(x);
                if (md.find()) {
                    final String descriptor = md.group(1);
                    assert (null != descriptor);
                    if ('1' == descriptor.charAt(0)) { // 標準出力をリダイレクト
                        rid = md.group(2);
                        final String name = md.group(REGX_GROUP3);
                        assert (null != rid) && (null != name);
                        if ("&1".equals(name)) {
                            file = Io.STDOUT; // 1>&1
                        } else if ("&2".equals(name)) {
                            file = Io.STDERR; // 1>&2
                        } else {
                            file = name; // 1>name
                        }
                        System.err.println("`redirect " + x);
                    }
                    cmd.remove(k); // 逆順のため、安全にリダイレクトを削除
                }
            }
        }
        final Writer writer = Device.openOutput(rid, file);
        return Util.spawn(list, writer);
    }
}