/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.bench.target;

/**
 * Benchmark Targets - Strip, Javaのホワイトスペース規約に基づいた実装.
 * Implementation based on Java's whitespace convention.
 *
 * @author kunio himei.
 */
public class Strip {

    private static final char SPACE = '\u0020'; // 半角スペース
    private static final char FULL_SPACE = '\u3000'; // 全角スペース
    private static final char TAB = '\t';

    // left Strip Parts.
    private static int indexOfNonWhitespaceParts(String s, int end) {
        int start = 0;
        while (start < end) {
            int c = s.codePointAt(start);
            if (c != SPACE && c != FULL_SPACE && c != TAB &&
                    !Character.isWhitespace(c)) break;
            start += Character.charCount(c);
        }
        return start;
    }

    // right Strip Parts.
    private static int lastIndexOfNonWhitespaceParts(String s, int start, int end) {
        while (start < end) {
            int c = s.codePointBefore(end);
            if (c != SPACE && c != FULL_SPACE && c != TAB &&
                    !Character.isWhitespace(c)) break;
            end -= Character.charCount(c);
        }
        return end;
    }

    //////////////////////////////////////////////////////////////////////
    /*
     * 2. strip.Challenger
     */
    public static String strip02Challenger(String s) {
        int length = s.length();
        int start = indexOfNonWhitespaceParts(s, length);
        int end = lastIndexOfNonWhitespaceParts(s, start, length);
        return end < length || 0 < start ? s.substring(start, end) : s;
    }

    /*
     * 3. strip.Join (Integration)
     */
    public static String strip03Join(String s) {
        int length = s.length();
        int start = 0;
        while (start < length) {
            int c = s.codePointAt(start);
            if (c != SPACE && c != FULL_SPACE && c != TAB &&
                    !Character.isWhitespace(c)) break;
            start += Character.charCount(c);
        }
        int end = length;
        while (start < end) {
            int c = s.codePointBefore(end);
            if (c != SPACE && c != FULL_SPACE && c != TAB &&
                    !Character.isWhitespace(c)) break;
            end -= Character.charCount(c);
        }
        return end < length || 0 < start ? s.substring(start, end) : s;
    }
}
/*
    // ORIGINAL: String#strip for UTF-16.
	public static String strip(byte[] value) {
		int length = value.length >>> 1;
		int left = indexOfNonWhitespace(value);
		if (left == length) {
			return "";
		}
		int right = lastIndexOfNonWhitespace(value);
		boolean ifChanged = (left > 0) || (right < length);
		return ifChanged ? newString(value, left, right - left) : null;
	}

	public static int indexOfNonWhitespace(byte[] value) {
		int length = value.length >> 1;
		int left = 0;
		while (left < length) {
			int codepoint = codePointAt(value, left, length);
			if (codepoint != ' ' && codepoint != '\t' && !Character.isWhitespace(codepoint)) {
				break;
			}
			left += Character.charCount(codepoint);
		}
		return left;
	}

	public static int lastIndexOfNonWhitespace(byte[] value) {
		int length = value.length >>> 1;
		int right = length;
		while (0 < right) {
			int codepoint = codePointBefore(value, right);
			if (codepoint != ' ' && codepoint != '\t' && !Character.isWhitespace(codepoint)) {
				break;
			}
			right -= Character.charCount(codepoint);
		}
		return right;
	}
 */