/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.bench;

import org.awk4j.bench.target.Trim;

import static org.awk4j.bench.Bench.*;

/**
 * Benchmark Framework 3 - Functional mods.
 *
 * @author kunio himei.
 */
public class F03Future {

    // Superclass definition
    abstract static class Future<T, R> {
        abstract R apply(T t);
    }

    // ラッパー（Wrapper）
    static class rTrimRegex01Classic extends Future<String, String> {
        public String apply(String t) {
            return Trim.rTrimRegex01Classic(t);
        }
    }

    static class rTrimRegex02New extends Future<String, String> {
        public String apply(String t) {
            return Trim.rTrimRegex02New(t);
        }
    }

    static class trim03Original extends Future<String, String> {
        public String apply(String t) {
            return t.trim();
        }
    }

    static class trim04String extends Future<String, String> {
        public String apply(String t) {
            return Trim.trim04String(t);
        }
    }

    static class trim05CharSeq extends Future<CharSequence, String> {
        public String apply(CharSequence t) {
            return Trim.trim05CharSeq(t);
        }
    }

    static class rTrim06String extends Future<String, String> {
        public String apply(String t) {
            return Trim.rTrim06String(t);
        }
    }

    static class rTrim07Builder extends Future<StringBuilder, StringBuilder> {
        public StringBuilder apply(StringBuilder t) {
            return Trim.rTrim07Builder(t);
        }
    }

    // Instantiate a class
    private static final rTrimRegex01Classic rTrimRegex01Classic = new rTrimRegex01Classic();
    private static final rTrimRegex02New rTrimRegex02New = new rTrimRegex02New();
    private static final trim03Original trim03Original = new trim03Original();
    private static final trim04String trim04String = new trim04String();
    private static final trim05CharSeq trim05CharSeq = new trim05CharSeq();
    private static final rTrim06String rTrim06String = new rTrim06String();
    private static final rTrim07Builder rTrim07Builder = new rTrim07Builder();

    //////////////////////////////////////////////////////////////////////
    // driver- Generics (T: String と CharSequence を受け付ける)
    static <T> int driver00Generics(Future<T, String> fnc, String ans) {
        long nano = Bench.driver_loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            //noinspection unchecked
            rs = fnc.apply((T) BENCH_DATA); // ※
            nano -= System.nanoTime() - start;
            count++;
            if (isDriverComplete(++count)) break; // Driver is Complete
        } while (0 < nano);
        assert ans.equals(rs) : rs;
        return count;
    }

    // driver - String
    static int driver01String(Future<String, String> fnc, String ans) {
        long nano = Bench.driver_loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = fnc.apply(BENCH_DATA);
            nano -= System.nanoTime() - start;
            if (isDriverComplete(++count)) break; // Driver is Complete
        } while (0 < nano);
        assert ans.equals(rs) : rs;
        return count;
    }

    // driver - CharSequence
    @SuppressWarnings("SameParameterValue")
    static int driver02CharSequence(Future<CharSequence, String> fnc, String ans) {
        long nano = Bench.driver_loop_time;
        int count = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = fnc.apply(BENCH_DATA);
            nano -= System.nanoTime() - start;
            if (isDriverComplete(++count)) break; // Driver is Complete
        } while (0 < nano);
        assert ans.equals(rs) : rs;
        return count;
    }

    // driver - StringBuilder
    @SuppressWarnings("SameParameterValue")
    static int driver03Builder(Future<StringBuilder, StringBuilder> fnc, String ans) {
        long nano = Bench.driver_loop_time;
        int count = 0;
        do {
            sb.setLength(0);
            sb.append(BENCH_DATA);
            long start = System.nanoTime();
            fnc.apply(sb);
            nano -= System.nanoTime() - start;
            if (isDriverComplete(++count)) break; // Driver is Complete
        } while (0 < nano);
        assert ans.contentEquals(sb) : sb;
        return count;
    }

    private static final StringBuilder sb = new StringBuilder(BENCH_DATA.length());

    //////////////////////////////////////////////////////////////////////
    // クルー - crew
    @SuppressWarnings("SameParameterValue")
    static void crew(int[] ANSWER) {
        ANSWER[1] = driver01String(rTrimRegex01Classic, ANS_RIGHT);
        Bench.cool();
        ANSWER[2] = driver01String(rTrimRegex02New, ANS_RIGHT);
        Bench.cool();
        ANSWER[3] = driver01String(trim03Original, ANS_BOTH);
        Bench.cool();
        ANSWER[4] = driver01String(trim04String, ANS_BOTH);
        Bench.cool();
        ANSWER[5] = driver02CharSequence(trim05CharSeq, ANS_BOTH);
        Bench.cool();
        ANSWER[6] = driver01String(rTrim06String, ANS_RIGHT);
        Bench.cool();
        ANSWER[7] = driver03Builder(rTrim07Builder, ANS_RIGHT);
    }

    // クルー - crew for Generics
    @SuppressWarnings("SameParameterValue")
    static void GenericsCrew(int[] ANSWER) {
        ANSWER[1] = driver00Generics(rTrimRegex01Classic, ANS_RIGHT);
        Bench.cool();
        ANSWER[2] = driver00Generics(rTrimRegex02New, ANS_RIGHT);
        Bench.cool();
        ANSWER[3] = driver00Generics(trim03Original, ANS_BOTH);
        Bench.cool();
        ANSWER[4] = driver00Generics(trim04String, ANS_BOTH);
        Bench.cool();
        ANSWER[5] = driver00Generics(trim05CharSeq, ANS_BOTH);
        Bench.cool();
        ANSWER[6] = driver00Generics(rTrim06String, ANS_RIGHT);
        Bench.cool();
        ANSWER[7] = driver03Builder(rTrim07Builder, ANS_RIGHT);
    }
}
/*
    // Warmup Target - ウオームアップ ターゲット.
    //* F01Classic#warmupTarget を使用する.
    static void warmupTarget() {
        rTrimRegex01Classic.apply(BENCH_DATA); // #1
        rTrimRegex02New.apply(BENCH_DATA); // #2
        trim03Original.apply(BENCH_DATA); // #3 Original
        trim04String.apply(BENCH_DATA); // #4
        trim05CharSeq.apply(BENCH_DATA); // #5
        rTrim06String.apply(BENCH_DATA); // #6
        sb.setLength(0);
        sb.append(BENCH_DATA);
        rTrim07Builder.apply(sb); // #7
    }
*/