/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.bench;

import java.util.Date;

/**
 * Benchmark Var.2 - 単位時間に実行可能な回数を求める.
 * Find the number of times that can be executed in a unit time.
 * <p>
 * ・テストケース - test case
 * ・ベンチマーク - benchmark
 * ・クルー - crew
 * ・ドライバ - driver
 * ・ターゲット - target
 *
 * @author kunio himei.
 */
class BenchV2 {

    @SuppressWarnings("unused")
    private static final String PATTERN_01 = "^^);"; // LATIN1
    @SuppressWarnings("unused")
    private static final String PATTERN_02 = "※"; // UTF-16
    private static final String pattern = PATTERN_01;
    private static final String SPACE_256 = " ".repeat(256);
    private static final String test_data = " " + pattern + SPACE_256;
    private static final String ans_right = " " + pattern;
    private static final String ans_both = pattern;
    private static final long MILLIS = 1000_000; // 10^6 ns.
    private static final long MAIN_TIMES = 500 * MILLIS; // ※ 0.5 sec.

    private static long loop_time = MAIN_TIMES; // nano sec.
    private static int cooling_time; // Main.cool or Main.ice ms.
    private static boolean isPrint = false;
    private static int Original_repeat_count = 0; // 10k回を目処に、loop回数を設定

    /*
     * Warm up - ウオームアップ.
     */
    private static void warmup() {
        if (Main.prefetch > 0) {
            var sb = new StringBuilder(test_data);
            String rs = "";
            for (int i = 0; i < Main.prefetch; i++) {
                // オリジナルの対抗馬を最初に JIT化する.
                // JIT the original rival horse first.
                rs = Trim.trim04String(test_data); // JIT compilation 'charAt'
                Trim.trim05CharSeq(test_data);
                Trim.rTrim06String(test_data);
            }
            assert ans_both.equals(rs) : rs;

            for (int i = 0; i < Main.prefetch; i++) {
                // 文字列生成(共通)を JIT化する.
                // Convert character string generation (common) to JIT.
                String str = test_data.substring(1);
                CharSequence cs1 = sb.subSequence(1, sb.length());
                assert str.contentEquals(cs1) : cs1;
                sb.setLength(sb.length());
            }
            ice();
        }
        if (Main.loop > 0) { // 本番環境にストレスをかける
            // Stress the production environment
            benchV2(Main.loop * MILLIS); // Rehearsal exercise
        }
    }

    // クールダウン - cool down
    private static void cool() {
        sleep(cooling_time);
    }

    // 強制冷却 - Forced cooling
    private static void ice() {
        sleep(Main.ice);
    }

    private static void sleep(int millis) {
        try {
            if (millis > 0)
                Thread.sleep(millis);
        } catch (Throwable e) {
            throw new IllegalStateException(e);
        }
    }

    //////////////////////////////////////////////////////////////////////
    // ベンチマーク - benchmark
    static void bench() {
        long start = System.currentTimeMillis();
        if (Main.warmup) {
            cooling_time = Main.ice; // 強制冷却
            warmup();
        }
        cooling_time = Main.cool; // クールダウン
        isPrint = true;

        for (int i = 0; i < 12; i++) {
            benchV2(MAIN_TIMES);
            System.err.println();
        }
        double elapsed = (System.currentTimeMillis() - start) / 1000.;
        System.err.print("\tbenchTrim: " + elapsed + " sec. - " + Main.title + '\n' +
                "\t-priority: " + Main.priority + " " + new Date() + '\n' +
                "\t-warmup: " + Main.warmup + '\n' +
                "\t  1st loop -prefetch: " + Main.prefetch + " times\n" +
                "\t  2nd -loop: " + Main.loop + " ms. (" + Original_repeat_count + " times)\n" +
                "\t-cool(down): " + Main.cool + " ms.\n" +
                "\t-ice: " + Main.ice + " ms. (Forced cooling)\n");
    }

    //////////////////////////////////////////////////////////////////////
    // クルー - crew
    @SuppressWarnings("SameParameterValue")
    private static void benchV2(long time) {
        loop_time = time;
        ice();
        bench1Regex();
        cool();
        bench2Regex();
        cool();
        bench3TrimOriginal();
        cool();
        bench4TrimString();
        cool();
        bench5TrimCharSeq();
        cool();
        bench6rTrimString();
        cool();
        bench7rTrimBuilder();
    }

    private static void bench1Regex() {
        long nano = loop_time;
        int counter = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = Trim.rTrimRegex01Classic(test_data);
            nano -= System.nanoTime() - start;
            counter++;
        } while (0 < nano);
        assert ans_right.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t1. regex.Classic\t" + counter);
    }

    private static void bench2Regex() {
        long nano = loop_time;
        int counter = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = Trim.rTrimRegex02New(test_data);
            nano -= System.nanoTime() - start;
            counter++;
        } while (0 < nano);
        assert ans_right.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t2. regex.New\t" + counter);
    }

    private static void bench3TrimOriginal() {
        long nano = loop_time;
        int counter = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = test_data.trim();
            nano -= System.nanoTime() - start;
            counter++;
        } while (0 < nano);
        assert ans_both.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t3. Trim.Original\t" + counter);
        else
            Original_repeat_count = counter;
    }

    private static void bench4TrimString() {
        long nano = loop_time;
        int counter = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = Trim.trim04String(test_data);
            nano -= System.nanoTime() - start;
            counter++;
        } while (0 < nano);
        assert ans_both.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t4. Trim.String\t" + counter);
    }

    private static void bench5TrimCharSeq() {
        long nano = loop_time;
        int counter = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = Trim.trim05CharSeq(test_data);
            nano -= System.nanoTime() - start;
            counter++;
        } while (0 < nano);
        assert ans_both.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t5. Trim.CharSeq\t" + counter);
    }

    private static void bench6rTrimString() {
        long nano = loop_time;
        int counter = 0;
        String rs;
        do {
            long start = System.nanoTime();
            rs = Trim.rTrim06String(test_data);
            nano -= System.nanoTime() - start;
            counter++;
        } while (0 < nano);
        assert ans_right.equals(rs) : rs;
        if (isPrint)
            System.err.println("\t6. rTrim.String\t" + counter);
    }

    private static void bench7rTrimBuilder() {
        var sb = new StringBuilder(test_data.length());
        long nano = loop_time;
        int counter = 0;
        do {
            sb.setLength(0);
            sb.append(test_data);
            long start = System.nanoTime();
            Trim.rTrim07Builder(sb);
            nano -= System.nanoTime() - start;
            counter++;
        } while (0 < nano);
        assert ans_right.contentEquals(sb) : sb;
        if (isPrint)
            System.err.println("\t7. rTrim.Builder\t" + counter);
    }

    //////////////////////////////////////////////////////////////////////
    @SuppressWarnings("unused")
    private static void test() {
        int no = 0;
        String rs = Trim.trim05CharSeq("");
        System.err.println(++no + " rs:[" + rs + "] len:" + rs.length());
        rs = Trim.trim05CharSeq("XX");
        System.err.println(++no + " rs:[" + rs + "] len:" + rs.length());
        rs = Trim.trim05CharSeq(" X");
        System.err.println(++no + " rs:[" + rs + "] len:" + rs.length());
        rs = Trim.trim05CharSeq("X ");
        System.err.println(++no + " rs:[" + rs + "] len:" + rs.length());
        rs = Trim.trim05CharSeq(" X ");
        System.err.println(++no + " rs:[" + rs + "] len:" + rs.length());
    }
}