/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;

/**
 * noSpace - Remove unnecessary whitespace in html
 *
 * @author kunio himei.
 */
public class Main {

    private static final String USAGE =
            """
                    $(noSpace.jar) [-options] <output folder> <input file>
                    -d debug
                    -v Version 1.0.2""";

    private static final Parser parser = new Parser();
    private static BufferedReader reader;
    private static String in = "";
    static int lineNumber;
    static boolean isDebug;

    private static void error(String msg) {
        System.err.println(msg);
        System.exit(1);
    }

    static String getLine() {
        try {
            if (null != in) {
                in = reader.readLine();
                if (null != in) {
                    lineNumber += 1;
                    return Parser.rTrim(in); // Right trim
                }
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return null;
    }

    /*
     * Main.
     */
    public static void main(String[] args) {
        int argsLength = args.length;
        int ix = 0;
        for (; ix < argsLength; ix++) {
            String arg = args[ix];
            if ('-' != arg.charAt(0)) break;
            if (arg.contains("-v")) {
                System.err.println(USAGE);
                return;

            } else if (arg.startsWith("-d")) { // -d(debug)
                isDebug = true;
            }
        }
        try {
            if (ix >= argsLength)
                error("作業フォルダが指定されていません！");
            File workFolder = new File(args[ix++]); // 作業フォルダ
            if (!workFolder.exists() && !workFolder.mkdirs()) {
                error("作業フォルダの作成に失敗しました！ " + workFolder);
            }
            if (ix >= argsLength)
                error("入力ファイルが指定されていません！");
            File targetFile = new File(args[ix]); // (入)出力ファイル
            if (!targetFile.exists()) {
                error("入力ファイルは存在しません！ " + targetFile);
            }
            String name = targetFile.getName(); // 対象ファイルの名前
            File inputFile = new File(workFolder, name);
            // 入出力ファイルは同一でないことをチェック
            File inA = inputFile.getAbsoluteFile().getCanonicalFile();
            File outA = targetFile.getAbsoluteFile().getCanonicalFile();
            if (inA.equals(outA)) {
                error("入出力ファイルは同じです！");
            }
            if (inputFile.exists() && !inputFile.delete()) {
                error("作業ファイルの削除に失敗しました！ " + inputFile);
            }
            // ターゲットファイルを作業フォルダに移動
            Path pTarget = targetFile.toPath();
            Path pInput = inputFile.toPath();
            Files.move(pTarget, pInput);

            System.out.println("Input File: " + targetFile + " → " + inputFile);
            InputStream inputStream =
                    new FileInputStream(inputFile);
            reader = new BufferedReader(
                    new InputStreamReader(inputStream,
                            StandardCharsets.UTF_8), 1024);

            System.out.println("Output File: " + targetFile);
            FileWriter writer =
                    new FileWriter(targetFile, StandardCharsets.UTF_8);

            File debugFile = new File(workFolder, "~debug.html");
            System.out.println("Debug File: " + debugFile);
            FileWriter debugger =
                    new FileWriter(debugFile, StandardCharsets.UTF_8);

            while (true) { // Main loop
                String input = getLine();
                if (null == input) break;
                String output = parser.parse(input);
                writer.write(output);
                debugger.write(parser.debug);
            }
            debugger.write("<!-- " + Parser.C_DELETE +
                    " Removed " + parser.cutSpace + " spaces -->");
            System.out.println("Removed " + parser.cutSpace + " spaces, " +
                    lineNumber + " lines");
            writer.flush();
            debugger.flush();
            writer.close();
            debugger.close();
            reader.close();
        } catch (Throwable e) {
            e.printStackTrace();
        } finally {
            System.out.flush();
            System.err.flush();
        }
    }
}