/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.space;

import org.jetbrains.annotations.NotNull;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * cutSpace - Tools.
 *
 * @author kunio himei.
 */
class Tools {

    static int RSTART; // Match start
    static int RLENGTH; // Match length

    /**
     * match(r,s) - AWK specifications.
     *
     * @param group number 0, 1.. m.groupCount()
     */
    static boolean match(@NotNull Pattern r, CharSequence s, int group) {
        Matcher m = r.matcher(s);
        if (m.find()) {
            RSTART = m.start(group);
            RLENGTH = m.end(group) - RSTART;
            return true;
        }
        return false;
    }

    /**
     * Right trim (String).
     */
    private static final Pattern RIGHT_TRIM = Pattern.compile("[\\s]+$");

    @NotNull
    static String rTrim(@NotNull String s) {
        Matcher m = RIGHT_TRIM.matcher(s);
        if (m.find())
            return s.substring(0, m.start());
        return s;
    }

    /**
     * Right trim (StringBuilder).
     */
    @NotNull
    static StringBuilder rTrim(@NotNull StringBuilder sb) {
        int i = sb.length();
        while (0 <= --i) {
            if (!isSpace(sb.charAt(i))) break;
        }
        sb.setLength(i + 1);
        return sb;
    }

    /**
     * Right twin trim (Source, Template).
     */
    static void rTrim(@NotNull StringBuilder sb, @NotNull StringBuilder sx) {
        int len = rTrim(sx).length(); // This is the first for DEL measures.
        sb.setLength(len);
    }

    /**
     * Left trim (String).
     */
    private static final Pattern LEFT_TRIM = Pattern.compile("^[\\s]+");


    @NotNull
    static String lTrim(@NotNull String s) {
        Matcher m = LEFT_TRIM.matcher(s);
        if (m.find())
            return s.substring(m.end());
        return s;
    }

    /**
     * Backward match (char).
     */
    static boolean endsWith(@NotNull CharSequence cs, char c) {
        int ix = cs.length() - 1;
        return 0 <= ix && c == cs.charAt(ix);
    }

    /**
     * charAt - Gets the character of the specified index.
     *
     * @return a Blank if White Space.
     * @throws IndexOutOfBoundsException if index mistake.
     */
    static char charAt(@NotNull CharSequence cs, int ix) {
        if (0 <= ix && ix < cs.length()) {
            char c = cs.charAt(ix);
            return isSpace(c) ? ' ' : c;
        }
        throw new IndexOutOfBoundsException(ix + "/" + cs.length() + ": " + cs);
    }

    /**
     * charAtLast - Get the character of the last index.
     *
     * @return ${NUL} Source is empty.
     */
    static char charAtLast(@NotNull CharSequence cs) {
        int ix = cs.length() - 1;
        char c = 0 <= ix ?
                cs.charAt(ix) : Template.NUL;
        return isSpace(c) ? ' ' : c;
    }

    // To White Space
    static char toWhiteSpace(char c) {
        return isSpace(c) ? ' ' : c;
    }

    // Is White space.
    private static final String WHITE_SPACES = " \t\n\r" + Template.RS;

    static boolean isSpace(char c) {
        return 0 <= WHITE_SPACES.indexOf(c);
    }
}