/**
 * Copyright (C) 2023 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
use std::fs;
use std::io;
use std::path::Path;
// use std::path::{Path, PathBuf};

use crate::iomod;
use crate::thmod;

fn visit_dir<P: AsRef<Path>>(path: P, opath: &Path, is_copy: bool) -> io::Result<()> {
    for entry in fs::read_dir(path)? {
        let entry = entry?;
        let _name: String = iomod::get_filename(&entry.path());
        let _opath: &Path = &opath.join(_name); // output file
        if entry.file_type()?.is_dir() {
            // println!("+{:?}\t{:?}", entry.path(), _opath);
            iomod::mkdir(_opath); // Create folder
            visit_dir(entry.path(), _opath, is_copy)?;
        } else {
            // println!(".{:?}\t{:?}", entry.path(), _opath);
            make_dd(&entry.path(), _opath, is_copy);
        }
    }
    Ok(())
}

pub fn get_fils(input: String, output: String, is_copy: bool) -> io::Result<()> {
    let ipath: &Path = Path::new(&input);
    let opath: &Path = Path::new(&output);
    visit_dir(ipath, opath, is_copy)?;
    Ok(())
}

/*
fn visit_remove<P: AsRef<Path>>(path: P) -> io::Result<()> {
    for entry in fs::read_dir(path)? {
        let entry = entry?;
        if entry.file_type()?.is_dir() {
            // println!("+{:?}\t{:?}", entry.path(), _opath);
            visit_remove(entry.path())?;
        } else {
            // println!(".{:?}\t{:?}", entry.path(), _opath);
        }
    }
    Ok(())
}
 */

// remove dir all
pub fn remove_dir(input: String) -> io::Result<()> {
    let _input: &Path = Path::new(&input);
    iomod::remove_dir_all(_input);
    Ok(())
}

// https://runebook.dev/ja/docs/rust/std/fs/struct.metadata
fn make_dd(_input: &Path, _output: &Path, is_copy: bool) {
    let input: String = iomod::path_to_string(_input);
    let output: String = iomod::path_to_string(_output);
    let action: i32 = judgment(&_input, _output);
    let dd = DD {
        input,   // input file
        output,  // output file
        is_copy, // copy, move
        action,  // DO, SKIP
    };
    thmod::put(dd);
}

// judgment equals
fn judgment(input: &Path, output: &Path) -> i32 {
    if !output.is_file() {
        return DO; // 出力ファイルが存在しない
    }
    let ilen = iomod::get_meta_len(input);
    let olen = iomod::get_meta_len(output);
    if ilen != olen {
        return DO; // 長さが異なる
    }
    let itime = iomod::get_meta_modified(input);
    let otime = iomod::get_meta_modified(output);
    if let Ok(epoch) = itime.duration_since(otime) {
        if epoch.as_secs() == 0 && epoch.as_millis() == 0 {
            // println!("same = {}.{:03}", epoch.as_secs(), epoch.as_millis());
            return SKIP; // 日時が等しい
        }
    } else {
        return SKIP; // 出力側の日時が新しい
    }
    DO
}

// Action
pub const DO: i32 = 1;
pub const SKIP: i32 = 2;

// Daemon descriptor - 構造体、クローン可能
#[derive(Debug, Clone)]
pub struct DD {
    pub input: String,  // input file
    pub output: String, // output file
    pub is_copy: bool,  // copy, move
    pub action: i32,    // DO, SKIP
}
impl DD {
    // pub fn _get_input(&self) -> String {
    //     self.input.clone()
    // }
}
