/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.util;

import plus.runtime.BuiltInVar;
import plus.runtime.RegExp;
import plus.runtime.RunHelper;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * [%helper%] Compare NumHelper.
 * <p>
 * The class to which this annotation is applied is immutable.
 *
 * @author kunio himei.
 */
public final class Compare {

    /**
     * Don't let anyone instantiate this class.
     */
    private Compare() {
        super();
    }

    /**
     * 配列キーの存在確認.
     */
    public static boolean compareIn(Object array, Object index) {
        if (array instanceof Map<?, ?>) { // Map
            String key = RunHelper.toString(null, index);
            return ((Map<?, ?>) array).containsKey(key);
        }
        int ix = NumHelper.intValue(index);
        if (0 <= ix) {
            if (array instanceof List<?>) { // List
                if (array instanceof CopyOnWriteArrayList) // REMIND 個別対応☆
                    return null != ((List<?>) array).get(ix);
                return (ix < ((List<?>) array).size());
            }
            if (array instanceof Object[]) // []
                return (ix < ((Object[]) array).length);

            // NOTE Groovy 可変長引数を受け取るためのプリミティブ配列.
            if (array instanceof int[])
                return (ix < ((int[]) array).length);

            if (array instanceof double[])
                return (ix < ((double[]) array).length);

            if (array instanceof float[])
                return (ix < ((float[]) array).length);

            if (array instanceof long[])
                return (ix < ((long[]) array).length);

            if (array instanceof char[])
                return (ix < ((char[]) array).length);

            if (array instanceof short[])
                return (ix < ((short[]) array).length);

            if (array instanceof byte[])
                return (ix < ((byte[]) array).length);

            if (array instanceof boolean[])
                return (ix < ((boolean[]) array).length);
        }
        throw new IllegalArgumentException(array.getClass() + " " + array);
    }

    /**
     * Compares the two specified {@code Object} values.
     * <p>
     * 片方が文字列定数なら文字列、 以外なら数値比較.
     * <table border="1">
     * <tr>
     * <th></th>
     * <th>STRING</th>
     * <th>NUMERIC</th>
     * <th>STRNUM</th>
     * </tr>
     * <tr>
     * <th>STRING</th>
     * <td>STRING</td>
     * <td>STRING</td>
     * <td>STRING</td>
     * </tr>
     * <tr>
     * <th>NUMERIC</th>
     * <td>STRING</td>
     * <td>NUMERIC</td>
     * <td>NUMERIC</td>
     * </tr>
     * <tr>
     * <th>STRNUM</th>
     * <td>STRING</td>
     * <td>NUMERIC</td>
     * <td>NUMERIC</td>
     * </tr>
     * </table>
     *              '&gt;=' | '&gt;' | '!=' | '===' | '!==')
     * <p>
     * (USE; CONVFMT, IGNORECASE)
     */
    public static boolean compareTo(String op,
                                    final Object left, final Object right) {
        int oplen = op.length();
        switch (op.charAt(oplen - 1)) {
            case '=':
                if (3 == oplen) { // 3byte演算子, '===' or '!=='
                    //noinspection ObjectEquality
                    boolean bo = (left == right); // NOPMD: == this surely will never happen
                    return ('!' == op.charAt(0)) != bo;
                }
                // Fallthrough //
            case '<':
            case '>':
                if (left instanceof Number) {
                    if (right instanceof Number) {
                        return compareToImpl(op, Double.compare(
                                ((Number) left).doubleValue(),
                                ((Number) right).doubleValue()));
                    } else if (null == right) {
                        return compareToImpl(op,
                                Double.compare(((Number) left).doubleValue(), 0));
                    }
                    // continue
                } else if ((null == left) && (right instanceof Number)) {
                    return compareToImpl(op,
                            Double.compare(0, ((Number) right).doubleValue()));
                }
                break; // String Compare

            case '~': // 正規表現適合  ~ or !~
                String s1 = RunHelper.toString(null, left);
                String s2 = RunHelper.toString(null, right);
                boolean bo = RegExp.compile(s2).matcher(s1).find();
                return ('!' == op.charAt(0)) != bo;

            default:
                if ("in".equals(op)) {
                    return compareIn(right, left);

                } else if ("is".equals(op) || "instanceof".equals(op)) {
                    Class<?> clazz = (null == right) ? Object.class
                            : ((right instanceof Class<?>) ? (Class<?>) right
                            : right.getClass());
                    return clazz.isInstance(left);
                }
                // else; String Compare で演算子エラーにする
        }
        // String Compare
        CharSequence convfmt = BuiltInVar.CONVFMT.toString(); // 数値から文字列への変換書式 (既定値は %.6g)
        String s1 = RunHelper.toString(convfmt, left);
        String s2 = RunHelper.toString(convfmt, right);
        int ignoreCase = BuiltInVar.IGNORECASE.intValue(); // 英大小文字の区別指示子
        return compareToImpl(op, (0 == ignoreCase) ? //
                s1.compareTo(s2) : s1.compareToIgnoreCase(s2));
    }

    /**
     * Compare the compare operator.
     * '!=')
     */
    private static boolean compareToImpl(String op, int x) {
        switch (op.charAt(0)) {
            case '<':
                //noinspection ConstantOnRightSideOfComparison
                return (1 == op.length()) ? (x < 0) : (x <= 0); // <. <=
            case '>':
                //noinspection ConstantOnRightSideOfComparison
                return (1 == op.length()) ? (x > 0) : (x >= 0); // >. >=
            case '=':
                return (0 == x); // ==
            case '!':
                return (0 != x); // !=
            default:
                // break;
        }
        throw new IllegalArgumentException(op + ' ' + x);
    }

    /**
     * Pattern to Pattern Matcher.
     */
    public static boolean compareToPattern(Map<String, Object> p2p,
                                           String id, boolean start, boolean end) {
        if (p2p.containsKey(id)) { // 以前は開始状態で
            if (end) { // 今は終了条件を満たしておれば
                p2p.remove(id); // フラグを削除
            }
            return true; // 範囲内
        }
        if (start) { // 以前は開始状態でなく今は開始条件を満たしており
            if (!end) { // 終了条件でなければ
                p2p.put(id, Boolean.TRUE); // 開始を記憶
            }
            return true; // 範囲内
        }
        return false; // 範囲外
    }

    /**
     * C言語仕様の論理値を考慮した論理値を返す.
     */
    public static boolean boolValue(final Object x) {
        if (x instanceof Boolean) {
            return (Boolean) x;
        } else if (x instanceof Number) {
            return (0 != ((Number) x).longValue()); // 0以外なら真
        } else if (x instanceof CharSequence) {
            return 0 != (long) NumHelper.doubleValue(x); // 0以外なら真
        }
        return false;
    }
}