/*
 * Copyright (C) 2009 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.io;

import java.io.*;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantLock;

/**
 * [%builtIn%] AWK 組み込み入出力関数の実装.
 * <p>
 * The class to which this annotation is applied is immutable.
 *
 * @author kunio himei.
 */
public class Io {

    /**
     * [%special file%] 標準エラー出力 (ファイル記述子 2).
     */
    public static final String STDERR = "/dev/stderr";
    /**
     * [%special file%] 標準入力 (ファイル記述子 0).
     */
    public static final String STDIN = "/dev/stdin";
    /**
     * [%special file%] NULL (ビットバケツ bit bucket).
     */
    public static final String STDNUL = "/dev/null";
    /**
     * [%special file%] 標準出力 (ファイル記述子 1).
     */
    public static final String STDOUT = "/dev/stdout";
    /**
     * 復帰値: INTEGER 0 (OK).
     */
    private static final int CRET_ZERO = 0;
    /**
     * INET SOCKET (スレッドローカル変数).
     */
    private static final ThreadLocal<Map<String, ReadWriteable>>
            INET_SOCKETS = new ThreadLocal<>() {
        /**
         * 初期値.
         */
        @Override
        protected Map<String, ReadWriteable> initialValue() {
            return new ConcurrentHashMap<>();
        }
    };
    /**
     * system device.
     */
    private static final Map<String, String> SPECIAL_FILES = new HashMap<>();

    static {
        SPECIAL_FILES.put("-", "");
        SPECIAL_FILES.put(STDNUL, "");
        SPECIAL_FILES.put(STDERR, "");
        SPECIAL_FILES.put(STDIN, "");
        SPECIAL_FILES.put(STDOUT, "");
    }

    /**
     * INET CONNECTION pool.
     */
    private final ConcurrentHashMap<String, Connectable> inetConnects = new ConcurrentHashMap<>();
    /**
     * input pool.
     */
    private final Map<String, TextReader> inputPool = new ConcurrentHashMap<>();
    /**
     * Reentrant Lock Object.
     */
    private final ReentrantLock SYNC = new ReentrantLock(); // SYNC.
    /**
     * output pool.
     */
    private final Map<String, Writer> outputPool = new ConcurrentHashMap<>();

    /**
     * PrintStream, PrintWriter 以外のストリームを閉じる.
     *
     * @param io ストリーム | リーダ | ライタ
     */
    public static void close(Closeable... io) throws IOException {
        IOException error = null;
        for (Closeable x : io) {
            try {
                boolean isSystemIO = (x instanceof PrintStream)
                        || (x instanceof PrintWriter);
                if (!isSystemIO) {
                    x.close();
                }
            } catch (IOException e) {
                if (null == error) {
                    error = e;
                } // else ひき逃げ
            }
        }
        if (null != error) {
            throw error; // 最初のエラーを投げる
        }
    }

    /**
     * [%I/O関数%] close 関数の実装.
     *
     * @param io ストリーム | リーダ | ライタ
     * @return 0: 正常
     */
    @SuppressWarnings("UnusedReturnValue")
    public int close(String... io) throws IOException {
        IOException error = null;
        for (String x : io) {
            try {
                if (this.outputPool.containsKey(x)) {
                    close(this.outputPool.get(x));
                    if (!SPECIAL_FILES.containsKey(x)) {
                        this.outputPool.remove(x);
                    }
                } else if (this.inputPool.containsKey(x)) {
                    close(this.inputPool.get(x));
                    if (!SPECIAL_FILES.containsKey(x)) {
                        this.inputPool.remove(x);
                    }
                } else if (INET_SOCKETS.get().containsKey(x)) {
                    close(INET_SOCKETS.get().get(x));
                    INET_SOCKETS.get().remove(x);
                    this.outputPool.remove(x);
                    this.inputPool.remove(x);
                }
            } catch (IOException e) {
                if (null == error) {
                    error = e;
                } // else ひき逃げ
            }
        }
        if (null != error) {
            throw error; // 最初のエラーを投げる
        }
        return CRET_ZERO; // OK.
    }

    /**
     * 開いている全てのストリームを閉じる.
     */
    public void closeAll() throws IOException {
        List<String> list = new ArrayList<>(
                this.inputPool.keySet());
        list.addAll(this.outputPool.keySet());
        String[] a = list.toArray(new String[0]);
        close(a);
        this.inputPool.clear(); // Android クリーンアップ.
        this.outputPool.clear();
        this.inetConnects.clear();
        INET_SOCKETS.get().clear();
    }

    /**
     * [%I/O関数%] flush 関数の実装.
     *
     * @param files ファイル ("": 全ての出力ストリーム)
     * @return 0: 正常
     */
    @SuppressWarnings("UnusedReturnValue")
    public int fflush(Object[] files) throws IOException {
        // System.err.println("fflush: " + Arrays.toString(files));
        if (0 == files.length) {
            System.out.flush(); // 標準出力をフラッシュ
            System.err.flush(); // 標準エラー出力をフラッシュ

        } else if ((1 == files.length) &&
                files[0].toString().isEmpty()) {
            // 空文字列は全ての出力ストリーム
            for (Writer out : this.outputPool.values()) {
                if (null != out) {
                    out.flush();
                }
            }
        } else {
            for (Object o : files) {
                Writer out = this.outputPool.get(String.valueOf(o));
                if (null != out) {
                    out.flush();
                }
            }
        }
        return CRET_ZERO; // OK.
    }

    /**
     * 指定されたストリームを開く.
     *
     * @param rid  リダイレクト指定子
     * @param file ファイル名
     * @return リーダ
     */
    public TextReader getReader(String rid, String file)
            throws IOException {
        TextReader in = this.inputPool.get(file);

        if (IoConstants.START_WITH_INET_URL.matcher(file).find()) { // INET
            ReadWriteable handle = openInet(file);
            in = new TextReader(handle.getInputStream());
            this.inputPool.put(file, in);
        }

        if (null == in) {
            if (STDIN.equals(file)) {
                in = new TextReader(System.in); // 標準入力 (ファイル記述子 0)

            } else if (file.isEmpty() || "-".equals(file)) {
                in = getReader(rid, STDIN); // recursive call

            } else if ("|".equals(rid)) {
                in = new TextReader(Command.processReader(file)); // PROCESS

            } else {
                in = new TextReader(Device.openInput(file)); // FILE
            }
            this.inputPool.put(file, in);
        }
        return in;
    }

    /**
     * 指定されたストリームを開く.
     *
     * @param rid  リダイレクト指定子
     * @param file ファイル名
     * @return ライタ
     */
    Writer getWriter(String rid, String file) throws IOException {
        Writer out = this.outputPool.get(file);
        if (IoConstants.START_WITH_INET_URL.matcher(file).find()) { // INET
            // new Exceptions("'" + rid + "' " + file).printStackTrace();
            ReadWriteable handle = openInet(file);
            String charset = Device.getCharset(file);
            out = new StreamWriter(handle.getOutputStream(), charset);
            this.outputPool.put(file, out);
        }

        if (null == out) {
            if (STDOUT.equals(file)) {
                out = new StreamWriter(System.out);  // OS. dependent encoding.

            } else if (STDERR.equals(file)) {
                out = new StreamWriter(System.err);  // OS. dependent encoding.

            } else if (file.isEmpty() || "-".equals(file)) {
                out = getWriter("", STDOUT); // recursive call

            } else if ("|".equals(rid)) {
                out = Command.processWriter(file); // PROCESS

            } else {
                out = Device.openOutput(rid, file); // FILE
            }
            this.outputPool.put(file, out);
        }
        return out;
    }

    /**
     * 指定されたストリームは閉じてているか?.
     *
     * @param io ストリーム | リーダ | ライタ
     */
    public boolean noStream(String io) {
        return !(this.inputPool.containsKey(io)
                || this.outputPool.containsKey(io));
    }

    /**
     * inet を開く.
     *
     * @param file ファイル名
     * @return リーダ|ライタ
     * @throws InterruptedIOException 待機中に割り込みが発生した
     */
    private ReadWriteable openInet(String file) throws IOException {
        ReadWriteable socs;
        Connectable conn, conx;
        this.SYNC.lock(); // SYNC. openInet.
        try {
            socs = INET_SOCKETS.get().get(file);
            if (null != socs) return socs;
            conn = this.inetConnects.get(file);
            if (null == conn) {
                conn = InetHelper.open(file); // 新しい接続を取得し、
                if (conn instanceof InetServerSocket) { // TCP/IP サーバは
                    conx = this.inetConnects.putIfAbsent(file, conn);
                    if (null != conx) conn = conx; // REMIND: singleton として記憶
                }
            }
        } finally {
            this.SYNC.unlock();
        }
        socs = conn.connect(); // ここで、接続完了までブロックする
        INET_SOCKETS.get().put(file, socs);
        return socs;
    }

    /**
     * [%I/O関数%] print 関数の実装.
     *
     * @param rid   リダイレクト指定子
     * @param file  ファイル名
     * @param value 出力文字列
     */
    public int print(String rid, String file, String value) throws IOException {
        Writer out = getWriter(rid, file);
        try {
            out.write(value);
            out.flush();
        } catch (IOException e) {
            // NOTE 既存の接続はリモート ホストに強制的に切断された.
            // 'DayTime のように一方的に出力するプロトコルの場合に、
            // クライアント終了時にこの状態になる.
            System.err.println("plus.Io.print: " + e.getMessage());
            return -1;
        }
        return CRET_ZERO; // OK.
    }
}