/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.gen

import groovy.transform.CompileStatic
import plus.lex.*

/**
 * AWK~plus - Code Generator.
 *
 * @author kunio himei.
 */
@CompileStatic
class Gen extends GenImpl {

    private static String outputHolder = './' // スクリプトの出力ディレクトリ.

    Gen(Parser.SyntaxTree tree) {
        sTree = tree
        functionMap = tree.func // ユーザ定義関数 (エイリアス).
    }

    String generate() {
        String scriptPath = sTree.scriptName // 拡張子を削除
                .replaceFirst(/[.][\w$]*$/, '')
        String scriptName = scriptPath.replaceFirst(/^.*[\\/]/, '')
        String scriptFile = scriptName + '.groovy'
        callMap = sTree.call
        if (exists(sTree.comment)) { // 注釈
            globalBuf.add(mkString(sTree.comment, '/**', '\n ', '\n */'))
        }
        if (exists(sTree.packageName))
            genTop('package ' + sTree.packageName)
        for (x in sTree.userImport)
            genTop('import ' + x)
        for (x in sTree.annotation) // Class Annotation 生成
            genTop(x)
        boolean hasMain = exists(sTree.main) || exists(sTree.end)
        genTop('class ' + scriptName + ' extends RunTime {')
        genTop('static void main(String[] args) {')
        genTop("    ${scriptName} THIS = new ${scriptName}()")
        genTop("    THIS._run(args, THIS, ${hasMain})")
        genTop('}')

        Symbols.clearGlobal()
        isGlobalDef = true // global変数定義開始
        yyIndent -= 1
        //* global変数の初期化 (一般変数は明示的に代入するため生成しない)
        for (x in sTree.globalValues.keySet()) {
            int typ = sTree.globalValues.get(x)
            if (Flags.isArray(typ)) mkGlobalArray(x)  // delete T など明に宣言していない場合.
        }
        if (exists(sTree.global))
            statement(sTree.global) // 定義順に処理する.
        yyIndent += 1
        isGlobalDef = false

        //* アクション生成
        action('BEGIN', sTree.annotateBEGIN, sTree.begin)
        action('MAIN', sTree.annotateMAIN, sTree.main)
        action('END', sTree.annotateEND, sTree.end)
        //* 関数生成
        for (x in new TreeSet(callMap.keySet())) {
            Node.Func fn = functionMap.get(x)
            if (exists(fn) && !excludeMap.containsKey(x)) {
                functionDecl(fn)
            }
        }
        gen('}')
        //* コード生成 - UTF-8
        File file = new File(outputHolder)
        if (!file.exists()) file.mkdirs()
        file = new File(file, scriptFile)
        PrintWriter src = new PrintWriter(file, 'UTF-8') // NOTE codeのコード.
        src.print(mkString(globalBuf.toArray(), '', '\n', '\n'))
        src.print(mkString(codeBuf.toArray(), '', '\n', '\n'))
        src.close()
        return file.toString()
    }

    // ------------------------------------------------------------------------
    // eval
    // ------------------------------------------------------------------------
    @Override
    Object eval(Object e) {
        while (true) {
            //noinspection GroovyFallthrough
            switch (e) {
                case List:
                    e = (e as List).toArray() // List() -> Array[]
            //* Fallthrough //
                case Object[]:
                    Object[] arr = e as Object[]
                    if (1 == arr.length) { // 最後の要素.
                        e = arr[0]
                    } else if (1 < arr.length) {
                        eval(arr[0]) // Tail recursive call
                        e = Arrays.copyOfRange(arr, 1, arr.length)
                    } else {
                        return null
                    }
                    break
                case CharSequence:
                case Number:
                case Boolean:
                    return e
                case Term.NUMBER: // 数値
                    return (e as Term.NUMBER).value
                case Term.YyValue: // (数値) 文字列, 正規表現, BOXING - for Compiler.
                    return (e as Term.YyValue).toString()
                case Node.Root:
                    Node.Root root = e as Node.Root
                    scriptLineNumber = root.linenumber // 行番号
                    return evalRoot(root)
                default:
                    throw new IllegalStateException(e.getClass().getName() + " " + e)
            }
        }
    }

    //* eval Root
    private Object evalRoot(Node.Root e) {
        switch (e) {
            case Node.YyVariable: /* 変数参照 */
                Node.YyVariable x = e as Node.YyVariable
                x.name = getAlias(x.name) // POSTIT エイリアスの設定☆
                String[] vName = varName(x)
                String name = vName[0]
                if (functionMap.containsKey(name)) {
                    callMap.put(name, new Integer[]{1}) // ユーザ関数を呼び出した.
                    return name + '()'
                }
                checkAndGlobalAlloc(name)
                return mkVerName(x)

            case Node.YyeValue:
                return yyeValue(e as Node.YyeValue)

            case Node.YyStatement:
                return yyStatement(e as Node.YyStatement)

            case Node.Call:
                Node.Call x = e as Node.Call
                return callFunction(x.name as String, x.args) /* 関数呼出し */

            case Node.Invoke:
                return invoke(e as Node.Invoke) // invoke

            case Node.Getline:
                Node.Getline x = e as Node.Getline /* getline */
                String rid = addQuote(x.rid)
                Object file = eval(x.filename)
                if (!(x.filename instanceof Node.YyVariable)) // NAME,Arr
                    file = addQuote(file)
                Node.YyVariable vv = x.args[0] as Node.YyVariable
                String name = getAlias(vv.name) // POSTIT エイリアスの設定☆
                String index = mkIndex(vv.index)
                String var = isNil(index) ? name : "${name}[${index}]"
                String stmt
                if ('$' == vv.name) { // $[index]
                    stmt = "(null == getline(${rid}, ${file}, ${index}) ? 0 : 1)"
                } else { // var
                    stmt = "(null == (${var} = getline(${rid}, ${file})) ? 0 : 1)"
                }
                return stmt

            case Node.YyNop:
                return (e as Node.YyNop).value // 注釈
            default:
                throw new IllegalStateException(e.getClass().getName() + " " + e)
        }
    }

    //* 論理比較
    private String yyeCompare(Node.YyeValue e) {
        String op = e.name
        String a = eval((e as Node.Comp).left)
        String b = eval((e as Node.Comp).right)
        if ((op.matches(/[|]{1,2}/)) || // ||, &&
                (op.matches(/&{1,2}/)))
            return "${paren(a)} $op ${paren(b)}"
        if (op.matches(/^\d+$/))
            return "_p2p(${op}, ${a}, ${b})" // pattern to pattern
        if (Operator.YY_OPIN == op)
            return "_in(${b}, ${mkIndex2(a)})" // 'in'
        if (op.contains('~'))
            return "_if('${op}', ${a}, ${b})" // 正規表現適合 ~ !~
        if (b.contains('=')) b = paren(b) // '()'を解析
        return "(${a} ${op} ${b})" // `< > != <= >= =='
    }

    private String yyeValue(Node.YyeValue e) {
        String name = e.name
        String rs
        switch (e) {
            case Node.B00l: /* 真値判定 */
                rs = boolIf(name, (e as Node.B00l).expr)
                break
            case Node.Comp:
                Node.Comp bc = e as Node.Comp
                if (Operator.YY_IS == name || // 'is' instanceOf
                        Operator.YY_AS == name) { // 'as' 強制型変換
                    String left = eval(bc.left)
                    String right = applyImport((e as Node.Comp).right.toString())
                    if (Operator.YY_IS == name)
                        rs = "_is(${left}, '${right}')" // is
                    else
                        rs = "${left} as ${right}" // as

                } else if (Operator.YY_OPIN == name) { // `in` 配列要素の存在検査
                    switch (bc.left) { // INDEX in Array
                        case Node.Call:
                            Node.Call call = bc.left as Node.Call // 左辺は、配列
                            rs = '_in(' + bc.right + ', ' + mkIndex(call.args) + ')'
                            break
                        default:
                            rs = yyeCompare(e)
                    }
                } else {
                    rs = yyeCompare(e)
                }
                break
            case Node.IncDec:
                Node.IncDec x = e as Node.IncDec
                Node.YyVariable expr = x.expr as Node.YyVariable
                String op = ('`~' == name) ? '~' : name // POSTIT 読み替え☆
                if (Node.NUMBER_OP == op) {
                    rs = "_int(${eval(expr)})" // ダウンキャスト
                } else {
                    rs = incdecNegate(op, expr) // インクリメント, デクリメント、ビット否定
                }
                break
            case Node.Calc: // 四則演算
                Node.Calc x = e as Node.Calc
                String a = eval(x.left)
                String b = eval(x.right)
                if ('0' == a && '-' == name)
                    rs = '-' + b
                else
                    rs = '(' + a + ' ' + name + ' ' + b + ')'
                break
            default:
                throw new IllegalStateException(e.getClass().getName() + " " + e)
        }
        assert null != rs
        return rs
    }

    private Object yyStatement(Node.YyStatement e) {
        Keyword id = e.id
        switch (e) {
            case Node.Ass: // 代入
                Node.Ass x = e as Node.Ass
                x.name = getAlias(x.name) // POSTIT エイリアスの設定☆
                return assignStmt(x)

            case Node.If:
                Node.If x = e as Node.If
                if (Keyword.SyyQIF == id) { // c ? a: b
                    return '' + eval(x.cc) + ' ? ' +
                            '(' + inline(x.left) + ') : (' + inline(x.right) + ')'
                } else {
                    String ex = paren(inline(x.cc))
                    gen("if ${ex} {") // if()
                    statement(x.left)
                    if (0 < x.right.length) {
                        gen(('} else {'))
                        statement(x.right)
                    }
                    return '}'
                }

            case Node.Stmt: /* 割り込み, return */
                return simpleStmt(e as Node.Stmt)

            case Node.For: /* for(;;) */
                Node.For x = e as Node.For
                String e1 = eval(x.e1); if (isNil(e1)) e1 = ''
                String e2 = eval(x.e2); if (isNil(e2)) e2 = ''
                String e3 = eval(x.e3); if (isNil(e3)) e3 = ''
                Symbols.startBlock()
                gen("for (${e1}; ${e2}; ${e3}) {")
                yyIndent += 1
                statement(x.stmt)
                yyIndent -= 1
                Symbols.endBlock()
                return '}'

            case Node.ForIn: /* for(in) */
                Node.ForIn x = e as Node.ForIn
                Symbols.startBlock()
                gen("for (${eval(x.e1)} in ${eval(x.e2)}) {")
                yyIndent += 1
                statement(x.stmt)
                yyIndent -= 1
                Symbols.endBlock()
                return '}'

            case Node.While: /* while */
                Node.While x = e as Node.While
                Symbols.startBlock()
                String rs
                if ((Keyword.SymWHILE == id) || (Keyword.SyyBWHILE == id)) {
                    String cc = paren(eval(x.cc) as String)
                    gen("while ${cc} {")
                    statement(x.stmt)
                    rs = '}'
                } else if ((Keyword.SymDO == id) || (Keyword.SyyBDO == id)) {
                    gen("do {")
                    statement(x.stmt)
                    String cc = paren(eval(x.cc) as String)
                    rs = "} while ${cc}"
                } else {
                    // NOTE Keyword.SyyBDO,SyyBWHILE エミュレータ - 雛形として残す.
                    //　- do{}while(), while(){}
                    // - /*do*/ while (hasNext) { if(hasnext) nasnext=next}
                    System.err.println(".plan to delete! " + x)
                    boolean isDO = Keyword.SyyBDO == id
                    varSequence += 1
                    String hasNext = 'hasNext' + varSequence + '_'
                    String cc = paren(eval(x.cc) as String)
                    gen("boolean ${hasNext} = true")
                    if (isDO) gen("/*do*/ while (${hasNext}) {")
                    else gen("while (${hasNext} && ${cc}) {")
                    beginBreak() // break 開始
                    statement(x.stmt)
                    endBreak(hasNext) // break 終了
                    if (isDO) gen("  if (${hasNext}) ${hasNext} = ${cc}")
                    rs = '}'
                }
                Symbols.endBlock()
                return rs

            case Node.Del: /* delete */
                Node.Del x = e as Node.Del
                if (isNil(x.e.index)) {
                    return x.e.name + '.clear()'
                } else {
                    String index = mkIndex2(x.e.index)
                    return x.e.name + '.remove(' + index + ')'
                }

            case Node.Print: /* print */
                Node.Print x = e as Node.Print
                String rid = addQuote(x.rid)
                String fname = (isNil(x.filename)) ? "''" : eval(x.filename)
                String sargs = mkStringNil(evalArgs(x.args), ', ', ', ', '')
                //* Groovyとバッティングするため、変名する
                return builtInFuncName(x.name) + "(${rid}, ${fname}${sargs})"

            case Node.Try: // try　{} { catch (exception |... e) {... finally {}.
                Node.Try x = e as Node.Try
                gen('try {')
                statement(x.e1)
                if (exists(x.e2)) {
                    for (Node.Catch xx in x.e2) {
                        String claz = mkString(xx.claz, "", " | ", "")
                        gen("} catch (${claz} ${xx.name}) {")
                        statement(xx.stmt)
                    }
                }
                if (exists(x.e3)) {
                    gen('} finally {')
                    statement(x.e3)
                }
                return '}'

            case Node.Throw:
                Node.Throw x = e as Node.Throw
                Symbols.startBlock()
                String xx = 'throw ' + eval(x.expr)
                Symbols.endBlock()
                return xx
        }
        throw new IllegalStateException(e.getClass().getName() + " " + e)
    }

//* アクション
    private void action(String name, String[] anno, Object[] e) {
        for (x in anno) gen(x) // アクション Annotation 生成
        gen("void ${name}() {")
        if (0 != e.length) {
            Object scope = Symbols.startLocal(name)
            statement(e)
            Symbols.endLocal(scope)
        }
        gen('}')
    }

//* 文内の文
    private String inline(Object[] a) {
        List backup = codeBuf.toList()
        codeBuf.clear()
        statement(a)
        List buf = []
        int srcBuflen = codeBuf.size()
        for (int i = 0; srcBuflen > i; i++)
            buf.add(("" + codeBuf.get(i)).replaceFirst("^[^\t]+[ \t]*", ""))
        codeBuf.clear()
        codeBuf.addAll(backup) // バッファを復元
        mkString(buf.toArray(), "", "; ", "")
    }
}