/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import java.util.Arrays;
import java.util.Objects;

/**
 * Lexical Analyzer.
 *
 * @author kunio himei.
 */
abstract class Lex extends Token {

    /**
     * 指定桁数の空白文字列を返す.
     */
    private static String space(int n) {
        char[] ca = new char[n];
        Arrays.fill(ca, ' ');
        return new String(ca);
    }

    /**
     * エスケープシーケンスを表示文字に変換して返す.
     */
    private static String unescapeM(Object x) {
        return Objects.toString(x, "").
                replace("\r", "\\r").replace("\n", "\\n")
                .replace("\t", " ");
    }

    /**
     * dump メッセージを出力する.
     */
    void dumpLog() {
        System.err.println("------------------------------------------------------------");
        Symbols.dump();
        System.err.println(this.yyScriptName + ' ' + editLinepos() + '\n'
                + unescapeM(super.yyOline) + '\n'
                + space(super.yyLexColumn + 1) + '^');
    }

    /**
     *
     */
    private String editLinepos() {
        return "[" + Advance.yyLineNumber()
                + ((0 == super.yyLexColumn) ? "" : ("." + super.yyLexColumn))
                + ']';
    }

    /**
     *
     */
    @Override
    void yyINFOMATION(Object x) {
        System.err.println(editLinepos() + '\t' + "INFO: " + unescapeM(x));
    }

    /**
     *
     */
    void yyOPTIMIZE(Object x) {
        System.err.println(editLinepos() + '\t' + "OPTIMIZE: " + unescapeM(x));
    }

    /**
     *
     */
    void yyWARNING(Object x) {
        System.err.println(editLinepos() + '\t' + "WARNING: " + unescapeM(x));
    }
}