/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.eval;

import org.jetbrains.annotations.Nullable;
import plus.concurrent.AtomicNumber;
import plus.exception.BreakException;
import plus.exception.ContinueException;
import plus.exception.ExitException;
import plus.exception.NextException;
import plus.lex.*;
import plus.runtime.BuiltInVar;
import plus.util.Debug;
import plus.util.NumHelper;
import plus.variable.Frame;

import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;

/**
 * AWK~plus - Evaluate.
 *
 * @author kunio himei.
 */
public final class Eval extends EvalAssign {

    public static final String AWK_PLUS_FOR_ANDROID = "AWK˜plus for Android";

    //* 整数 [0-9] を表わす正規表現
    private static final Pattern RX_INT_NUMBER = Pattern.compile("^\\d+$");
    //
    private static final Integer C_INTEGER1 = 1;
    //* 'as' 未サポートメッセージを出力済み.
    private static boolean appliedAsMessage = false;

    /**
     * 構文木を設定.
     */
    public Eval(final Parser.SyntaxTree tree) {
        sTree = tree;
    }

    /**
     * ;;]
     * スクリプトを実行する.
     */
    public int execute(String[] args) throws Throwable {
        BuiltInVar.reInitialize(); // for Android.
        Frame.cleanup(); // 御守り.
        _GLOBALS("false", Boolean.FALSE);
        _GLOBALS("true", Boolean.TRUE);

        isGlobalDef = true; // global変数定義開始
        // global 変数の初期化
        for (Map.Entry<String, Integer> x : sTree.globalValues.entrySet()) {
            String k = x.getKey();
            if (Flags.isArray(x.getValue())) {
                _VARIABLES._allocArray(k); // delete T など配列を宣言していない場合.
            } else {
                _GLOBALS(k, ZERO); // REMIND 0で初期化.
            }
        }
        if (exists(sTree.global)) {
            eval(sTree.global); // 定義順に値を設定する.
        }
        isGlobalDef = false;
        _newArgs(args, null); // global初期化後に呼び出す(-v var=val対策)

        int exitCode = 0;
        try {
            try {
                if (0 != sTree.begin.length) { // BEGIN
                    Frame._startBlockContext();
                    eval(sTree.begin);
                }
                if ((0 != sTree.main.length) || (0 != sTree.end.length)) { // MAIN
                    Object file = (0 == BuiltInVar.ARGIND.intValue()) ?
                            nextfile() : BuiltInVar.FILENAME.get();
                    Frame._startBlockContext(); // MAIN
                    while (exists(file)) { // パラメータの終端まで
                        BuiltInVar.FNR.put(0); // FNRをリセット
                        while (null != getline("", file, 0)) {
                            // NOTE 'The AWK 2-5 getline関数'参照.
                            BuiltInVar.NR.calculate("+", C_INTEGER1); // 入力したレコード数の合計
                            BuiltInVar.FNR.calculate("+", C_INTEGER1); // 現在のファイルから入力したレコード数
                            try {
                                for (Object x : sTree.main) {
                                    eval(x);
                                }
                            } catch (NextException e) {
                                file = BuiltInVar.FILENAME.get();
                            }
                        }
                        file = nextfile();
                    }
                }
            } catch (ExitException e) {
                if (Frame.GLOBALS.containsKey(AWK_PLUS_FOR_ANDROID)) // for Android
                    Frame.GLOBALS.put(AWK_PLUS_FOR_ANDROID, e.value);
                else
                    exitCode = e.value();
            }
            if (0 != sTree.end.length) {
                Frame._startBlockContext(); // END
                eval(sTree.end);
            }
        } catch (ExitException e) {
            if (Frame.GLOBALS.containsKey(AWK_PLUS_FOR_ANDROID)) // for Android
                Frame.GLOBALS.put(AWK_PLUS_FOR_ANDROID, e.value);
            else
                exitCode = e.value();

        } catch (Throwable e) {
            Throwable cause = e;
            while (null != cause.getCause())
                cause = cause.getCause();
            cause.printStackTrace();
            System.err.println(logging.snap());
        } finally {
            close(); // Android クリーンアップ.
        }
        return exitCode;
    }

    /**
     * オブジェクトの評価結果を返す.
     */
    @Nullable
    @Override
    Object eval(Object ee) {
        try {
            Object e = ee;
            while (true) {
                if (e instanceof Object[] x) { // 配列
                    int xxlen = x.length;
                    if (1 == xxlen) {
                        e = x[0]; // Tail recursive call
                    } else if (1 < xxlen) {
                        eval(x[0]); // Tail recursive call
                        e = Arrays.copyOfRange(x, 1, x.length);
                    } else {
                        return x;
                    }
                } else if (e instanceof CharSequence || //☆ PATCH Var/Val
                        e instanceof Number || e instanceof Boolean) {
                    return e;

                } else if (e instanceof Term.YyValue) {
                    // 文字列, 正規表現, 数値, BOXING - for Interpreter.
                    return ((Term.YyValue) e).value;

                } else if (e instanceof Node.Root) {
                    logging.log((Node.Root) e);

                    if (e instanceof Node.YyVariable x) {
                        String name = x.name;
                        if ((x instanceof Node.NAME)
                                && sTree.func.containsKey(name)) {
                            return callFunction(name, EMPTY_ARRAY); // パラメータなし、ユーザ関数呼出し
                        }
                        return _getValue(name, mkIndex(x.index)); /* 変数参照 */

                    } else if (e instanceof Node.YyeValue) {
                        return yyeValue((Node.YyeValue) e);

                    } else if (e instanceof Node.YyStatement) {
                        return yyStatement((Node.YyStatement) e);

                    } else if (e instanceof Node.Call x) { /* 関数呼出し */
                        return callFunction(x.name, x.args);

                    } else if (e instanceof Node.Invoke) { /* invoke */
                        return invoke((Node.Invoke) e);

                    } else if (e instanceof Node.Getline x) { /* getline */
                        String file = "" + eval(x.filename);
                        Node.YyVariable vv = (Node.YyVariable) x.args[0];
                        String index = mkIndex(vv.index);
                        String line;
                        // $(index) $0は省略可、[$?は'?'(実行時に決定)]、その他は""
                        if ("$".equals(vv.name)) {
                            line = getline(x.rid, file, index); // $[index]
                        } else {
                            line = getline(x.rid, file); // var
                            _putValue(vv.name, index, "=", line);
                        }
                        return null == line ? 0 : 1;

                    } else if (e instanceof Node.YyNop) {
                        return NORMAL_RETURN;

                    } else {
                        throw new IllegalStateException("unmatch: " + Debug.objectOf(e));
                    }
                } else {
                    throw new IllegalStateException("unmatch: " + Debug.objectOf(e));
                }
            }
        } catch (RuntimeException ex) {
            throw ex; // 割り込みをそのまま投げる
        } catch (Throwable ex) {
            throw new RuntimeException(ex); // アンラップした例外を投げる
        }
    }

    /**
     * オブジェクトの評価結果を返す.
     */
    private Object yyeValue(Node.YyeValue e) {
        String name = e.name;
        if (e instanceof Node.B00l x) { /* 真値判定(Boolean|0|1) ! */
            boolean bo = _if(eval(x.expr));
            return (Node.NOT_OP.equals(name)) != bo;

        } else if (e instanceof Node.Comp x) {
            if ((Operator.YY_IS.equals(name) // 'is' instanceOf
                    || Operator.YY_AS.equals(name))) { // 'as' 強制型変換
                Object left = eval(x.left);
                Object right = applyImport(x.right.toString()); // ☆
                if (Operator.YY_IS.equals(name)) { // 'is' instanceOf
                    return _is(left, right.toString());
                } else {
                    if (!appliedAsMessage) {
                        appliedAsMessage = true; // メッセージ出力済.
                        System.err.println( // NOTE REMIND 'as'は、未サポート.
                                " 'as'(forced cast) is not supported by the interpreter.");
                    }
                    return NORMAL_RETURN;
                }
            } else if (Operator.YY_OPIN.equals(name)  // `in` 配列要素の存在検査
                    && (x.left instanceof Node.Call a)) {
                return _in(eval(x.right), mkIndex(a.args));
            }
            boolean bo; /* 論理比較 */
            if (name.matches("[|]{1,2}")) {
                bo = (_if(eval(x.left)) || _if(eval(x.right))); // ||
            } else if (name.matches("&{1,2}")) {
                bo = (_if(eval(x.left)) && _if(eval(x.right))); // &&
            } else if (RX_INT_NUMBER.matcher(name).find()) {
                bo = _p2p(name, eval(x.left), eval(x.right));
            } else if (Operator.YY_OPIN.equals(name)) {
                bo = _in(eval(x.right), eval(x.left));
            } else {
                bo = _if(name, eval(x.left), eval(x.right)); // `< > == != <= >= ~ !~'
            }
            return bo;

        } else if (e instanceof Node.IncDec x) { /* ++ -- 数値正規化 */
            if (Node.NUMBER_OP.equals(name)) {
                return NumHelper.normalise(NumHelper.doubleValue(eval(x.expr)));
            }
            //* インクリメント, デクリメント、ビット否定(bigwiseNegate)
            Node.YyVariable a = (Node.YyVariable) x.expr;
            return _putValue(a.name, mkIndex(a.index), name, 1);

        } else if (e instanceof Node.Calc x) { /* 四則演算 */
            return AtomicNumber.calculate(name, eval(x.left), eval(x.right));

        }
        throw new IllegalStateException("unmatch: " + Debug.objectOf(e));
    }

    /**
     * オブジェクトの評価結果を返す.
     */
    private Object yyStatement(Node.YyStatement e) throws Throwable {
        Keyword id = e.id;

        if (e instanceof Node.Ass) { /* 代入 */
            return assignStmt((Node.Ass) e);

        } else if (e instanceof Node.If x) { /* c ? a: b */
            Object bo = _if(eval(x.cc)) ?
                    eval(x.left) : eval(x.right);
            return (Keyword.SymIF == id) ? NORMAL_RETURN : bo;

        } else if (e instanceof Node.Stmt) { /* 単純な文 */
            simpleStmt((Node.Stmt) e);
            return NORMAL_RETURN;

        } else if (e instanceof Node.ForIn x) {
            Iterator<?> jIte = iterator(eval(x.e2));
            if ((Keyword.SymVAL == x.e1.id) || (Keyword.SymVAR == x.e1.id)) {
                assignStmt(new Node.Ass(Keyword.SymVAR, "=", x.e1,
                        NIL_VAR, Flags.T11ANY, "")); //☆ PATCH Var/Val
            }
            boolean hasNext = true;
            while (hasNext && jIte.hasNext()) {
                Object k = jIte.next();
                try {
                    _putValue(x.e1.name, null, "=", eval(k));
                    eval(x.stmt);
                } catch (BreakException e1) {
                    hasNext = false;
                } catch (ContinueException e1) { // NOPMD: empty - this surely will never happen
                    // continue
                }
            }
            return NORMAL_RETURN;

        } else if (e instanceof Node.For x) { /* for(;;) */
            eval(x.e1);
            boolean is1st = true, hasNext = true;
            while (hasNext) {
                try {
                    if (is1st) {
                        is1st = false;
                    } else {
                        eval(x.e3);
                    }
                    if (_if(eval(x.e2))) {
                        eval(x.stmt);
                    } else {
                        hasNext = false;
                    }
                } catch (BreakException e1) {
                    hasNext = false;
                } catch (ContinueException e1) { // NOPMD: empty - this surely will never happen.
                    // continue
                }
            }
            return NORMAL_RETURN;

        } else if (e instanceof Node.While x) { /* while */
            boolean isDO = (Keyword.SymDO == id)
                    || (Keyword.SyyBDO == id);
            boolean hasNext = true;
            while (hasNext && (isDO || _if(eval(x.cc)))) { // do or while
                try {
                    eval(x.stmt);
                } catch (BreakException ex) {
                    hasNext = false;
                } catch (ContinueException ex) { // NOPMD: empty - this surely will never happen
                    // continue
                }
                if (hasNext && isDO) { // do-while
                    hasNext = _if(eval(x.cc));
                }
            }
            return NORMAL_RETURN;

        } else if (e instanceof Node.Del x) { /* delete */
            String idx = mkIndex(x.e.index); // 全削除の場合は、null
            if ("$".equals(x.e.name)) {
                BuiltInVar.$.putAt(idx, "");
            } else {
                _VARIABLES._remove(x.e.name, idx);
            }
            return NORMAL_RETURN;

        } else if (e instanceof Node.Print x) { /* print */
            String file = Objects.toString(eval(x.filename), "");
            Object[] vals = evalArgs(x.args);
            if ("print".equals(x.name)) {
                return print(x.rid, file, vals);
            }
            return _printf(x.rid, file, vals);

        } else if (e instanceof Node.Try) { /* try, catch */
            return tryCatch((Node.Try) e);

        } else if (e instanceof Node.Throw x) { /* throw */
            Object o = eval(x.expr);
            if (o instanceof Throwable) {
                throw (Throwable) o;
            }
            return NORMAL_RETURN;

        } else if (e instanceof Node.FnI) { /* 関数定義行 (Tag) */
            // System.err.println(" .yyStatement.FnI: " + e);
            return NORMAL_RETURN;
        }
        throw new IllegalStateException("unmatch: " + Debug.objectOf(e));
    }

}