/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.concurrent;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Atomic 開始、終了、増分値で指定された整数値を返す 範囲オブジェクトの実装.
 * <p>
 * The class to which this annotation is applied is thread-safe.
 *
 * @author Kunio Himei.
 */
public final class Range implements Iterable<Integer>, Iterator<Integer> {

    /**
     * atomic value.
     */
    private final AtomicInteger atom;

    /**
     * end value.
     */
    private final int endValue;

    /**
     * 要素の一時スタック.
     */
    private final Queue<Integer> queue = new ConcurrentLinkedQueue<>();

    /**
     * non-zero step value.
     */
    private final int stepValue;

    /**
     * The Range class represents integer values in range.
     */
    public Range(Number start, Number end, Number step) {
        this.atom = new AtomicInteger(start.intValue());
        this.endValue = end.intValue();
        this.stepValue = step.intValue();
    }

    /**
     * 次の要素を返す.
     */
    @Nullable
    private Integer getNext() {
        int x = this.atom.addAndGet(this.stepValue) - this.stepValue;
        // 符号が反転していなければ OK (step は負もある)
        return (0 <= (this.stepValue * (this.endValue - x))) ? x : null;
    }

    /**
     * 次の要素の有無を返す.
     */
    @Override
    public boolean hasNext() {
        Integer x = getNext(); // 次の要素を取得する
        return (null != x) && this.queue.add(x); // push
    }

    /**
     * Iterator を返す.
     */
    @NotNull
    @Override
    public Iterator<Integer> iterator() {
        return this;
    }

    /**
     * 現在の要素値を返す.
     * <p>
     * {@literal NoSuchElementException} - 繰り返し処理でそれ以上要素がない場合
     */
    @Override
    public Integer next() {
        Integer x = this.queue.poll(); // pop
        if (null == x) { // hasNext より先に呼ばれたときは、
            x = getNext(); // 次の要素を取得する
            if (null == x) {
                throw new NoSuchElementException();
            }
        }
        return x;
    }

    /**
     * サポートしない.
     */
    @Deprecated
    @Override
    public void remove() {
        throw new IllegalStateException();
    }
}