/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import org.jetbrains.annotations.Nullable;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * Parser generator (構文解析).
 *
 * @author kunio himei.
 */
@SuppressWarnings("unused")
public final class Parser extends Action {

    //* 構文木.
    private final SyntaxTree sTree = new SyntaxTree();
    //* pre import file name.
    private static final String PRE_IMPORT_FILE_NAME = "import.awk";
    //* Class Annotation.
    private final List<String> annotation = new LexArray<>();
    private boolean isDefinable = true;
    //* MAIN.
    private final List<String> annoteMain = new LexArray<>();
    private final List<Object> mainBlock = new LexArray<>();
    // スクリプトリーダ.
    private BufferedReader reader;
    //* COMMENT.
    private final List<String> commentList = new LexArray<>();
    //* GLOBAL.
    private final List<Object> globalBlock = new LexArray<>();
    //* BEGIN.
    private final List<Object> beginBlock = new LexArray<>();
    //* END.
    private final List<Object> endBlock = new LexArray<>();
    //* スクリプトファイル名 []
    private String[] scriptFiles;
    //* 現在処理中のスクリプトファイルの位置.
    private int scriptIndex;

    /**
     * 構文解析の結果を返す.
     */
    public SyntaxTree parse(String[] scripts) throws Throwable {
        scriptFiles = scripts;
        try {
            advance();
            while (null != super.tok) {
                assert !"\n".equals(super.tok);
                if (Keyword.SymPACKAGE == super.tok) { // package
                    eat(super.tok);
                    sTree.packageName = super.tok.toString().trim();
                    advance();
                    nl();

                } else if (Keyword.SymIMPORT == super.tok) { // Import
                    // 'Path.*'は、変数名として認められないため迂回手段を取る.
                    String path = super.yyText.trim()
                            .replaceFirst("^\\w+\\s+", "")
                            .replaceFirst("\\s+.*$", "");
                    if (yyScriptName.contains(PRE_IMPORT_FILE_NAME)) {
                        sTree.preImport.add(path); // path.class
                    } else {
                        sTree.userImport.add(path); // path.(class|*)
                    }
                    eat(super.tok); // import
                    eat(super.tok); // path
                    advance();
                    nl();

                } else if (super.tok instanceof Node.Annotation) {
                    // クラスAnnotationが複数定義されVarが無い場合は、NG!
                    // ここでは、1個のみ定義.
                    if (isDefinable && annotation.isEmpty())
                        annotation.addAll(yyAnnotation.moveList());
                    eat(super.tok);
                    advance();
                    nl();

                } else if ((Keyword.SymVAR == super.tok) // global定義
                        || (Keyword.SymVAL == super.tok)) {
                    if (isDefinable) { // 最初の変数が現れたら在庫を追加.
                        annotation.addAll(yyAnnotation.moveList());
                        isDefinable = false;
                    }
                    super.isGlobalDef = true;
                    globalBlock.add(expression());
                    super.isGlobalDef = false;
                    nl();

                } else if (Keyword.SymBEGIN == super.tok) { // BEGIN
                    if (hasComment())
                        commentList.addAll(getComment());
                    sTree.annotateBEGIN = yyAnnotation.moveArray();
                    eat(super.tok);
                    requireLeftBraces(); // "{"
                    beginBlock.addAll(Arrays.asList(pattern(false)));

                } else if (Keyword.SymEND == super.tok) { // END
                    if (hasComment())
                        commentList.addAll(getComment());
                    sTree.annotateEND = yyAnnotation.moveArray();
                    eat(super.tok);
                    requireLeftBraces(); // "{"
                    endBlock.addAll(Arrays.asList(pattern(false)));

                } else if (Keyword.SymFUNCTION == super.tok) { // Function
                    eat(super.tok);
                    functionDecl();

                } else { // MAIN
                    if (hasComment())
                        commentList.addAll(getComment());
                    annoteMain.addAll(yyAnnotation);
                    yyAnnotation.clear();
                    mainBlock.addAll(Arrays.asList(pattern(true)));
                }
            }
            updateFunctionDef();

            // 構文木
            sTree.comment = commentList.toArray(new String[0]);
            sTree.annotation = annotation.toArray(new String[0]);
            sTree.annotateMAIN = annoteMain.toArray(new String[0]);
            sTree.global = globalBlock.toArray();
            sTree.begin = beginBlock.toArray();
            sTree.main = mainBlock.toArray();
            sTree.end = endBlock.toArray();
            sTree.globalValues = Symbols.cleanGlobal();
            sTree.func = new HashMap<>(super.functionMap);
            sTree.call = new HashMap<>(super.callMAP);
            super.functionMap.clear();
            super.callMAP.clear();
            return sTree;

        } catch (Throwable e) {
            dumpLog();
            Throwable cause = e;
            while (null != cause.getCause()) {
                cause = cause.getCause();
            }
            throw cause;
        }
    }

    /**
     * スクリプトを読み込んで返す.
     */
    @Nullable
    @Override
    String readline() {
        String line = null;
        while (null == line) {
            try {
                if (null == reader) {
                    if (scriptFiles.length <= scriptIndex) {
                        return null;
                    }
                    sTree.scriptName = super.yyScriptName = scriptFiles[scriptIndex];
                    scriptIndex++;
                    commentList.clear();
                    if (new File(sTree.scriptName).isFile()) {
                        reader = new BufferedReader(new InputStreamReader(
                                new FileInputStream(super.yyScriptName),
                                StandardCharsets.UTF_8)); // NOTE codeのコード.
                    } else {
                        reader = new BufferedReader( // eval
                                new StringReader(sTree.scriptName));
                    }
                }
                line = reader.readLine();
                if (null == line) {
                    reader.close();
                    reader = null;
                    Advance.yyLineNumber(0); // スクリプト行番号をクリア
                    super.yyLexNumber = 0;
                }
            } catch (IOException e) {
                throw new RuntimeException(e); // ラップした例外を投げる
            }
        }
        return line;
    }

    //* 構文木
    public static class SyntaxTree {
        // for Interpreter - java.lang,util.
        public final Set<String> preImport = new TreeSet<>(); // import.awk.
        public final Set<String> userImport = new TreeSet<>(); // Pass.* を含む.
        public String packageName = ""; // パケージ名.
        public String[] comment; // 注釈
        public String[] annotation; // 注釈 (Annotation).
        public String[] annotateBEGIN;
        public String[] annotateMAIN;
        public String[] annotateEND;
        public Object[] global;
        public Object[] begin;
        public Object[] main;
        public Object[] end;
        public Map<String, Integer> globalValues; // グローバル定義.
        public Map<String, Node.Func> func; // 関数定義.
        public Map<String, Integer[]> call; // 定義した関数を呼び出した.
        public String scriptName; // 最後に処理したスクリプト名.
    }
}